<?php

use App\Http\Controllers\Admin\DashboardController as AdminDashboardController;
use App\Http\Controllers\Admin\GroupPairController;
use App\Http\Controllers\Admin\MatchesController as AdminMatchesController;
use App\Http\Controllers\Admin\PlayerController as AdminPlayerController;
use App\Http\Controllers\Admin\RegistrationController as AdminRegistrationController;
use App\Http\Controllers\Admin\TournamentController as AdminTournamentController;
use App\Http\Controllers\Admin\TournamentGroupController as AdminTournamentGroupController;
use App\Http\Controllers\Admin\TournamentPairController as AdminTournamentPairController;
use App\Http\Controllers\Admin\TournamentRoundController as AdminTournamentRoundController;
use App\Http\Controllers\Admin\UserController as AdminUserController;
use App\Http\Controllers\Admin\VenueController as AdminVenueController;
use App\Http\Controllers\Api\JobStatusController;
use App\Http\Controllers\Auth\LoginController;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\PlayerController;
use App\Http\Controllers\RegisterMemberController;
use App\Http\Controllers\RegistrationController;
use App\Http\Controllers\ReviewController;
use App\Http\Controllers\TournamentController;
use App\Http\Controllers\TournamentListController;
use App\Http\Controllers\TournamentSettingController;
use App\Http\Controllers\VenueController;
use App\Http\Middleware\TournamentPasswordMiddleware;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Public Routes
|--------------------------------------------------------------------------
*/
Route::controller(HomeController::class)->group(function () {
    Route::get('/', 'index')->name('home');
});

Route::controller(PlayerController::class)->prefix('players')->name('players.')->group(function () {
    Route::post('/register', 'register')->name('register'); // Public API
    Route::get('/api/check', 'checkByPhone')->name('api.check'); // Public API
    Route::get('/{id}', 'show')->name('show');
    Route::post('/{id}/update-avatar', 'updateAvatar')->name('updateAvatar');
});

// Trang chi tiết giải đấu
Route::controller(TournamentController::class)->group(function () {
    Route::get('/giai-dau-pickleball-viet-nam/{slug}', 'show')->name('tournaments.show');
    // Đăng ký player cho giải đấu
    Route::post('/giai-dau-pickleball-viet-nam/{tournament}/register', 'register')->name('tournament.register');
});

Route::get('/danh-sach-giai-dau-pickleball-viet-nam', [TournamentListController::class, 'index'])->name('tournaments.list');

Route::controller(RegistrationController::class)->group(function () {
    Route::post('/tournaments/{tournament}/register', 'store')->name('registrations.store');
    Route::post('/registrations/{id}/update-status', 'updateStatus')->name('registration.updateStatus')->middleware(TournamentPasswordMiddleware::class); // Protected by password
});

Route::post('/san-pickleball/{venue}/reviews', [ReviewController::class, 'store'])->name('reviews.store');
// Public venues search - must be before resource route
Route::get('/san-pickleball/search', [VenueController::class, 'search'])->name('venues.search');
Route::resource('san-pickleball', VenueController::class)
    ->only(['index', 'show'])
    ->names('venues')
    ->parameters(['san-pickleball' => 'venue']);

Route::controller(RegisterMemberController::class)->name('register-member.')->group(function () {
    Route::get('/register-member', 'show')->name('show');
    Route::post('/register-member', 'register')->name('register');
    Route::get('/test-telegram', 'testTelegram')->name('test-telegram')->middleware('auth');
});

/*
|--------------------------------------------------------------------------
| Authentication Routes
|--------------------------------------------------------------------------
*/
Route::controller(LoginController::class)->group(function () {
    Route::get('/login', 'showLoginForm')->name('login');
    Route::post('/login', 'login');
    Route::post('/logout', 'logout')->name('logout');
});

/*
|--------------------------------------------------------------------------
| Tournament Setting Routes (Password Protected)
|--------------------------------------------------------------------------
*/
Route::controller(TournamentSettingController::class)->prefix('tournament-setting')->name('tournament.setting.')->group(function () {
    Route::get('/{tournament}/setting', 'index')->name('index');
    Route::post('/{tournament}/setting/check', 'checkPassword')->name('check');
    Route::post('/logout', 'logout')->name('logout');

    Route::middleware(TournamentPasswordMiddleware::class)->group(function () {
        Route::get('/{tournament}/groups', 'getGroups')->name('groups.get');
        Route::post('/{tournament}/groups', 'addGroup')->name('groups.add');
        Route::post('/groups/{group}/delete', 'deleteGroup')->name('groups.delete'); // Should be DELETE
        Route::post('/groups/{group}/save', 'saveGroup')->name('groups.save'); // Should be PUT/PATCH

        Route::post('/{tournament}/groups/{group}/pairs', 'addPair')->name('groups.pairs.add');
        // Show create-pair form for organizers (GET) so they see the UI instead of raw JSON
        Route::get('/{tournament}/groups/{group}/pairs/create', 'showAddPairForm')->name('groups.pairs.create');
        // Organizer: edit/update a pair (reuse admin views but scoped)
        Route::get('/{tournament}/groups/{group}/pairs/{pair}/edit', 'editPair')->name('groups.pairs.edit');
        Route::post('/{tournament}/groups/{group}/pairs/{pair}/update', 'updatePair')->name('groups.pairs.update');
        Route::post('/pairs/{pair}/delete', 'deletePair')->name('pairs.delete'); // Should be DELETE

        Route::post('/{tournament}/auto-create-rounds', 'autoCreateRounds')->name('rounds.auto-create');
        Route::post('/{tournament}/rounds', 'addRound')->name('rounds.add');
        Route::post('/rounds/{round}/matches', 'addMatch')->name('rounds.matches.add');

        Route::get('/{tournament}/approved-players', 'getApprovedPlayers')->name('players.approved');
        Route::get('/{tournament}/pairs', 'getTournamentPairs')->name('pairs.get');
        // Organizer-level generation actions (mirror admin actions)
        Route::post('/{tournament}/auto-generate-groups', 'autoGenerateGroups')->name('auto-generate-groups');
        Route::post('/{tournament}/auto-generate-group-rounds', 'autoGenerateGroupRounds')->name('auto-generate-group-rounds');
        Route::post('/{tournament}/generate-playoffs', 'generatePlayoffs')->name('generate-playoffs');
        Route::post('/{tournament}/generate-round-32', 'generateRoundOf32')->name('generate-round-32');
        Route::post('/{tournament}/generate-round-16', 'generateRoundOf16')->name('generate-round-16');
        Route::post('/{tournament}/generate-quarterfinals', 'generateQuarterFinals')->name('generate-quarterfinals');
        Route::post('/{tournament}/generate-semifinals', 'generateSemiFinals')->name('generate-semifinals');
        Route::post('/{tournament}/generate-finals', 'generateFinals')->name('generate-finals');

        Route::get('/{tournament}/export-schedule', 'exportSchedule')->name('export-schedule');
        // Organizer: export pairs list per group
        Route::get('/{tournament}/export-pairs', 'exportPairs')->name('export-pairs');
        // Organizer activity log (view only)
        Route::get('/{tournament}/activity-log', 'activityLog')->name('activity-log');

        // Organizer: update match score
        Route::post('/matches/{match}/update-score', 'updateScore')->name('matches.updateScore');

        // Organizer: view and manage registrations
        Route::get('/{tournament}/registrations', 'registrations')->name('registrations.index');
        Route::get('/{tournament}/registrations/export', 'exportRegistrations')->name('registrations.export');
        // Organizer: update registration status
        Route::post('/registrations/{registration}/update-status', 'updateRegistrationStatus')->name('registrations.updateStatus');

        // Organizer-level clear actions
        Route::delete('/{tournament}/clear-groups', 'clearGroups')->name('clear-groups');
        Route::delete('/{tournament}/clear-group-rounds', 'clearGroupRounds')->name('clear-group-rounds');
        Route::delete('/{tournament}/clear-playoffs', 'clearPlayoffs')->name('clear-playoffs');
        Route::post('/pairs/{pair}/swap', [TournamentSettingController::class, 'swapPlayers'])->name('pairs.swap');

        // AJAX endpoints for organizer dashboard
        Route::get('/{tournament}/standings', 'getStandings')->name('standings');
        Route::get('/{tournament}/schedule', 'getSchedule')->name('schedule');
        Route::get('/{tournament}/playoffs', 'getPlayoffs')->name('playoffs');
        // Organizer: export standings
        Route::get('/{tournament}/export-standings', 'exportStandings')->name('export-standings');

    });
});

/*
|--------------------------------------------------------------------------
| Referee Setting Routes (Password Protected)
|--------------------------------------------------------------------------
*/
Route::post('/referee-setting/logout', [TournamentSettingController::class, 'logout'])->name('referee.setting.logout');

/*
|--------------------------------------------------------------------------
| Admin Routes
|--------------------------------------------------------------------------
*/
Route::prefix('admin')->name('admin.')->middleware(['auth', 'role:admin|mod'])->group(function () {
    Route::get('/dashboard', [AdminDashboardController::class, 'index'])->name('dashboard');

    // Tournament Management - Consolidated Routes
    Route::controller(AdminTournamentController::class)->prefix('tournaments')->name('tournaments.')->group(function () {
        // Standard CRUD
        Route::get('/', 'index')->name('index');
        Route::get('/search', 'search')->name('search'); // AJAX search endpoint
        Route::get('/create', 'create')->name('create');
        Route::post('/', 'store')->name('store');
        Route::get('/{tournament}', 'show')->name('show');
        Route::get('/{tournament}/edit', 'edit')->name('edit');
        Route::put('/{tournament}', 'update')->name('update');
        Route::delete('/{tournament}', 'destroy')->name('destroy');

        // Tournament-specific actions
        Route::prefix('{tournament}')->group(function () {
            // Registration Management
            Route::get('/registrations', 'registrations')->name('registrations');
            Route::get('/registrations/export', 'exportRegistrations')->name('registrations.export');
            Route::put('/registrations/{registration}/update-status', 'updateRegistrationStatus')->name('registrations.update-status');

            // Tournament Generation
            Route::post('/auto-generate-groups', 'autoGenerateGroups')->name('auto-generate-groups');
            Route::post('/auto-generate-group-rounds', 'autoGenerateGroupRounds')->name('auto-generate-group-rounds');
            Route::post('/generate-playoffs', 'generatePlayoffs')->name('generate-playoffs');
            Route::post('/generate-round-32', 'generateRoundOf32')->name('generate-round-32');
            Route::post('/generate-round-16', 'generateRoundOf16')->name('generate-round-16');
            Route::post('/generate-quarterfinals', 'generateQuarterFinals')->name('generate-quarterfinals');
            Route::post('/generate-semifinals', 'generateSemiFinals')->name('generate-semifinals');
            Route::post('/generate-finals', 'generateFinals')->name('generate-finals');

            // Clear Operations
            Route::delete('/clear-groups', 'clearGroups')->name('clear-groups');
            Route::delete('/clear-group-rounds', 'clearGroupRounds')->name('clear-group-rounds');
            Route::delete('/clear-playoffs', 'clearPlayoffs')->name('clear-playoffs');

            // AJAX endpoints for lazy loading
            Route::get('/standings', 'getStandings')->name('standings');
            Route::get('/schedule', 'getSchedule')->name('schedule');
            Route::get('/playoffs', 'getPlayoffs')->name('playoffs');

            // Exports & Analytics
            Route::get('/export-standings', 'exportStandings')->name('export-standings');
            Route::get('/export-schedule', 'exportSchedule')->name('export-schedule');
            Route::get('/export-pairs', 'exportPairs')->name('export-pairs');
            Route::get('/health-check', 'healthCheck')->name('health-check');
            Route::get('/activity-log', 'activityLog')->name('activity-log');
        });
    });

    // Tournament Sub-resources
    Route::prefix('tournaments/{tournament}')->group(function () {
        // Manual Group Management
        Route::post('/groups', [AdminTournamentGroupController::class, 'store'])->name('groups.store');
        Route::delete('/groups/{group}', [AdminTournamentGroupController::class, 'destroy'])->name('groups.destroy');

        // Tournament Pairs
        Route::get('/pairs', [AdminTournamentPairController::class, 'index'])->name('tournaments.pairs.index');

        // Tournament Rounds
        Route::resource('rounds', AdminTournamentRoundController::class)->except(['show']);
    });

    // Group Pairs Management
    Route::resource('groups.pairs', GroupPairController::class)->except(['show', 'edit', 'update']);
    Route::get('/groups/{group}/pairs/{pair}/edit', [GroupPairController::class, 'edit'])->name('groups.pairs.edit');
    Route::put('/groups/{group}/pairs/{pair}', [GroupPairController::class, 'update'])->name('groups.pairs.update');
    Route::post('/pairs/{pair}/swap', [GroupPairController::class, 'swapPlayers'])->name('pairs.swap');

    // Matches Management
    Route::post('/matches/{match}/update-score', [AdminMatchesController::class, 'updateScore'])->name('matches.updateScore');
    Route::resource('matches', AdminMatchesController::class);

    // Players Management
    Route::controller(AdminPlayerController::class)->prefix('players')->name('players.')->group(function () {
        Route::get('/search', 'search')->name('search'); // AJAX search endpoint
        Route::post('/{player}/approve-avatar', 'approveAvatar')->name('approveAvatar');
        Route::post('/{player}/reject-avatar', 'rejectAvatar')->name('rejectAvatar');
        Route::post('/process-pending-avatars', 'processPendingAvatars')->name('processPendingAvatars');
    });
    Route::resource('players', AdminPlayerController::class);

    // Other Admin Resources
    Route::middleware('admin')->group(function () {
        Route::controller(AdminUserController::class)->prefix('users')->name('users.')->group(function () {
            Route::get('/search', 'search')->name('search'); // AJAX search endpoint
        });
        Route::resource('users', AdminUserController::class);
    });
    // Venues search - must be before resource route
    Route::get('/venues/search', [AdminVenueController::class, 'search'])->name('venues.search');
    Route::resource('venues', AdminVenueController::class);

    // Registrations Management
    Route::post('/registrations/{registration}/update-status', [AdminRegistrationController::class, 'updateStatus'])->name('registrations.updateStatus');
    Route::resource('registrations', AdminRegistrationController::class);
});

// API Routes for Job Status
Route::prefix('api')->name('api.')->group(function () {
    Route::get('/tournaments/{tournament}/job-status', [JobStatusController::class, 'getTournamentJobStatus'])->name('job-status');
    Route::delete('/tournaments/{tournament}/job-status', [JobStatusController::class, 'clearTournamentJobStatus'])->name('job-status.clear');
});

// System Health Check Routes
Route::get('/health', [\App\Http\Controllers\SystemHealthController::class, 'check'])->name('health.check');
Route::get('/health/memory', [\App\Http\Controllers\SystemHealthController::class, 'memoryTest'])->name('health.memory');

// Tournament API routes for mobile
Route::prefix('api/tournaments')->group(function () {
    Route::post('/{tournament}/auth', [\App\Http\Controllers\Api\TournamentApiController::class, 'authenticate']);
    Route::post('/{tournament}/logout', [\App\Http\Controllers\Api\TournamentApiController::class, 'logout']);
    Route::get('/{tournament}/session', [\App\Http\Controllers\Api\TournamentApiController::class, 'getSessionInfo']);
    Route::get('/{tournament}/data/{dataType}', [\App\Http\Controllers\Api\TournamentApiController::class, 'getTournamentData'])->where('dataType', 'standings|groups|matches|rounds');
    Route::get('/{tournament}/matches', [\App\Http\Controllers\Api\TournamentApiController::class, 'getMatches']);
    Route::post('/matches/{match}/update-score', [\App\Http\Controllers\Api\TournamentApiController::class, 'updateMatchScore']);
    Route::post('/pairs/{pair}/swap-players', [\App\Http\Controllers\Api\TournamentApiController::class, 'swapPlayers']);
});