@extends('layouts.app')

@section('title', 'Danh sách sân Pickleball')

@push('layout_css')
<style>
    .venue-card {
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        border: 1px solid var(--border-color);
        background: var(--card-bg);
    }

    .venue-card:hover {
        transform: translateY(-8px);
        box-shadow: 0 12px 24px var(--shadow-dark) !important;
    }

    .venue-card .card-img-top {
        height: 220px;
        object-fit: cover;
        border-bottom: 1px solid var(--border-color);
    }

    .venue-card .card-body {
        display: flex;
        flex-direction: column;
    }

    .venue-card .card-title {
        font-weight: 700;
        color: var(--accent-primary);
    }

    .venue-card .card-title a {
        color: var(--accent-primary);
        text-decoration: none;
    }

    .venue-card .card-title a:hover {
        color: var(--accent-secondary);
    }

    .venue-card .card-text {
        flex-grow: 1;
        color: var(--text-secondary);
    }

    /* Card and form styles for dark theme */
    .card {
        background: var(--card-bg) !important;
        border: 1px solid var(--border-color) !important;
        color: var(--text-primary) !important;
    }

    .form-control,
    .form-select {
        background: var(--card-bg) !important;
        border: 1px solid var(--border-color) !important;
        color: var(--text-primary) !important;
    }

    .form-control:focus,
    .form-select:focus {
        background: var(--card-bg) !important;
        border-color: var(--accent-primary) !important;
        color: var(--text-primary) !important;
        box-shadow: 0 0 0 0.2rem var(--shadow-primary) !important;
    }

    .form-label {
        color: var(--text-primary) !important;
    }

    .btn-primary {
        background: linear-gradient(135deg, var(--accent-primary), var(--accent-secondary)) !important;
        border: none !important;
        color: var(--text-primary) !important;
    }

    .btn-primary:hover {
        background: linear-gradient(135deg, var(--accent-hover), var(--accent-primary)) !important;
        transform: translateY(-1px);
    }

    .badge {
        background: var(--accent-primary) !important;
        color: var(--text-primary) !important;
    }
</style>
@endpush

@section('content')
<div class="container py-5">
    <div class="text-center mb-5">
        <h1 class="display-5 fw-bold" style="color: var(--accent-primary);">Khám phá các sân Pickleball</h1>
        <p class="lead" style="color: var(--text-secondary);">Tìm kiếm và lựa chọn sân chơi phù hợp nhất cho bạn.</p>
    </div>

    <!-- Filter and Search Form -->
    <div class="card shadow-sm mb-5">
        <div class="card-body p-4">
            <form id="searchForm" class="row g-3 align-items-end">
                <div class="col-md-6">
                    <label for="search" class="form-label">Tìm kiếm theo tên, địa chỉ, khu vực, liên hệ</label>
                    <input type="text" name="search" id="search" value="{{ request('search') }}" class="form-control"
                        placeholder="VD: Pickleball Zone, Quận 1...">
                </div>
                <div class="col-md-4">
                    <label for="location" class="form-label">Lọc theo khu vực</label>
                    <select name="location" id="location" class="form-select">
                        <option value="">Tất cả khu vực</option>
                        @foreach($locations as $location)
                        <option value="{{ $location }}" @selected(request('location')==$location)>
                            {{ $location }}
                        </option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2 d-grid">
                    <button type="button" id="clearSearch" class="btn btn-outline-secondary">
                        <i class="bi bi-x-circle me-2"></i>Xóa bộ lọc
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Venues Grid Container -->
    <div class="position-relative">
        <!-- Loading overlay -->
        <div id="loadingOverlay" class="d-none position-absolute w-100 h-100"
            style="background: rgba(255, 255, 255, 0.8); z-index: 1000; border-radius: 0.375rem; min-height: 300px;">
            <div class="d-flex justify-content-center align-items-center h-100">
                <div class="text-center">
                    <div class="spinner-border text-primary mb-3" role="status">
                        <span class="visually-hidden">Đang tải...</span>
                    </div>
                    <p style="color: var(--text-primary);">Đang tìm kiếm sân...</p>
                </div>
            </div>
        </div>

        <!-- Content container -->
        <div id="venuesContent">
            @include('venues.partials._venues_grid')
        </div>
    </div>
</div>
@endsection

@push('layout_script')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        let searchTimeout;
        const searchInput = document.getElementById('search');
        const locationSelect = document.getElementById('location');
        const clearSearchBtn = document.getElementById('clearSearch');
        const loadingOverlay = document.getElementById('loadingOverlay');
        const venuesContent = document.getElementById('venuesContent');

        // Debounced search function
        function performSearch() {
            const searchValue = searchInput.value.trim();
            const locationValue = locationSelect.value;

            // Show loading
            loadingOverlay.classList.remove('d-none');
            
            // Build URL params
            const params = new URLSearchParams();
            if (searchValue) params.append('search', searchValue);
            if (locationValue) params.append('location', locationValue);

            // Update URL without reload
            const newUrl = new URL(window.location);
            newUrl.search = params.toString();
            window.history.replaceState({}, '', newUrl);

            // Make AJAX request
            fetch(`{{ route('venues.search') }}?${params.toString()}`, {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'text/html'
                }
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(data => {
                        throw new Error(data.error || 'Network response was not ok');
                    }).catch(() => {
                        throw new Error('Network response was not ok');
                    });
                }
                return response.text();
            })
            .then(html => {
                venuesContent.innerHTML = html;
                loadingOverlay.classList.add('d-none');
            })
            .catch(error => {
                console.error('Search error:', error);
                loadingOverlay.classList.add('d-none');
                venuesContent.innerHTML = `<div class="alert alert-danger text-center">${error.message}</div>`;
            });
        }

        // Search input with debounce
        searchInput.addEventListener('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(performSearch, 500);
        });

        // Location filter change
        locationSelect.addEventListener('change', performSearch);

        // Clear search
        clearSearchBtn.addEventListener('click', function() {
            searchInput.value = '';
            locationSelect.value = '';
            performSearch();
        });

        // Handle pagination clicks
        document.addEventListener('click', function(e) {
            if (e.target.closest('.pagination a')) {
                e.preventDefault();
                const link = e.target.closest('.pagination a');
                const url = new URL(link.href);
                
                // Add current search params to pagination
                const currentParams = new URLSearchParams(window.location.search);
                for (const [key, value] of currentParams) {
                    if (key !== 'page') {
                        url.searchParams.set(key, value);
                    }
                }
                
                loadingOverlay.classList.remove('d-none');
                
                fetch(`{{ route('venues.search') }}?${url.searchParams.toString()}`, {
                    method: 'GET',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'text/html'
                    }
                })
                .then(response => {
                    if (!response.ok) {
                        return response.json().then(data => {
                            throw new Error(data.error || 'Network response was not ok');
                        }).catch(() => {
                            throw new Error('Network response was not ok');
                        });
                    }
                    return response.text();
                })
                .then(html => {
                    venuesContent.innerHTML = html;
                    loadingOverlay.classList.add('d-none');
                    
                    // Update URL
                    window.history.replaceState({}, '', url);
                    
                    // Scroll to top of venues section
                    venuesContent.scrollIntoView({ behavior: 'smooth' });
                })
                .catch(error => {
                    console.error('Pagination error:', error);
                    loadingOverlay.classList.add('d-none');
                    venuesContent.innerHTML = `<div class="alert alert-danger text-center">${error.message}</div>`;
                });
            }
        });
    });
</script>
@endpush