/**
 * Tournament Mobile API Handler
 * Handles all mobile-specific tournament interactions
 */
class TournamentMobileAPI {
    constructor (tournamentId) {
        this.tournamentId = tournamentId;
        this.baseUrl = `/api/tournaments/${tournamentId}`;
        this.cache = new Map();
        this.retryDelay = 1000; // 1 second
        this.maxRetries = 3;
    }

    /**
     * Authenticate user with tournament
     */
    async authenticate(password, userType = 'auto') {
        try {
            const response = await this.fetchWithRetry(`${this.baseUrl}/auth`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCSRFToken()
                },
                body: JSON.stringify({ password, user_type: userType })
            });

            const result = await response.json();

            if (result.success) {
                this.showToast('Xác thực thành công!', 'success');
                // Store session info
                localStorage.setItem(`tournament_${this.tournamentId}_session`, JSON.stringify(result.permissions));
                return result;
            } else {
                this.showToast(result.message || 'Xác thực thất bại', 'error');
                return result;
            }
        } catch (error) {
            console.error('Authentication error:', error);
            this.showToast('Lỗi kết nối. Vui lòng thử lại.', 'error');
            return { success: false, message: 'Network error' };
        }
    }

    /**
     * Update match score
     */
    async updateMatchScore(matchId, score1, score2, status = 'Hoàn thành') {
        const loadingElement = document.querySelector(`[data-match-id="${matchId}"] .loading-spinner-mobile`);
        if (loadingElement) loadingElement.style.display = 'inline-block';

        try {
            const response = await this.fetchWithRetry(`/api/tournaments/matches/${matchId}/update-score`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCSRFToken()
                },
                body: JSON.stringify({ score1, score2, status })
            });

            const result = await response.json();

            if (result.success) {
                this.showToast('Tỷ số đã được cập nhật!', 'success');
                // Update UI
                this.updateMatchUI(matchId, result.match);
                // Invalidate cache
                this.invalidateCache(['standings', 'matches']);
                return result;
            } else {
                this.showToast(result.message || 'Không thể cập nhật tỷ số', 'error');
                return result;
            }
        } catch (error) {
            console.error('Update score error:', error);
            this.showToast('Lỗi kết nối. Vui lòng thử lại.', 'error');
            return { success: false, message: 'Network error' };
        } finally {
            if (loadingElement) loadingElement.style.display = 'none';
        }
    }

    /**
     * Swap players in a pair
     */
    async swapPlayers(pairId) {
        const loadingElement = document.querySelector(`[data-pair-id="${pairId}"] .loading-spinner-mobile`);
        if (loadingElement) loadingElement.style.display = 'inline-block';

        try {
            const response = await this.fetchWithRetry(`/api/tournaments/pairs/${pairId}/swap-players`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCSRFToken()
                }
            });

            const result = await response.json();

            if (result.success) {
                this.showToast('Đã hoán đổi vị trí thành công!', 'success');
                // Update UI
                this.updatePairUI(pairId, result.pair);
                // Invalidate cache
                this.invalidateCache(['groups', 'standings']);
                return result;
            } else {
                this.showToast(result.message || 'Không thể hoán đổi vị trí', 'error');
                return result;
            }
        } catch (error) {
            console.error('Swap players error:', error);
            this.showToast('Lỗi kết nối. Vui lòng thử lại.', 'error');
            return { success: false, message: 'Network error' };
        } finally {
            if (loadingElement) loadingElement.style.display = 'none';
        }
    }

    /**
     * Get tournament data with caching
     */
    async getTournamentData(dataType) {
        // Check cache first
        const cacheKey = `${dataType}_${Date.now()}`;
        const cached = this.cache.get(dataType);

        if (cached && (Date.now() - cached.timestamp < 30000)) { // 30 seconds cache
            return cached.data;
        }

        try {
            const response = await this.fetchWithRetry(`${this.baseUrl}/data/${dataType}`);
            const result = await response.json();

            if (result.success) {
                // Cache the result
                this.cache.set(dataType, {
                    data: result.data,
                    timestamp: Date.now()
                });
                return result.data;
            } else {
                console.error('Failed to get tournament data:', result.message);
                return null;
            }
        } catch (error) {
            console.error('Get tournament data error:', error);
            return null;
        }
    }

    /**
     * Get matches with pagination
     */
    async getMatches(roundType = 'all', page = 1, perPage = 20) {
        try {
            const params = new URLSearchParams({
                round_type: roundType,
                page: page.toString(),
                per_page: perPage.toString()
            });

            const response = await this.fetchWithRetry(`${this.baseUrl}/matches?${params}`);
            const result = await response.json();

            if (result.success) {
                return result;
            } else {
                console.error('Failed to get matches:', result.message);
                return null;
            }
        } catch (error) {
            console.error('Get matches error:', error);
            return null;
        }
    }

    /**
     * Get session info
     */
    async getSessionInfo() {
        try {
            const response = await this.fetchWithRetry(`${this.baseUrl}/session`);
            const result = await response.json();

            if (result.success) {
                return result.session;
            } else {
                return null;
            }
        } catch (error) {
            console.error('Get session info error:', error);
            return null;
        }
    }

    /**
     * Logout from tournament
     */
    async logout() {
        try {
            const response = await this.fetchWithRetry(`${this.baseUrl}/logout`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCSRFToken()
                }
            });

            const result = await response.json();

            if (result.success) {
                // Clear session storage
                localStorage.removeItem(`tournament_${this.tournamentId}_session`);
                this.showToast('Đã đăng xuất thành công', 'success');
                // Reload page or redirect
                window.location.reload();
                return result;
            } else {
                this.showToast(result.message || 'Lỗi đăng xuất', 'error');
                return result;
            }
        } catch (error) {
            console.error('Logout error:', error);
            this.showToast('Lỗi kết nối', 'error');
            return { success: false, message: 'Network error' };
        }
    }

    /**
     * Fetch with retry mechanism
     */
    async fetchWithRetry(url, options = {}, retryCount = 0) {
        try {
            const response = await fetch(url, options);

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            return response;
        } catch (error) {
            if (retryCount < this.maxRetries) {
                console.warn(`Request failed, retrying... (${retryCount + 1}/${this.maxRetries})`);
                await this.delay(this.retryDelay * (retryCount + 1));
                return this.fetchWithRetry(url, options, retryCount + 1);
            } else {
                throw error;
            }
        }
    }

    /**
     * Update match UI after score update
     */
    updateMatchUI(matchId, matchData) {
        const matchElement = document.querySelector(`[data-match-id="${matchId}"]`);
        if (!matchElement) return;

        // Update score display
        const scoreElement = matchElement.querySelector('.match-score');
        if (scoreElement) {
            scoreElement.textContent = matchData.score;
        }

        // Update status
        const statusElement = matchElement.querySelector('.match-status');
        if (statusElement) {
            statusElement.textContent = matchData.status;
            statusElement.className = `match-status-mobile ${this.getStatusClass(matchData.status)}`;
        }

        // Update timestamp
        const timestampElement = matchElement.querySelector('.match-timestamp');
        if (timestampElement) {
            timestampElement.textContent = new Date(matchData.updated_at).toLocaleString('vi-VN');
        }
    }

    /**
     * Update pair UI after player swap
     */
    updatePairUI(pairId, pairData) {
        const pairElement = document.querySelector(`[data-pair-id="${pairId}"]`);
        if (!pairElement) return;

        // Trigger a visual feedback
        pairElement.style.transform = 'scale(1.05)';
        setTimeout(() => {
            pairElement.style.transform = 'scale(1)';
        }, 200);

        // Update last modified time
        const timestampElement = pairElement.querySelector('.pair-timestamp');
        if (timestampElement) {
            timestampElement.textContent = new Date(pairData.updated_at).toLocaleString('vi-VN');
        }

        // Update player names and avatars if pair data is available
        if (pairData.player1 && pairData.player2) {
            // Find and update player names
            const playerNameElements = pairElement.querySelectorAll('.player-name');
            if (playerNameElements.length >= 2) {
                playerNameElements[0].textContent = pairData.player1.name;
                playerNameElements[0].setAttribute('data-player-id', pairData.player1.id);
                playerNameElements[1].textContent = pairData.player2.name;
                playerNameElements[1].setAttribute('data-player-id', pairData.player2.id);
            }

            // Update avatars
            const avatarElements = pairElement.querySelectorAll('.avatar, .avatar-placeholder');
            if (avatarElements.length >= 2) {
                // Update avatar for player 1
                if (pairData.player1.avatar) {
                    avatarElements[0].outerHTML = `<img src="${window.location.origin}/${pairData.player1.avatar}" alt="Avatar" class="avatar">`;
                } else {
                    const initial = pairData.player1.name.charAt(0).toUpperCase();
                    const colors = ['#3498db', '#e74c3c', '#2ecc71', '#f39c12', '#9b59b6', '#1abc9c', '#e67e22', '#34495e'];
                    const bgColor = colors[pairData.player1.id % colors.length];
                    avatarElements[0].outerHTML = `<div class="avatar-placeholder" style="background-color: ${bgColor};">${initial}</div>`;
                }

                // Update avatar for player 2
                if (pairData.player2.avatar) {
                    avatarElements[1].outerHTML = `<img src="${window.location.origin}/${pairData.player2.avatar}" alt="Avatar" class="avatar">`;
                } else {
                    const initial = pairData.player2.name.charAt(0).toUpperCase();
                    const colors = ['#3498db', '#e74c3c', '#2ecc71', '#f39c12', '#9b59b6', '#1abc9c', '#e67e22', '#34495e'];
                    const bgColor = colors[pairData.player2.id % colors.length];
                    avatarElements[1].outerHTML = `<div class="avatar-placeholder" style="background-color: ${bgColor};">${initial}</div>`;
                }
            }
        }
    }

    /**
     * Show toast notification
     */
    showToast(message, type = 'info', duration = 3000) {
        // Remove existing toasts
        const existingToasts = document.querySelectorAll('.toast-mobile');
        existingToasts.forEach(toast => toast.remove());

        // Create new toast
        const toast = document.createElement('div');
        toast.className = `toast-mobile ${type}`;
        toast.textContent = message;

        // Add to page
        document.body.appendChild(toast);

        // Auto remove
        setTimeout(() => {
            if (toast.parentNode) {
                toast.parentNode.removeChild(toast);
            }
        }, duration);
    }

    /**
     * Get status CSS class
     */
    getStatusClass(status) {
        const statusMap = {
            'Chưa bắt đầu': 'not-started',
            'Đang diễn ra': 'in-progress',
            'Hoàn thành': 'completed',
            'Tạm hoãn': 'postponed',
            'Bị hủy': 'cancelled'
        };
        return statusMap[status] || 'not-started';
    }

    /**
     * Get CSRF token
     */
    getCSRFToken() {
        const token = document.querySelector('meta[name="csrf-token"]');
        return token ? token.getAttribute('content') : '';
    }

    /**
     * Invalidate cache
     */
    invalidateCache(types = []) {
        if (types.length === 0) {
            this.cache.clear();
        } else {
            types.forEach(type => this.cache.delete(type));
        }
    }

    /**
     * Delay utility
     */
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    /**
     * Check if user is online
     */
    isOnline() {
        return navigator.onLine;
    }

    /**
     * Initialize offline handling
     */
    initOfflineHandling() {
        window.addEventListener('online', () => {
            this.showToast('Đã kết nối lại internet', 'success');
        });

        window.addEventListener('offline', () => {
            this.showToast('Mất kết nối internet', 'warning');
        });
    }
}

/**
 * Tournament Mobile UI Handler
 * Handles mobile-specific UI interactions
 */
class TournamentMobileUI {
    constructor (tournamentId) {
        this.tournamentId = tournamentId;
        this.api = new TournamentMobileAPI(tournamentId);
        this.currentView = 'overview';
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.setupNavigation();
        this.setupModals();
        this.setupPullToRefresh();
        this.api.initOfflineHandling();
    }

    /**
     * Setup event listeners
     */
    setupEventListeners() {
        // Score update forms
        document.addEventListener('submit', (e) => {
            if (e.target.classList.contains('score-update-form')) {
                e.preventDefault();
                this.handleScoreUpdate(e.target);
            }
        });

        // Player swap buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('swap-players-btn')) {
                e.preventDefault();
                const pairId = e.target.dataset.pairId;
                this.handlePlayerSwap(pairId);
            }
        });

        // Auth form
        const authForm = document.getElementById('tournament-auth-form');
        if (authForm) {
            authForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleAuthentication(authForm);
            });
        }

        // Logout button
        const logoutBtn = document.getElementById('logout-btn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.handleLogout();
            });
        }
    }

    /**
     * Setup mobile navigation
     */
    setupNavigation() {
        const navItems = document.querySelectorAll('.tournament-nav-mobile .nav-item');
        navItems.forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const view = item.dataset.view;
                this.switchView(view);
            });
        });
    }

    /**
     * Setup modal handling
     */
    setupModals() {
        // Modal close buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal-close') || e.target.classList.contains('mobile-modal')) {
                if (e.target === e.currentTarget) {
                    this.closeModal();
                }
            }
        });

        // ESC key to close modals
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeModal();
            }
        });
    }

    /**
     * Setup pull to refresh
     */
    setupPullToRefresh() {
        let startY = 0;
        let pullDistance = 0;
        const pullThreshold = 100;
        let isPulling = false;

        document.addEventListener('touchstart', (e) => {
            if (window.scrollY === 0) {
                startY = e.touches[0].clientY;
                isPulling = true;
            }
        });

        document.addEventListener('touchmove', (e) => {
            if (!isPulling) return;

            pullDistance = e.touches[0].clientY - startY;
            if (pullDistance > 0 && pullDistance < pullThreshold * 2) {
                // Visual feedback for pull to refresh
                document.body.style.transform = `translateY(${pullDistance * 0.5}px)`;
            }
        });

        document.addEventListener('touchend', () => {
            if (isPulling && pullDistance > pullThreshold) {
                this.refreshData();
            }

            document.body.style.transform = '';
            isPulling = false;
            pullDistance = 0;
        });
    }

    /**
     * Handle score update
     */
    async handleScoreUpdate(form) {
        const formData = new FormData(form);
        const matchId = formData.get('match_id');
        const score1 = parseInt(formData.get('score1'));
        const score2 = parseInt(formData.get('score2'));
        const status = formData.get('status') || 'Hoàn thành';

        // Validation
        if (score1 === score2) {
            this.api.showToast('Tỷ số không thể hòa', 'error');
            return;
        }

        const result = await this.api.updateMatchScore(matchId, score1, score2, status);

        if (result.success) {
            this.closeModal();
            // Refresh standings if needed
            if (this.currentView === 'standings') {
                this.loadStandings();
            }
        }
    }

    /**
     * Handle player swap
     */
    async handlePlayerSwap(pairId) {
        if (!confirm('Bạn có chắc muốn hoán đổi vị trí 2 người chơi?')) {
            return;
        }

        const result = await this.api.swapPlayers(pairId);

        if (result.success && this.currentView === 'groups') {
            this.loadGroups();
        }
    }

    /**
     * Handle authentication
     */
    async handleAuthentication(form) {
        const formData = new FormData(form);
        const password = formData.get('password');
        const userType = formData.get('user_type') || 'auto';

        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.textContent = 'Đang xác thực...';
        submitBtn.disabled = true;

        try {
            const result = await this.api.authenticate(password, userType);

            if (result.success) {
                // Reload page to show authenticated content
                window.location.reload();
            }
        } finally {
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
        }
    }

    /**
     * Handle logout
     */
    async handleLogout() {
        if (!confirm('Bạn có chắc muốn đăng xuất?')) {
            return;
        }

        await this.api.logout();
    }

    /**
     * Switch view
     */
    switchView(view) {
        // Update navigation
        document.querySelectorAll('.tournament-nav-mobile .nav-item').forEach(item => {
            item.classList.remove('active');
        });
        document.querySelector(`[data-view="${view}"]`).classList.add('active');

        // Update content
        this.currentView = view;
        this.loadViewContent(view);
    }

    /**
     * Load view content
     */
    async loadViewContent(view) {
        const contentContainer = document.getElementById('tournament-content');
        if (!contentContainer) return;

        // Show loading
        contentContainer.innerHTML = '<div class="flex justify-center py-8"><div class="loading-spinner-mobile"></div></div>';

        let data;
        switch (view) {
            case 'standings':
                data = await this.api.getTournamentData('standings');
                this.renderStandings(data);
                break;
            case 'groups':
                data = await this.api.getTournamentData('groups');
                this.renderGroups(data);
                break;
            case 'matches':
                data = await this.api.getMatches();
                this.renderMatches(data);
                break;
            case 'rounds':
                data = await this.api.getTournamentData('rounds');
                this.renderRounds(data);
                break;
            default:
                this.renderOverview();
        }
    }

    /**
     * Show modal
     */
    showModal(content) {
        const modal = document.createElement('div');
        modal.className = 'mobile-modal';
        modal.innerHTML = `
            <div class="mobile-modal-content">
                <div class="mobile-modal-header">
                    <h3 class="text-lg font-semibold">Cập nhật</h3>
                    <button class="modal-close text-gray-500 hover:text-gray-700">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="mobile-modal-body">
                    ${content}
                </div>
            </div>
        `;

        document.body.appendChild(modal);

        // Prevent body scroll
        document.body.style.overflow = 'hidden';
    }

    /**
     * Close modal
     */
    closeModal() {
        const modal = document.querySelector('.mobile-modal');
        if (modal) {
            modal.remove();
            document.body.style.overflow = '';
        }
    }

    /**
     * Refresh data
     */
    async refreshData() {
        this.api.invalidateCache();
        this.api.showToast('Đang làm mới dữ liệu...', 'info');
        await this.loadViewContent(this.currentView);
        this.api.showToast('Đã cập nhật dữ liệu mới nhất', 'success');
    }

    // Additional rendering methods would go here...
    // renderStandings, renderGroups, renderMatches, etc.
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    const tournamentElement = document.querySelector('[data-tournament-id]');
    if (tournamentElement) {
        const tournamentId = tournamentElement.dataset.tournamentId;
        window.tournamentMobileUI = new TournamentMobileUI(tournamentId);
    }
});