/**
 * Job Status Polling Handler for Tournament Operations
 */
class TournamentJobPoller {
    constructor () {
        this.pollInterval = 2000; // 2 seconds
        this.maxAttempts = 150; // 5 minutes maximum
        this.currentPolls = new Map();
    }

    /**
     * Start polling for job status
     */
    startPolling(jobType, tournamentId, options = {}) {
        const pollId = `${tournamentId}_${jobType}`;

        // Stop existing polling if any
        this.stopPolling(pollId);

        const config = {
            url: `/api/tournaments/${tournamentId}/job-status?type=${jobType}`,
            attempts: 0,
            maxAttempts: options.maxAttempts || this.maxAttempts,
            onProgress: options.onProgress || this.defaultProgressHandler,
            onComplete: options.onComplete || this.defaultCompleteHandler,
            onError: options.onError || this.defaultErrorHandler,
            ...options
        };

        console.log(`Starting job polling for ${jobType} on tournament ${tournamentId}`);

        this.poll(pollId, config);
    }

    /**
     * Stop polling for a specific job
     */
    stopPolling(pollId) {
        if (this.currentPolls.has(pollId)) {
            clearTimeout(this.currentPolls.get(pollId).timeoutId);
            this.currentPolls.delete(pollId);
            console.log(`Stopped polling for ${pollId}`);
        }
    }

    /**
     * Stop all polling
     */
    stopAllPolling() {
        this.currentPolls.forEach((poll, pollId) => {
            clearTimeout(poll.timeoutId);
            console.log(`Stopped polling for ${pollId}`);
        });
        this.currentPolls.clear();
    }

    /**
     * Poll for job status
     */
    async poll(pollId, config) {
        try {
            const response = await fetch(config.url, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            const jobData = data.data;

            console.log(`Poll ${config.attempts + 1}: Status = ${jobData.status}, Progress = ${jobData.progress}%`);

            // Handle different statuses
            switch (jobData.status) {
                case 'completed':
                    this.handleComplete(pollId, jobData, config);
                    return;

                case 'failed':
                    this.handleError(pollId, jobData, config);
                    return;

                case 'processing':
                case 'queued':
                    this.handleProgress(pollId, jobData, config);
                    break;

                default:
                    console.warn(`Unknown job status: ${jobData.status}`);
            }

            // Continue polling if not finished
            config.attempts++;

            if (config.attempts >= config.maxAttempts) {
                this.handleTimeout(pollId, config);
                return;
            }

            // Schedule next poll
            const timeoutId = setTimeout(() => {
                this.poll(pollId, config);
            }, this.pollInterval);

            this.currentPolls.set(pollId, { timeoutId, config });

        } catch (error) {
            console.error(`Polling error for ${pollId}:`, error);
            this.handleError(pollId, { message: error.message }, config);
        }
    }

    /**
     * Handle job completion
     */
    handleComplete(pollId, jobData, config) {
        this.stopPolling(pollId);
        console.log(`Job completed: ${pollId}`);
        config.onComplete(jobData);
    }

    /**
     * Handle job progress
     */
    handleProgress(pollId, jobData, config) {
        config.onProgress(jobData);
    }

    /**
     * Handle job error
     */
    handleError(pollId, jobData, config) {
        this.stopPolling(pollId);
        console.error(`Job failed: ${pollId}`, jobData);
        config.onError(jobData);
    }

    /**
     * Handle polling timeout
     */
    handleTimeout(pollId, config) {
        this.stopPolling(pollId);
        console.warn(`Polling timeout for ${pollId}`);
        config.onError({
            message: 'Quá thời gian chờ. Vui lòng refresh trang để kiểm tra kết quả.',
            timeout: true
        });
    }

    /**
     * Default progress handler
     */
    defaultProgressHandler(jobData) {
        console.log(`Progress: ${jobData.progress}% - ${jobData.message}`);

        // Update progress bar if exists
        const progressBar = document.querySelector('.job-progress-bar');
        if (progressBar) {
            progressBar.style.width = `${jobData.progress}%`;
            progressBar.setAttribute('aria-valuenow', jobData.progress);
        }

        // Update status message if exists
        const statusMessage = document.querySelector('.job-status-message');
        if (statusMessage) {
            statusMessage.textContent = jobData.message;
        }
    }

    /**
     * Default completion handler
     */
    defaultCompleteHandler(jobData) {
        console.log('Job completed successfully:', jobData.message);

        // Show success message
        this.showMessage(jobData.message, 'success');

        // Reload page after short delay
        setTimeout(() => {
            window.location.reload();
        }, 2000);
    }

    /**
     * Default error handler
     */
    defaultErrorHandler(jobData) {
        console.error('Job failed:', jobData.message);

        // Show error message
        this.showMessage(jobData.message || 'Có lỗi xảy ra', 'error');
    }

    /**
     * Show message to user
     */
    showMessage(message, type = 'info') {
        // Try to use existing notification system
        if (typeof Swal !== 'undefined') {
            Swal.fire({
                text: message,
                icon: type === 'error' ? 'error' : type === 'success' ? 'success' : 'info',
                timer: type === 'success' ? 3000 : 0,
                timerProgressBar: true
            });
            return;
        }

        // Fallback to browser alert
        if (type === 'error') {
            alert('Lỗi: ' + message);
        } else {
            alert(message);
        }
    }

    /**
     * Create progress modal
     */
    createProgressModal(title = 'Đang xử lý...') {
        const modal = document.createElement('div');
        modal.className = 'modal fade job-progress-modal';
        modal.setAttribute('data-bs-backdrop', 'static');
        modal.innerHTML = `
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">${title}</h5>
                    </div>
                    <div class="modal-body text-center">
                        <div class="mb-3">
                            <div class="progress" style="height: 20px;">
                                <div class="progress-bar job-progress-bar progress-bar-striped progress-bar-animated" 
                                     role="progressbar" style="width: 0%" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100">
                                </div>
                            </div>
                        </div>
                        <p class="job-status-message text-muted">Đang khởi tạo...</p>
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(modal);

        // Show modal
        const bootstrapModal = new bootstrap.Modal(modal);
        bootstrapModal.show();

        return {
            modal: modal,
            bootstrapModal: bootstrapModal,
            hide: () => {
                bootstrapModal.hide();
                setTimeout(() => {
                    document.body.removeChild(modal);
                }, 300);
            }
        };
    }
}

// Global instance
window.tournamentJobPoller = new TournamentJobPoller();

// Enhanced form submission handler
function handleTournamentFormSubmission(form, jobType) {
    form.addEventListener('submit', async function (e) {
        e.preventDefault();

        const formData = new FormData(form);
        const submitButton = form.querySelector('button[type="submit"]');
        const originalText = submitButton.textContent;

        // Disable submit button
        submitButton.disabled = true;
        submitButton.textContent = 'Đang xử lý...';

        try {
            const response = await fetch(form.action, {
                method: 'POST',
                body: formData,
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            const data = await response.json();

            if (data.success) {
                if (data.use_polling) {
                    // Create progress modal
                    const progressModal = window.tournamentJobPoller.createProgressModal(
                        jobType === 'groups' ? 'Đang tạo bảng đấu...' :
                            jobType === 'rounds' ? 'Đang tạo vòng đấu...' :
                                jobType === 'playoffs' ? 'Đang tạo playoffs...' : 'Đang xử lý...'
                    );

                    // Start polling
                    const tournamentId = form.getAttribute('data-tournament-id') ||
                        window.location.pathname.match(/\/tournaments\/(\d+)/)?.[1];

                    window.tournamentJobPoller.startPolling(jobType, tournamentId, {
                        onComplete: (jobData) => {
                            progressModal.hide();
                            window.tournamentJobPoller.showMessage(jobData.message, 'success');
                        },
                        onError: (jobData) => {
                            progressModal.hide();
                            window.tournamentJobPoller.showMessage(jobData.message, 'error');
                        }
                    });
                } else {
                    // Direct success
                    window.tournamentJobPoller.showMessage(data.message, 'success');
                }
            } else {
                window.tournamentJobPoller.showMessage(data.message, 'error');
            }

        } catch (error) {
            console.error('Form submission error:', error);
            window.tournamentJobPoller.showMessage('Có lỗi xảy ra khi gửi yêu cầu', 'error');
        } finally {
            // Re-enable submit button
            submitButton.disabled = false;
            submitButton.textContent = originalText;
        }
    });
}

// Auto-initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function () {
    // Find and enhance tournament forms
    const groupsForm = document.querySelector('form[action*="auto-generate-groups"]');
    if (groupsForm) {
        handleTournamentFormSubmission(groupsForm, 'groups');
    }

    const roundsForm = document.querySelector('form[action*="auto-generate-group-rounds"]');
    if (roundsForm) {
        handleTournamentFormSubmission(roundsForm, 'rounds');
    }

    const playoffsForm = document.querySelector('form[action*="generate-playoffs"]');
    if (playoffsForm) {
        handleTournamentFormSubmission(playoffsForm, 'playoffs');
    }
});

// Cleanup when page unloads
window.addEventListener('beforeunload', function () {
    if (window.tournamentJobPoller) {
        window.tournamentJobPoller.stopAllPolling();
    }
});