/**
 * Advanced Image Lazy Loading and Performance Optimization
 * Sử dụng Intersection Observer API và WebP detection
 */

class ImageOptimizer {
    constructor () {
        this.isWebPSupported = null;
        this.imageObserver = null;
        this.loadedImages = new Set();

        this.init();
    }

    async init() {
        // Detect WebP support
        await this.detectWebPSupport();

        // Initialize Intersection Observer
        this.initIntersectionObserver();

        // Process existing images
        this.processExistingImages();

        // Listen for new images added dynamically
        this.observeNewImages();
    }

    /**
     * Detect WebP support using Canvas
     */
    detectWebPSupport() {
        return new Promise((resolve) => {
            const canvas = document.createElement('canvas');
            canvas.width = 1;
            canvas.height = 1;

            canvas.toBlob((blob) => {
                this.isWebPSupported = blob && blob.type === 'image/webp';
                resolve(this.isWebPSupported);
            }, 'image/webp');
        });
    }

    /**
     * Initialize Intersection Observer for lazy loading
     */
    initIntersectionObserver() {
        if (!('IntersectionObserver' in window)) {
            // Fallback for older browsers
            this.loadAllImages();
            return;
        }

        this.imageObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    this.loadImage(entry.target);
                    this.imageObserver.unobserve(entry.target);
                }
            });
        }, {
            // Load images 100px before they enter viewport
            rootMargin: '100px 0px',
            threshold: 0.1
        });
    }

    /**
     * Process images that are already in the DOM
     */
    processExistingImages() {
        const lazyImages = document.querySelectorAll('img[data-src], source[data-srcset]');

        lazyImages.forEach(element => {
            if (element.tagName === 'IMG') {
                this.observeImage(element);
            } else if (element.tagName === 'SOURCE') {
                this.observeImage(element.closest('picture')?.querySelector('img'));
            }
        });
    }

    /**
     * Observe a single image for lazy loading
     */
    observeImage(img) {
        if (!img || this.loadedImages.has(img)) return;

        if (this.imageObserver) {
            this.imageObserver.observe(img);
        } else {
            this.loadImage(img);
        }
    }

    /**
     * Load an image with WebP optimization
     */
    async loadImage(img) {
        if (this.loadedImages.has(img)) return;

        try {
            // Handle picture element
            const picture = img.closest('picture');
            if (picture) {
                this.loadPictureElement(picture);
            } else {
                this.loadSingleImage(img);
            }

            this.loadedImages.add(img);

        } catch (error) {
            console.warn('Failed to load image:', error);
            this.handleImageError(img);
        }
    }

    /**
     * Load picture element with WebP support
     */
    loadPictureElement(picture) {
        const sources = picture.querySelectorAll('source[data-srcset]');
        const img = picture.querySelector('img');

        // Load WebP source if supported
        if (this.isWebPSupported) {
            const webpSource = Array.from(sources).find(s =>
                s.getAttribute('type') === 'image/webp'
            );

            if (webpSource) {
                webpSource.srcset = webpSource.dataset.srcset;
                webpSource.removeAttribute('data-srcset');
            }
        }

        // Load fallback sources
        sources.forEach(source => {
            if (source.dataset.srcset) {
                source.srcset = source.dataset.srcset;
                source.removeAttribute('data-srcset');
            }
        });

        // Load main image
        if (img && img.dataset.src) {
            this.loadSingleImage(img);
        }
    }

    /**
     * Load a single image element
     */
    loadSingleImage(img) {
        const dataSrc = img.dataset.src;

        if (!dataSrc) return;

        // Create a new image to preload
        const newImg = new Image();

        newImg.onload = () => {
            img.src = dataSrc;
            img.removeAttribute('data-src');
            img.classList.add('loaded');

            // Remove placeholder blur effect
            img.style.filter = '';
        };

        newImg.onerror = () => {
            this.handleImageError(img);
        };

        // Start loading
        newImg.src = dataSrc;
    }

    /**
     * Handle image loading errors
     */
    handleImageError(img) {
        img.src = '/assets/images/default.jpg';
        img.classList.add('error');
        img.removeAttribute('data-src');
    }

    /**
     * Fallback: load all images immediately (for older browsers)
     */
    loadAllImages() {
        const images = document.querySelectorAll('img[data-src]');
        const sources = document.querySelectorAll('source[data-srcset]');

        images.forEach(img => this.loadImage(img));
        sources.forEach(source => {
            if (source.dataset.srcset) {
                source.srcset = source.dataset.srcset;
                source.removeAttribute('data-srcset');
            }
        });
    }

    /**
     * Observe for new images added to DOM
     */
    observeNewImages() {
        if (!('MutationObserver' in window)) return;

        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === 1) { // Element node
                        // Check if the node itself is a lazy image
                        if (node.tagName === 'IMG' && node.dataset.src) {
                            this.observeImage(node);
                        }

                        // Check for lazy images within the node
                        const lazyImages = node.querySelectorAll &&
                            node.querySelectorAll('img[data-src]');

                        if (lazyImages) {
                            lazyImages.forEach(img => this.observeImage(img));
                        }
                    }
                });
            });
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    /**
     * Preload critical images
     */
    preloadCriticalImages() {
        const criticalImages = document.querySelectorAll('img[data-priority="high"]');

        criticalImages.forEach(img => {
            this.loadImage(img);
        });
    }

    /**
     * Cleanup method
     */
    destroy() {
        if (this.imageObserver) {
            this.imageObserver.disconnect();
        }

        this.loadedImages.clear();
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.imageOptimizer = new ImageOptimizer();
});

// Preload critical images immediately
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        window.imageOptimizer?.preloadCriticalImages();
    });
} else {
    window.imageOptimizer?.preloadCriticalImages();
}