<?php

namespace App\View\Components;

use Illuminate\View\Component;

class ResponsiveImage extends Component
{
    public string $src;
    public string $alt;
    public string $class;
    public string $sizes;
    public bool $lazy;
    public string $aspectRatio;
    public array $breakpoints;
    
    public function __construct(
        string $src,
        string $alt = '',
        string $class = '',
        string $sizes = '(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw',
        bool $lazy = true,
        string $aspectRatio = 'auto',
        array $breakpoints = []
    ) {
        $this->src = $src;
        $this->alt = $alt;
        $this->class = $class;
        $this->sizes = $sizes;
        $this->lazy = $lazy;
        $this->aspectRatio = $aspectRatio;
        $this->breakpoints = $breakpoints;
    }
    
    public function render()
    {
        return view('components.responsive-image');
    }
    
    /**
     * Tạo srcset cho responsive images
     */
    public function getSrcSet(): string
    {
        $basePath = pathinfo($this->src, PATHINFO_DIRNAME);
        $filename = pathinfo($this->src, PATHINFO_FILENAME);
        
        $srcSet = [];
        $sizes = ['thumb' => '300w', 'medium' => '800w', 'large' => '1200w'];
        
        foreach ($sizes as $size => $width) {
            $webpPath = $basePath . '/' . $size . '/' . $filename . '.webp';
            $jpegPath = $basePath . '/' . $size . '/' . $filename . '.jpg';
            
            // Check if optimized versions exist
            if (file_exists(public_path($webpPath))) {
                $srcSet[] = asset($webpPath) . ' ' . $width;
            } elseif (file_exists(public_path($jpegPath))) {
                $srcSet[] = asset($jpegPath) . ' ' . $width;
            }
        }
        
        return implode(', ', $srcSet);
    }
    
    /**
     * Tạo picture element sources
     */
    public function getPictureSources(): array
    {
        $basePath = pathinfo($this->src, PATHINFO_DIRNAME);
        $filename = pathinfo($this->src, PATHINFO_FILENAME);
        
        $sources = [];
        
        // WebP sources (modern browsers)
        $webpSrcSet = [];
        $jpegSrcSet = [];
        
        $sizes = ['thumb' => 300, 'medium' => 800, 'large' => 1200];
        
        foreach ($sizes as $size => $width) {
            $webpPath = $basePath . '/' . $size . '/' . $filename . '.webp';
            $jpegPath = $basePath . '/' . $size . '/' . $filename . '.jpg';
            
            if (file_exists(public_path($webpPath))) {
                $webpSrcSet[] = asset($webpPath) . ' ' . $width . 'w';
            }
            
            if (file_exists(public_path($jpegPath))) {
                $jpegSrcSet[] = asset($jpegPath) . ' ' . $width . 'w';
            }
        }
        
        if (!empty($webpSrcSet)) {
            $sources[] = [
                'type' => 'image/webp',
                'srcset' => implode(', ', $webpSrcSet)
            ];
        }
        
        if (!empty($jpegSrcSet)) {
            $sources[] = [
                'type' => 'image/jpeg',
                'srcset' => implode(', ', $jpegSrcSet)
            ];
        }
        
        return $sources;
    }
}