<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class UserService
{
    /**
     * Tạo người dùng mới với dữ liệu được cung cấp.
     *
     * @throws \Exception
     */
    public function createUser(array $data): User
    {
        return DB::transaction(function () use ($data) {
            $user = User::create([
                'name' => $data['name'],
                'email' => $data['email'],
                'password' => Hash::make($data['password']),
                'phone' => $data['phone'] ?? null,
                'location' => $data['location'] ?? null,
                'gender' => $data['gender'] ?? null,
            ]);

            if (isset($data['avatar']) && $data['avatar'] instanceof UploadedFile) {
                $this->handleAvatarUpload($user, $data['avatar']);
            }

            if (! empty($data['role'])) {
                $user->assignRole($data['role']);
            }

            return $user;
        });
    }

    /**
     * Cập nhật thông tin người dùng.
     *
     * @throws \Exception
     */
    public function updateUser(User $user, array $data): User
    {
        return DB::transaction(function () use ($user, $data) {
            $updateData = [
                'name' => $data['name'],
                'email' => $data['email'],
                'phone' => $data['phone'] ?? null,
                'location' => $data['location'] ?? null,
                'gender' => $data['gender'] ?? null,
            ];

            if (! empty($data['password'])) {
                $updateData['password'] = Hash::make($data['password']);
            }

            $user->update($updateData);

            if (isset($data['avatar']) && $data['avatar'] instanceof UploadedFile) {
                $this->handleAvatarUpload($user, $data['avatar']);
            }

            if (! empty($data['role'])) {
                $user->syncRoles([$data['role']]);
            }

            return $user;
        });
    }

    /**
     * Xử lý tải lên và cập nhật avatar cho người dùng.
     */
    private function handleAvatarUpload(User $user, UploadedFile $file): void
    {
        try {
            // Sử dụng ImageOptimizationService để tối ưu hóa ảnh
            $imageService = app(\App\Services\ImageOptimizationService::class);
            $optimizedImages = $imageService->optimizeAndStore(
                $file, 
                'avatar', 
                $user->avatar
            );
            
            // Cập nhật đường dẫn avatar trong database (sử dụng medium size)
            $user->update(['avatar' => $optimizedImages['medium']['jpeg']]);

        } catch (\Exception $e) {
            // Log lỗi và ném exception để transaction rollback
            Log::error('Avatar upload failed: '.$e->getMessage());
            throw new \Exception('Không thể tải lên avatar: '.$e->getMessage());
        }
    }
}
