<?php

namespace App\Services;

use App\Models\Tournament;
use App\Models\TournamentRound;
use Illuminate\Support\Facades\Cache;

class TournamentStateService
{
    /**
     * Enhanced tournament state validation with caching
     */
    public function validateTournamentState(Tournament $tournament): array
    {
        $cacheKey = "tournament_state_{$tournament->id}";

        return Cache::remember($cacheKey, 300, function () use ($tournament) {
            return $this->performStateValidation($tournament);
        });
    }

    private function performStateValidation(Tournament $tournament): array
    {
        $errors = [];
        $warnings = [];
        $recommendations = [];

        // Basic tournament validation
        if (empty($tournament->name)) {
            $errors[] = 'Tên giải đấu không được để trống';
        }

        if (! $tournament->start_date) {
            $errors[] = 'Ngày bắt đầu giải đấu không được để trống';
        }

        if ($tournament->start_date && $tournament->end_date && $tournament->start_date > $tournament->end_date) {
            $errors[] = 'Ngày bắt đầu không thể sau ngày kết thúc';
        }

        // Registration validation
        $approvedRegistrations = $tournament->registrations()->where('status', 'approved')->count();
        $pendingRegistrations = $tournament->registrations()->where('status', 'pending')->count();

        if ($approvedRegistrations < 4) {
            $errors[] = "Cần ít nhất 4 cặp đấu được duyệt để tạo giải (hiện có: {$approvedRegistrations})";
        }

        if ($tournament->max_pairs && $approvedRegistrations > $tournament->max_pairs) {
            $warnings[] = "Số cặp đấu được duyệt ({$approvedRegistrations}) vượt quá giới hạn ({$tournament->max_pairs})";
        }

        if ($pendingRegistrations > 0) {
            $recommendations[] = "Còn {$pendingRegistrations} đăng ký chờ duyệt";
        }

        // Tournament structure validation
        $structureValidation = $this->validateTournamentStructure($tournament);
        $errors = array_merge($errors, $structureValidation['errors']);
        $warnings = array_merge($warnings, $structureValidation['warnings']);
        $recommendations = array_merge($recommendations, $structureValidation['recommendations']);

        return [
            'errors' => $errors,
            'warnings' => $warnings,
            'recommendations' => $recommendations,
            'can_proceed' => empty($errors),
            'validation_timestamp' => now(),
            'tournament_health_score' => $this->calculateHealthScore($errors, $warnings),
        ];
    }

    /**
     * Validate tournament structure integrity
     */
    private function validateTournamentStructure(Tournament $tournament): array
    {
        $errors = [];
        $warnings = [];
        $recommendations = [];

        $groups = $tournament->groups()->with('pairs')->get();
        $rounds = $tournament->rounds()->with('matches')->get();

        // Group validation
        if ($groups->isNotEmpty()) {
            $totalPairs = $groups->sum(fn ($g) => $g->pairs->count());
            $approvedRegistrations = $tournament->registrations()->where('status', 'approved')->count();

            if ($totalPairs !== $approvedRegistrations) {
                $warnings[] = "Số cặp trong bảng ({$totalPairs}) khác số đăng ký được duyệt ({$approvedRegistrations})";
            }

            foreach ($groups as $group) {
                if ($group->pairs->count() < 2) {
                    $errors[] = "Bảng {$group->name} chỉ có {$group->pairs->count()} cặp (tối thiểu 2)";
                }

                if ($group->pairs->count() > 6) {
                    $recommendations[] = "Bảng {$group->name} có {$group->pairs->count()} cặp (khuyến nghị ≤ 6)";
                }
            }
        }

        // Round validation
        $roundRobinRounds = $rounds->where('type', 'round_robin');
        $playoffRounds = $rounds->where('type', 'playoff');

        if ($roundRobinRounds->isNotEmpty()) {
            foreach ($roundRobinRounds as $round) {
                $expectedMatches = $this->calculateExpectedGroupMatches($groups);
                $actualMatches = $round->matches->count();

                if ($actualMatches !== $expectedMatches) {
                    $warnings[] = "Vòng bảng có {$actualMatches} trận, dự kiến {$expectedMatches} trận";
                }
            }
        }

        // Playoff validation
        if ($playoffRounds->isNotEmpty()) {
            $playoffValidation = $this->validatePlayoffStructure($playoffRounds);
            $errors = array_merge($errors, $playoffValidation['errors']);
            $warnings = array_merge($warnings, $playoffValidation['warnings']);
        }

        return compact('errors', 'warnings', 'recommendations');
    }

    /**
     * Calculate expected number of group stage matches
     */
    private function calculateExpectedGroupMatches($groups): int
    {
        $totalMatches = 0;
        foreach ($groups as $group) {
            $n = $group->pairs->count();
            $totalMatches += $n * ($n - 1) / 2; // C(n,2) combinations
        }

        return $totalMatches;
    }

    /**
     * Validate playoff bracket structure
     */
    private function validatePlayoffStructure($playoffRounds): array
    {
        $errors = [];
        $warnings = [];

        $existingLevels = $playoffRounds
            ->map(fn (TournamentRound $round) => $this->resolvePlayoffLevel($round))
            ->filter()
            ->unique()
            ->toArray();

        // Check logical round progression
        if (in_array('final', $existingLevels) && ! in_array('semi_final', $existingLevels)) {
            $errors[] = 'Không thể có chung kết mà không có bán kết';
        }

        if (in_array('semi_final', $existingLevels) && ! in_array('quarter_final', $existingLevels)) {
            $warnings[] = 'Có bán kết nhưng không có tứ kết (có thể bình thường tùy số đội)';
        }

        // Validate match counts in each round
        foreach ($playoffRounds as $round) {
            $level = $this->resolvePlayoffLevel($round);
            $expectedMatches = $this->getExpectedPlayoffMatchesForLevel($level, $round);
            $actualMatches = $round->matches->filter(fn ($match) => $match->status !== 'Bye')->count();

            if ($expectedMatches && $actualMatches !== $expectedMatches) {
                $warnings[] = "Vòng {$round->name} có {$actualMatches} trận, dự kiến {$expectedMatches}";
            }
        }

        return compact('errors', 'warnings');
    }

    private function getExpectedPlayoffMatchesForLevel(?string $level, TournamentRound $round): ?int
    {
        if (! $level) {
            return null;
        }

        $matches = match ($level) {
            'quarter_final' => 4,
            'semi_final' => 2,
            'final' => 1,
            'third_place' => 1,
            default => null,
        };

        if ($matches !== null) {
            return $matches;
        }

        if (str_starts_with($level, 'round_of_')) {
            $teams = (int) str_replace('round_of_', '', $level);
            return $teams > 1 ? (int) ($teams / 2) : null;
        }

        $meta = $this->decodeMetaData($round->meta_data);
        if (isset($meta['total_teams'])) {
            $teams = (int) $meta['total_teams'];
            return $teams > 1 ? (int) ($teams / 2) : null;
        }

        return null;
    }

    /**
     * Calculate tournament health score (0-100)
     */
    private function calculateHealthScore(array $errors, array $warnings): int
    {
        $score = 100;
        $score -= count($errors) * 20;      // Each error -20 points
        $score -= count($warnings) * 5;     // Each warning -5 points

        return max(0, $score);
    }

    /**
     * Enhanced current phase detection
     */
    public function getCurrentPhase(Tournament $tournament): array
    {
        $hasGroups = $tournament->groups()->exists();
        $hasGroupRounds = $tournament->rounds()->where('type', 'round_robin')->exists();
        $hasPlayoffs = $tournament->rounds()->where('type', 'playoff')->exists();

        if (! $hasGroups) {
            return [
                'phase' => 'setup',
                'name' => 'Thiết lập',
                'description' => 'Chưa tạo bảng đấu',
                'progress' => 0,
                'next_actions' => ['Tạo bảng đấu', 'Duyệt đăng ký'],
            ];
        }

        if (! $hasGroupRounds) {
            return [
                'phase' => 'groups_created',
                'name' => 'Đã tạo bảng',
                'description' => 'Đã tạo bảng đấu, chưa tạo vòng bảng',
                'progress' => 20,
                'next_actions' => ['Tạo vòng bảng'],
            ];
        }

        // Analyze group stage progress
        $groupProgress = $this->analyzeGroupStageProgress($tournament);

        if ($groupProgress['completion_percentage'] < 100) {
            return [
                'phase' => 'group_stage',
                'name' => 'Vòng bảng',
                'description' => "Vòng bảng đang diễn ra ({$groupProgress['completion_percentage']}% hoàn thành)",
                'progress' => 20 + ($groupProgress['completion_percentage'] * 0.4),
                'next_actions' => ['Cập nhật kết quả trận đấu'],
                'details' => $groupProgress,
            ];
        }

        if (! $hasPlayoffs) {
            return [
                'phase' => 'group_completed',
                'name' => 'Vòng bảng hoàn thành',
                'description' => 'Vòng bảng đã hoàn thành, sẵn sàng tạo playoffs',
                'progress' => 60,
                'next_actions' => ['Tạo vòng playoffs'],
            ];
        }

        // Analyze playoff progress
        $playoffProgress = $this->analyzePlayoffProgress($tournament);

        return [
            'phase' => $playoffProgress['phase'],
            'name' => $playoffProgress['name'],
            'description' => $playoffProgress['description'],
            'progress' => $playoffProgress['progress'],
            'next_actions' => $playoffProgress['next_actions'],
            'details' => $playoffProgress,
        ];
    }

    /**
     * Analyze group stage completion
     */
    private function analyzeGroupStageProgress(Tournament $tournament): array
    {
        $groupRounds = $tournament->rounds()->where('type', 'round_robin')->with('matches')->get();

        if ($groupRounds->isEmpty()) {
            return ['completion_percentage' => 0, 'matches_completed' => 0, 'total_matches' => 0];
        }

        $totalMatches = $groupRounds->sum(fn ($round) => $round->matches->count());
        $completedMatches = $groupRounds->sum(fn ($round) => $round->matches->where('status', 'Hoàn thành')->count());

        $completionPercentage = $totalMatches > 0 ? round(($completedMatches / $totalMatches) * 100, 1) : 0;

        return [
            'completion_percentage' => $completionPercentage,
            'matches_completed' => $completedMatches,
            'total_matches' => $totalMatches,
            'rounds' => $groupRounds->count(),
        ];
    }

    /**
     * Analyze playoff progress
     */
    private function analyzePlayoffProgress(Tournament $tournament): array
    {
        $playoffRounds = $tournament->rounds()->where('type', 'playoff')->orderBy('created_at')->with('matches')->get();

        if ($playoffRounds->isEmpty()) {
            return [
                'phase' => 'playoffs_ready',
                'name' => 'Sẵn sàng playoffs',
                'description' => 'Chưa tạo vòng playoffs',
                'progress' => 60,
                'next_actions' => ['Tạo playoffs'],
            ];
        }

        $allPlayoffMatches = $playoffRounds->sum(fn ($round) => $round->matches->count());
        $completedPlayoffMatches = $playoffRounds->sum(fn ($round) => $round->matches->where('status', 'Hoàn thành')->count());

        // Check for final round
        $finalRound = $playoffRounds->first(function (TournamentRound $round) {
            return $this->resolvePlayoffLevel($round) === 'final';
        });
        if ($finalRound && $finalRound->matches->where('status', 'Hoàn thành')->isNotEmpty()) {
            return [
                'phase' => 'completed',
                'name' => 'Hoàn thành',
                'description' => 'Giải đấu đã hoàn thành',
                'progress' => 100,
                'next_actions' => ['Xuất kết quả', 'Lưu trữ'],
            ];
        }

        $playoffProgress = 60 + ($allPlayoffMatches > 0 ? ($completedPlayoffMatches / $allPlayoffMatches) * 40 : 0);

        return [
            'phase' => 'playoffs_ongoing',
            'name' => 'Playoffs đang diễn ra',
            'description' => "Đang trong vòng playoffs ({$completedPlayoffMatches}/{$allPlayoffMatches} trận hoàn thành)",
            'progress' => $playoffProgress,
            'next_actions' => $this->getPlayoffNextActions($playoffRounds),
            'rounds' => $playoffRounds->mapWithKeys(function (TournamentRound $round) {
                $level = $this->resolvePlayoffLevel($round) ?? ('round_'.$round->id);

                return [$level => $round->name];
            })->toArray(),
        ];
    }

    /**
     * Get next recommended actions for playoff stage
     */
    private function getPlayoffNextActions($playoffRounds): array
    {
        $actions = [];
        $roundLevels = $playoffRounds
            ->map(fn (TournamentRound $round) => $this->resolvePlayoffLevel($round))
            ->filter()
            ->toArray();

        // Check for incomplete matches in latest round
        $latestRound = $playoffRounds->sortByDesc('created_at')->first();
        if ($latestRound && $latestRound->matches->where('status', '!=', 'Hoàn thành')->isNotEmpty()) {
            $actions[] = "Hoàn thành trận đấu {$latestRound->name}";
        } else {
            // Suggest next round creation
            if (! in_array('semi_final', $roundLevels) && in_array('quarter_final', $roundLevels)) {
                $actions[] = 'Tạo vòng bán kết';
            } elseif (! in_array('final', $roundLevels) && in_array('semi_final', $roundLevels)) {
                $actions[] = 'Tạo trận chung kết';
                if (! in_array('third_place', $roundLevels)) {
                    $actions[] = 'Tạo trận tranh hạng ba';
                }
            }
        }

        return $actions;
    }

    private function decodeMetaData($meta): array
    {
        if (is_array($meta)) {
            return $meta;
        }

        if (is_string($meta)) {
            $decoded = json_decode($meta, true);

            return is_array($decoded) ? $decoded : [];
        }

        return [];
    }

    private function resolvePlayoffLevel(TournamentRound $round): ?string
    {
        if (! empty($round->playoff_level)) {
            return $round->playoff_level;
        }

        $meta = $this->decodeMetaData($round->meta_data);
        if (! empty($meta['playoff_level'])) {
            return $meta['playoff_level'];
        }

        if (isset($meta['bracket_size'])) {
            $bracketSize = (int) $meta['bracket_size'];
            $totalTeams = (int) ($meta['total_teams'] ?? $bracketSize);

            return match ($bracketSize) {
                2 => 'final',
                4 => 'semi_final',
                8 => 'quarter_final',
                16 => 'round_of_16',
                32 => 'round_of_32',
                default => $totalTeams > 0 ? 'round_of_'.$totalTeams : null,
            };
        }

        $name = mb_strtolower($round->name ?? '', 'UTF-8');
        if (str_contains($name, 'chung kết')) {
            return 'final';
        }
        if (str_contains($name, 'bán kết')) {
            return 'semi_final';
        }
        if (str_contains($name, 'tứ kết')) {
            return 'quarter_final';
        }
        if (str_contains($name, 'hạng ba')) {
            return 'third_place';
        }

        if (preg_match('/(\d+)\s*đội/', $name, $matches)) {
            $teams = (int) ($matches[1] ?? 0);

            return $teams > 0 ? 'round_of_'.$teams : null;
        }

        return null;
    }

    /**
     * Enhanced tournament statistics with insights
     */
    public function getTournamentStats(Tournament $tournament): array
    {
        $stats = [
            'basic' => $this->getBasicStats($tournament),
            'progress' => $this->getProgressStats($tournament),
            'performance' => $this->getPerformanceStats($tournament),
            'insights' => $this->generateInsights($tournament),
        ];

        return $stats;
    }

    private function getBasicStats(Tournament $tournament): array
    {
        return [
            'total_registrations' => $tournament->registrations()->count(),
            'approved_registrations' => $tournament->registrations()->where('status', 'approved')->count(),
            'pending_registrations' => $tournament->registrations()->where('status', 'pending')->count(),
            'rejected_registrations' => $tournament->registrations()->where('status', 'rejected')->count(),
            'total_groups' => $tournament->groups()->count(),
            'total_pairs' => $tournament->groups()->with('pairs')->get()->sum(fn ($g) => $g->pairs->count()),
            'total_matches' => $tournament->matches()->count(),
        ];
    }

    private function getProgressStats(Tournament $tournament): array
    {
        $completed = $tournament->matches()->where('status', 'Hoàn thành')->count();
        $ongoing = $tournament->matches()->where('status', 'Đang diễn ra')->count();
        $pending = $tournament->matches()->where('status', 'Chưa bắt đầu')->count();
        $total = $completed + $ongoing + $pending;

        return [
            'completed_matches' => $completed,
            'ongoing_matches' => $ongoing,
            'pending_matches' => $pending,
            'completion_percentage' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
        ];
    }

    private function getPerformanceStats(Tournament $tournament): array
    {
        $matches = $tournament->matches()->whereNotNull('score')->get();

        if ($matches->isEmpty()) {
            return ['average_score' => 0, 'total_points' => 0, 'matches_with_scores' => 0];
        }

        $totalPoints = 0;
        $matchCount = 0;

        foreach ($matches as $match) {
            if (str_contains($match->score, '-')) {
                $scoreParts = explode('-', $match->score);
                if (count($scoreParts) === 2) {
                    [$score1, $score2] = array_map('intval', $scoreParts);
                    $totalPoints += $score1 + $score2;
                    $matchCount++;
                }
            }
        }

        return [
            'average_score' => $matchCount > 0 ? round($totalPoints / ($matchCount * 2), 1) : 0,
            'total_points' => $totalPoints,
            'matches_with_scores' => $matchCount,
        ];
    }

    private function generateInsights(Tournament $tournament): array
    {
        $insights = [];
        $phase = $this->getCurrentPhase($tournament);

        // Tournament duration insights
        if ($tournament->start_date && $tournament->end_date) {
            $duration = $tournament->start_date->diffInDays($tournament->end_date);
            if ($duration > 7) {
                $insights[] = 'Giải đấu kéo dài '.$duration.' ngày - khá lâu cho giải pickleball';
            }
        }

        // Participation insights
        $basic = $this->getBasicStats($tournament);
        if ($basic['approved_registrations'] > 32) {
            $insights[] = 'Giải đấu lớn với '.$basic['approved_registrations'].' cặp tham gia';
        }

        // Progress insights
        $progress = $this->getProgressStats($tournament);
        if ($progress['completion_percentage'] > 50 && $phase['phase'] === 'group_stage') {
            $insights[] = 'Vòng bảng sắp hoàn thành, chuẩn bị tạo playoffs';
        }

        return $insights;
    }

    /**
     * Get recommended actions for tournament based on current state
     */
    public function getRecommendedActions(Tournament $tournament): array
    {
        $phase = $this->getCurrentPhase($tournament);
        $validation = $this->validateTournamentState($tournament);

        $actions = [];

        // Add actions from current phase
        if (isset($phase['next_actions'])) {
            $actions = array_merge($actions, $phase['next_actions']);
        }

        // Add validation-based recommendations
        if (isset($validation['recommendations'])) {
            $actions = array_merge($actions, $validation['recommendations']);
        }

        // Add error resolution actions
        if (! empty($validation['errors'])) {
            array_unshift($actions, 'Khắc phục các lỗi hệ thống trước');
        }

        // Add warning resolution suggestions
        if (! empty($validation['warnings']) && empty($validation['errors'])) {
            $actions[] = 'Xem xét các cảnh báo hệ thống';
        }

        // Remove duplicates and return
        return array_unique($actions);
    }

    /**
     * Clear tournament state cache
     */
    public function clearStateCache(Tournament $tournament): void
    {
        Cache::forget("tournament_state_{$tournament->id}");
    }

    // Keep existing validation methods for backward compatibility
    public function canGenerateGroups(Tournament $tournament): array
    {
        return $this->validateTournamentState($tournament);
    }

    public function canGenerateGroupRounds(Tournament $tournament): array
    {
        $validation = $this->validateTournamentState($tournament);

        if (! $tournament->groups()->exists()) {
            $validation['errors'][] = 'Chưa có bảng đấu. Vui lòng tạo bảng đấu trước.';
            $validation['can_proceed'] = false;
        }

        return $validation;
    }

    public function canGeneratePlayoffs(Tournament $tournament): array
    {
        $validation = $this->validateTournamentState($tournament);
        $groupProgress = $this->analyzeGroupStageProgress($tournament);

        $completionPercentage = $groupProgress['completion_percentage'];
        $remainingMatches = $groupProgress['total_matches'] - $groupProgress['matches_completed'];

        if ($completionPercentage < 90) {
            $validation['errors'][] = "Cần hoàn thành ít nhất 90% vòng bảng để tạo playoffs. Hiện tại: {$completionPercentage}%. Còn {$remainingMatches} trận.";
            $validation['can_proceed'] = false;
        } elseif ($completionPercentage < 95) {
            $validation['warnings'][] = "Chỉ hoàn thành {$completionPercentage}% vòng bảng. Khuyến nghị hoàn thành thêm {$remainingMatches} trận để có kết quả chính xác hơn.";
            // Allow proceed but with warning
        } elseif ($completionPercentage < 100) {
            $validation['warnings'][] = "Còn {$remainingMatches} trận vòng bảng chưa hoàn thành. Playoffs sẽ dựa trên kết quả hiện tại.";
        }

        return $validation;
    }

    public function canGenerateNextPlayoffRound(Tournament $tournament, string $previousRoundName): array
    {
        $validation = $this->validateTournamentState($tournament);

        $previousRound = $tournament->rounds()
            ->where('name', 'like', '%'.$previousRoundName.'%')
            ->where('type', 'playoff')
            ->first();

        if (! $previousRound) {
            $validation['errors'][] = "Không tìm thấy vòng {$previousRoundName}";
            $validation['can_proceed'] = false;
        } else {
            $incompleteMatches = $tournament->matches()
                ->where('tournament_round_id', $previousRound->id)
                ->whereNotIn('status', ['Hoàn thành', 'Bye'])
                ->count();
            if ($incompleteMatches > 0) {
                $validation['errors'][] = "Còn {$incompleteMatches} trận trong vòng {$previousRoundName} chưa hoàn thành";
                $validation['can_proceed'] = false;
            }
        }

        return $validation;
    }
}