<?php

namespace App\Services;

use App\Models\Tournament;
use App\Models\TournamentGroup as Group;

class TournamentService
{
    /**
     * Calculate standings for a specific group.
     */
    public function calculateStandings(int $groupId): array
    {
        $group = Group::with('pairs.player1', 'pairs.player2', 'matches.pair1', 'matches.pair2')->find($groupId);

        if (! $group) {
            return [];
        }

        $standings = [];

        foreach ($group->pairs as $pair) {
            $standings[$pair->id] = [
                'pair' => $pair,
                'played' => 0,
                'wins' => 0,
                'losses' => 0,
                'points_for' => 0,
                'points_against' => 0,
                'points_diff' => 0,
            ];
        }

        foreach ($group->getAllMatches() as $match) {
            if (! $match->score || ! $match->pair1_id || ! $match->pair2_id) {
                continue;
            }

            $scores = explode('-', $match->score);
            if (count($scores) !== 2) {
                continue;
            }
            $score1 = (int) $scores[0];
            $score2 = (int) $scores[1];

            // Update stats for pair 1
            if (isset($standings[$match->pair1_id])) {
                $standings[$match->pair1_id]['played']++;
                $standings[$match->pair1_id]['points_for'] += $score1;
                $standings[$match->pair1_id]['points_against'] += $score2;
                if ($score1 > $score2) {
                    $standings[$match->pair1_id]['wins']++;
                } else {
                    $standings[$match->pair1_id]['losses']++;
                }
            }

            // Update stats for pair 2
            if (isset($standings[$match->pair2_id])) {
                $standings[$match->pair2_id]['played']++;
                $standings[$match->pair2_id]['points_for'] += $score2;
                $standings[$match->pair2_id]['points_against'] += $score1;
                if ($score2 > $score1) {
                    $standings[$match->pair2_id]['wins']++;
                } else {
                    $standings[$match->pair2_id]['losses']++;
                }
            }
        }

        foreach ($standings as &$standing) {
            $standing['points_diff'] = $standing['points_for'] - $standing['points_against'];
        }

        // Sort standings: 1. Wins, 2. Points Difference
        usort($standings, function ($a, $b) {
            if ($b['wins'] !== $a['wins']) {
                return $b['wins'] <=> $a['wins'];
            }

            return $b['points_diff'] <=> $a['points_diff'];
        });

        return $standings;
    }

    /**
     * Get all knockout rounds data for a tournament.
     */
    public function getKnockoutRounds(int $tournamentId): array
    {
        $rounds = Tournament::find($tournamentId)->rounds()
            ->where('type', 'playoff')
            ->whereIn('playoff_level', ['round_of_64', 'round_of_32', 'round_of_16', 'quarter_final', 'semi_final', 'final', 'third_place'])
            ->with([
                'matches.pair1.player1:id,name,avatar', 'matches.pair1.player2:id,name,avatar',
                'matches.pair2.player1:id,name,avatar', 'matches.pair2.player2:id,name,avatar',
            ])->get()->keyBy('playoff_level');

        $roundOf64 = $rounds->get('round_of_64') ?: null;
        $roundOf32 = $rounds->get('round_of_32') ?: null;
        $roundOf16 = $rounds->get('round_of_16') ?: null;
        $quarterFinals = $rounds->get('quarter_final') ?: null;
        $semiFinals = $rounds->get('semi_final') ?: null;
        $final = $rounds->get('final') ?: null;
        $thirdPlace = $rounds->get('third_place') ?: null;

        return [
            'roundOf64' => $roundOf64,
            'roundOf32' => $roundOf32,
            'roundOf16' => $roundOf16,
            'quarterFinals' => $quarterFinals,
            'semiFinals' => $semiFinals,
            'final' => $final,
            'thirdPlace' => $thirdPlace,
        ];
    }
}