<?php

namespace App\Services;

// Initialize ultra memory mode for tournament operations (namespaced helper)
try {
    if (class_exists(\App\Helpers\MemoryBooster::class)) {
        \App\Helpers\MemoryBooster::init('ultra');
    }
} catch (\Throwable $e) {
    // Ignore memory booster failures in testing environments
}

use App\Models\Registration;
use App\Models\Tournament;
use App\Models\TournamentGroup as Group;
use App\Models\TournamentRound;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TournamentGenerationService
{
    private const MAX_MEMORY_USAGE = 500 * 1024 * 1024; // 500MB

    private const BATCH_SIZE = 100;

    private const CACHE_TTL = 600; // 10 minutes

    protected $standingsService;

    protected $stateService;

    public function __construct(StandingsService $standingsService, TournamentStateService $stateService)
    {
        $this->standingsService = $standingsService;
        $this->stateService = $stateService;
    }

    /**
     * Improved playoff generation with proper bye handling
     */
    public function generatePlayoffsImproved(Tournament $tournament, int $teamsToAdvancePerGroup = 2): string
    {
        // Increase time limit for complex calculations
        $this->setTimeLimit(300);

        // Validation
        $stateCheck = $this->stateService->canGeneratePlayoffs($tournament);
        if (! $stateCheck['can_proceed']) {
            throw new \Exception(implode("\n", $stateCheck['errors']));
        }

        $advancingTeams = $this->getAdvancingTeams($tournament, $teamsToAdvancePerGroup);
        $totalAdvancing = $advancingTeams->count();

        if ($totalAdvancing < 2) {
            throw new \Exception('Không đủ đội để tạo vòng Play-off.');
        }

        // Calculate proper bracket size and byes
        $bracketInfo = $this->calculateOptimalBracket($totalAdvancing);

        // Seed teams properly
        $seededTeams = $this->seedTeams($advancingTeams);

        DB::transaction(function () use ($tournament, $bracketInfo, $seededTeams) {
            $this->clearPlayoffs($tournament);
            $this->createPlayoffRound($tournament, $bracketInfo, $seededTeams);
        });

        return $bracketInfo['round_name'];
    }

    /**
     * Distribute byes efficiently across bracket to minimize first round imbalance
     */
    private function distributeByes(array $teams, int $numberOfByes): array
    {
        $totalTeams = count($teams);

        // For small tournaments, distribute byes more strategically
        if ($totalTeams <= 16 && $numberOfByes > 4) {
            return $this->distributeByesSmallTournament($teams, $numberOfByes);
        }

        // Standard bye distribution for larger tournaments
        shuffle($teams); // Random distribution
        $teamsWithByes = array_slice($teams, 0, $numberOfByes);
        $teamsWithoutByes = array_slice($teams, $numberOfByes);

        return [
            'teams_with_byes' => $teamsWithByes,
            'teams_without_byes' => $teamsWithoutByes,
            'first_round_matchups' => $this->createFirstRoundMatchups($teamsWithoutByes),
        ];
    }

    /**
     * Strategic bye distribution for small tournaments to maintain competition balance
     */
    private function distributeByesSmallTournament(array $teams, int $numberOfByes): array
    {
        // Sort teams by ranking/seeding for strategic bye placement
        usort($teams, function ($a, $b) {
            return ($a['seed'] ?? 999) <=> ($b['seed'] ?? 999);
        });

        // Give byes to top seeds to maintain bracket integrity
        $teamsWithByes = array_slice($teams, 0, $numberOfByes);
        $teamsWithoutByes = array_slice($teams, $numberOfByes);

        return [
            'teams_with_byes' => $teamsWithByes,
            'teams_without_byes' => $teamsWithoutByes,
            'first_round_matchups' => $this->createBalancedFirstRoundMatchups($teamsWithoutByes),
            'strategic_distribution' => true,
        ];
    }

    /**
     * Create balanced first round matchups
     */
    private function createBalancedFirstRoundMatchups(array $teams): array
    {
        $matchups = [];
        $teamCount = count($teams);

        // Pair teams strategically: highest vs lowest remaining seed
        for ($i = 0; $i < $teamCount; $i += 2) {
            if (isset($teams[$i]) && isset($teams[$i + 1])) {
                $matchups[] = [
                    'team1' => $teams[$i],
                    'team2' => $teams[$i + 1],
                    'round' => 1,
                ];
            }
        }

        return $matchups;
    }

    /**
     * Standard first round matchup creation
     */
    private function createFirstRoundMatchups(array $teams): array
    {
        $matchups = [];
        $teamCount = count($teams);

        for ($i = 0; $i < $teamCount; $i += 2) {
            if (isset($teams[$i]) && isset($teams[$i + 1])) {
                $matchups[] = [
                    'team1' => $teams[$i],
                    'team2' => $teams[$i + 1],
                    'round' => 1,
                ];
            }
        }

        return $matchups;
    }

    private function calculateOptimalBracket(int $totalTeams): array
    {
        // Special optimization for smaller tournaments to minimize byes
        if ($totalTeams <= 16) {
            return $this->calculateOptimalSmallBracket($totalTeams);
        }

        // Standard power-of-2 logic for larger tournaments
        $bracketSize = 2;
        while ($bracketSize < $totalTeams) {
            $bracketSize *= 2;
        }

        $numberOfByes = $bracketSize - $totalTeams;

        // Calculate first round matches correctly
        // Teams that don't get byes need to play in first round
        $teamsInFirstRound = $totalTeams - $numberOfByes;
        $firstRoundMatches = $teamsInFirstRound / 2;

        // Determine round name based on bracket size
        $roundName = match ($bracketSize) {
            2 => 'Chung kết',
            4 => 'Bán kết',
            8 => 'Tứ kết',
            16 => 'Vòng 1/16',
            32 => 'Vòng 1/32',
            64 => 'Vòng 1/64',
            128 => 'Vòng 1/128',
            256 => 'Vòng 1/256',
            default => "Playoffs {$totalTeams} đội"
        };

        return [
            'bracket_size' => $bracketSize,
            'total_teams' => $totalTeams,
            'number_of_byes' => $numberOfByes,
            'first_round_matches' => $firstRoundMatches,
            'round_name' => $roundName,
            'teams_with_byes' => $numberOfByes,
            'teams_in_first_round' => $teamsInFirstRound,
        ];
    }

    /**
     * Optimized bracket calculation for tournaments <= 16 teams
     * Reduces unnecessary byes and creates more balanced competition
     */
    private function calculateOptimalSmallBracket(int $totalTeams): array
    {
        // Determine optimal bracket structure based on team count
        $optimizedStructure = match (true) {
            $totalTeams <= 2 => ['bracket_size' => 2, 'round_name' => 'Chung kết'],
            $totalTeams <= 4 => ['bracket_size' => 4, 'round_name' => 'Bán kết'],
            $totalTeams <= 8 => ['bracket_size' => 8, 'round_name' => 'Tứ kết'],
            // For 9-12 teams: Use 16-bracket but optimize bye distribution
            $totalTeams <= 12 => ['bracket_size' => 16, 'round_name' => 'Vòng 1/16'],
            // For 13-16 teams: Standard 16-bracket
            $totalTeams <= 16 => ['bracket_size' => 16, 'round_name' => 'Vòng 1/16'],
            default => ['bracket_size' => 32, 'round_name' => 'Vòng 1/32']
        };

        $bracketSize = $optimizedStructure['bracket_size'];
        $numberOfByes = $bracketSize - $totalTeams;
        $teamsInFirstRound = $totalTeams - $numberOfByes;
        $firstRoundMatches = $teamsInFirstRound / 2;

        return [
            'bracket_size' => $bracketSize,
            'total_teams' => $totalTeams,
            'number_of_byes' => $numberOfByes,
            'first_round_matches' => $firstRoundMatches,
            'round_name' => $optimizedStructure['round_name'],
            'teams_with_byes' => $numberOfByes,
            'teams_in_first_round' => $teamsInFirstRound,
            'optimized' => true, // Flag to indicate this is optimized for small tournament
        ];
    }

    /**
     * Improved seeding with proper bye distribution
     */
    private function seedTeams(Collection $advancingTeams): Collection
    {
        // Sort teams: Group winners first, then runners-up, etc.
        return $advancingTeams->sortBy(function ($team) {
            return $team['rank'] * 1000 + ord($team['group_name']);
        })->values();
    }

    private function determinePlayoffLevelFromBracket(array $bracketInfo): string
    {
        $totalTeams = $bracketInfo['total_teams'] ?? 0;

        return match ($bracketInfo['bracket_size'] ?? null) {
            2 => 'final',
            4 => 'semi_final',
            8 => 'quarter_final',
            16 => 'round_of_16',
            32 => 'round_of_32',
            default => $totalTeams > 0 ? 'round_of_'.$totalTeams : 'playoff_round',
        };
    }

    /**
     * Create playoff round with proper bye handling
     */
    private function createPlayoffRound(Tournament $tournament, array $bracketInfo, Collection $seededTeams): void
    {
        $playoffLevel = $this->determinePlayoffLevelFromBracket($bracketInfo);
        $round = TournamentRound::create([
            'tournament_id' => $tournament->id,
            'name' => $bracketInfo['round_name'],
            'type' => 'playoff',
            'playoff_level' => $playoffLevel,
            'meta_data' => json_encode(array_merge($bracketInfo, ['playoff_level' => $playoffLevel])), // Store bracket info for later use
        ]);

        $matches = $this->generateBracketMatches($bracketInfo, $seededTeams);

        foreach ($matches as $matchData) {
            if (isset($matchData['bye_team_id'])) {
                // Create a special bye record instead of a match
                $this->createByeRecord($round, $matchData['bye_team_id']);
            } else {
                $round->matches()->create([
                    'tournament_id' => $tournament->id,
                    'pair1_id' => $matchData['pair1_id'],
                    'pair2_id' => $matchData['pair2_id'],
                    'round_number' => 1,
                    'bracket_position' => $matchData['bracket_position'] ?? null,
                ]);
            }
        }
    }

    /**
     * Generate bracket matches with proper bye distribution
     */
    private function generateBracketMatches(array $bracketInfo, Collection $seededTeams): array
    {
        $matches = [];
        $bracketSize = $bracketInfo['bracket_size'];
        $totalTeams = $bracketInfo['total_teams'];
        $numberOfByes = $bracketInfo['number_of_byes'];
        $teamsInFirstRound = $bracketInfo['teams_in_first_round'];

        // For proper bracket seeding, top seeds get byes
        // Remaining teams are paired for first round matches

        if ($numberOfByes > 0) {
            // Top seeds get byes (automatically advance to next round)
            $byeTeams = $seededTeams->take($numberOfByes);
            $playingTeams = $seededTeams->skip($numberOfByes);

            // Create bye records for top seeds
            foreach ($byeTeams as $index => $team) {
                $matches[] = [
                    'bye_team_id' => $team['pair_id'],
                    'bracket_position' => $index + 1,
                ];
            }

            // Create matches for remaining teams
            $playingTeamsArray = $playingTeams->values()->toArray();
            for ($i = 0; $i < count($playingTeamsArray); $i += 2) {
                if (isset($playingTeamsArray[$i + 1])) {
                    $matches[] = [
                        'pair1_id' => $playingTeamsArray[$i]['pair_id'],
                        'pair2_id' => $playingTeamsArray[$i + 1]['pair_id'],
                        'bracket_position' => $numberOfByes + ($i / 2) + 1,
                    ];
                }
            }
        } else {
            // No byes - all teams play in first round
            $teamsArray = $seededTeams->toArray();
            for ($i = 0; $i < count($teamsArray); $i += 2) {
                if (isset($teamsArray[$i + 1])) {
                    $matches[] = [
                        'pair1_id' => $teamsArray[$i]['pair_id'],
                        'pair2_id' => $teamsArray[$i + 1]['pair_id'],
                        'bracket_position' => ($i / 2) + 1,
                    ];
                }
            }
        }

        return $matches;
    }

    /**
     * Calculate proper bracket position for seeded team
     */
    /**
     * Create bye record for tracking
     */
    private function createByeRecord(TournamentRound $round, int $pairId): void
    {
        // Create a special match record to track bye
        $round->matches()->create([
            'tournament_id' => $round->tournament_id,
            'pair1_id' => $pairId,
            'pair2_id' => null,
            'round_number' => 1,
            'status' => 'Bye',
            'score' => 'BYE',
            'result' => 'bye',
        ]);
    }

    /**
     * Get advancing teams with ultra-optimized memory usage
     */
    private function getAdvancingTeams(Tournament $tournament, int $teamsToAdvancePerGroup): Collection
    {
        // Emergency memory management
        ini_set('memory_limit', '1G');
        gc_collect_cycles();

        // Use cache key based on tournament and last match update
        $cacheKey = "tournament_standings_{$tournament->id}_{$teamsToAdvancePerGroup}";
        $lastMatchUpdate = $tournament->matches()->max('updated_at');

        return cache()->remember($cacheKey.'_'.md5($lastMatchUpdate ?? ''), 300, function () use ($tournament, $teamsToAdvancePerGroup) {
            $advancingTeams = collect();

            // Process groups one by one to minimize memory usage
            $groupIds = $tournament->groups()->pluck('id', 'name');

            foreach ($groupIds as $groupName => $groupId) {
                try {
                    // Load group with minimal data
                    $group = $tournament->groups()
                        ->where('id', $groupId)
                        ->with(['pairs' => function ($query) {
                            $query->select('id', 'group_id', 'player1_id', 'player2_id');
                        }])
                        ->first(['id', 'name', 'tournament_id']);

                    if (! $group || $group->pairs->isEmpty()) {
                        continue;
                    }

                    // Get only matches for this group to save memory
                    $groupMatches = DB::table('matches as m')
                        ->join('tournament_rounds as tr', 'm.tournament_round_id', '=', 'tr.id')
                        ->whereIn('m.pair1_id', $group->pairs->pluck('id'))
                        ->whereIn('m.pair2_id', $group->pairs->pluck('id'))
                        ->where('tr.type', 'round_robin')
                        ->whereNotNull('m.score')
                        ->where('m.score', '!=', '')
                        ->select('m.pair1_id', 'm.pair2_id', 'm.score')
                        ->get();

                    // Calculate standings with minimal memory usage
                    $standings = $this->calculateGroupStandingsMinimal($group->pairs, $groupMatches);

                    if (empty($standings)) {
                        \Log::warning("No standings calculated for group {$groupName} in tournament {$tournament->id}");

                        continue;
                    }

                    $topPairs = array_slice($standings, 0, $teamsToAdvancePerGroup);
                    foreach ($topPairs as $rank => $standing) {
                        $advancingTeams->push([
                            'pair_id' => $standing['pair_id'],
                            'group_name' => $groupName,
                            'rank' => $rank + 1,
                            'wins' => $standing['wins'] ?? 0,
                            'points_diff' => $standing['points_diff'] ?? 0,
                            'points_for' => $standing['points_for'] ?? 0,
                        ]);
                    }

                    // Clear variables to free memory
                    unset($group, $groupMatches, $standings);
                    gc_collect_cycles();

                } catch (\Exception $e) {
                    \Log::error("Error calculating standings for group {$groupName}: ".$e->getMessage());

                    continue;
                }
            }

            if ($advancingTeams->isEmpty()) {
                throw new \Exception('Không thể tính toán đội thăng hạng. Vui lòng kiểm tra kết quả các trận đấu.');
            }

            return $advancingTeams;
        });
    }

    /**
     * Calculate standings with minimal memory footprint
     */
    private function calculateGroupStandingsMinimal($pairs, $matches): array
    {
        $standings = [];

        // Initialize standings
        foreach ($pairs as $pair) {
            $standings[$pair->id] = [
                'pair_id' => $pair->id,
                'played' => 0,
                'wins' => 0,
                'losses' => 0,
                'points_for' => 0,
                'points_against' => 0,
                'points_diff' => 0,
            ];
        }

        // Process matches
        foreach ($matches as $match) {
            if (empty($match->score) || ! str_contains($match->score, '-')) {
                continue;
            }

            $scoreParts = explode('-', $match->score);
            if (count($scoreParts) !== 2) {
                continue;
            }
            [$score1, $score2] = array_map('intval', $scoreParts);

            if (! isset($standings[$match->pair1_id]) || ! isset($standings[$match->pair2_id])) {
                continue;
            }

            // Update pair 1
            $standings[$match->pair1_id]['played']++;
            $standings[$match->pair1_id]['points_for'] += $score1;
            $standings[$match->pair1_id]['points_against'] += $score2;

            // Update pair 2
            $standings[$match->pair2_id]['played']++;
            $standings[$match->pair2_id]['points_for'] += $score2;
            $standings[$match->pair2_id]['points_against'] += $score1;

            // Determine winner
            if ($score1 > $score2) {
                $standings[$match->pair1_id]['wins']++;
                $standings[$match->pair2_id]['losses']++;
            } elseif ($score2 > $score1) {
                $standings[$match->pair2_id]['wins']++;
                $standings[$match->pair1_id]['losses']++;
            }
        }

        // Calculate point differential and sort
        foreach ($standings as &$standing) {
            $standing['points_diff'] = $standing['points_for'] - $standing['points_against'];
        }

        // Sort by wins, then by point differential
        uasort($standings, function ($a, $b) {
            if ($a['wins'] !== $b['wins']) {
                return $b['wins'] <=> $a['wins'];
            }

            return $b['points_diff'] <=> $a['points_diff'];
        });

        return array_values($standings);
    }

    /**
     * Improved next round generation
     */
    public function generateNextPlayoffRoundImproved(Tournament $tournament, string $previousRoundName): string
    {
        $previousRound = $tournament->rounds()
            ->where('name', 'like', '%'.$previousRoundName.'%')
            ->where('type', 'playoff')
            ->with('matches')
            ->first();

        if (! $previousRound) {
            throw new \Exception("Không tìm thấy vòng {$previousRoundName}");
        }

        // Check if all matches in previous round are completed
        $incompleteMatches = $previousRound->matches->filter(function ($match) {
            return $match->status !== 'Hoàn thành' && $match->status !== 'Bye';
        });

        if ($incompleteMatches->isNotEmpty()) {
            throw new \Exception("Vẫn còn {$incompleteMatches->count()} trận đấu chưa hoàn thành trong vòng {$previousRoundName}");
        }

        $winners = $this->getWinnersFromRound($previousRound);
        $totalWinners = count($winners);

        if ($totalWinners < 2) {
            throw new \Exception('Không đủ đội thắng để tạo vòng tiếp theo');
        }

        $nextRoundInfo = $this->determineNextRound($totalWinners, $previousRoundName);

        DB::transaction(function () use ($tournament, $nextRoundInfo, $winners, $previousRound) {
            $this->createNextRound($tournament, $nextRoundInfo, $winners, $previousRound);
        });

        return 'Đã tạo thành công '.implode(' và ', $nextRoundInfo['names']).'!';
    }

    /**
     * Get winners from previous round including bye handling
     */
    private function getWinnersFromRound(TournamentRound $round): array
    {
        $winners = [];

        foreach ($round->matches as $match) {
            if ($match->status === 'Bye') {
                // Team automatically advances due to bye
                $winners[] = $match->pair1_id;
            } elseif ($match->status === 'Hoàn thành' && ! empty($match->score) && str_contains($match->score, '-')) {
                $scoreParts = explode('-', $match->score);
                if (count($scoreParts) === 2) {
                    [$score1, $score2] = array_map('intval', $scoreParts);
                    $winners[] = ($score1 > $score2) ? $match->pair1_id : $match->pair2_id;
                }
            }
        }

        return $winners;
    }

    /**
     * Determine next round configuration
     */
    private function determineNextRound(int $winnersCount, string $previousRoundName): array
    {
        return match ($winnersCount) {
            2 => ['names' => ['Chung kết'], 'playoff_level' => 'final'],
            4 => ['names' => ['Bán kết'], 'playoff_level' => 'semi_final'],
            8 => ['names' => ['Tứ kết'], 'playoff_level' => 'quarter_final'],
            16 => ['names' => ['Vòng 1/16'], 'playoff_level' => 'round_of_16'],
            32 => ['names' => ['Vòng 1/32'], 'playoff_level' => 'round_of_32'],
            64 => ['names' => ['Vòng 1/64'], 'playoff_level' => 'round_of_64'],
            128 => ['names' => ['Vòng 1/128'], 'playoff_level' => 'round_of_128'],
            default => ['names' => ["Vòng {$winnersCount} đội"], 'playoff_level' => "round_of_{$winnersCount}"]
        };
    }

    /**
     * Create next round matches
     */
    private function createNextRound(Tournament $tournament, array $roundInfo, array $winners, TournamentRound $previousRound): void
    {
        // Clean up any existing rounds
        $tournament->rounds()->whereIn('name', $roundInfo['names'])->delete();

        $mainRound = TournamentRound::create([
            'tournament_id' => $tournament->id,
            'name' => $roundInfo['names'][0],
            'type' => 'playoff',
            'playoff_level' => $roundInfo['playoff_level'] ?? null,
            'parent_round_id' => $previousRound->id,
            'meta_data' => [
                'playoff_level' => $roundInfo['playoff_level'] ?? null,
                'source_round_id' => $previousRound->id,
            ],
        ]);

        // Create matches
        for ($i = 0; $i < count($winners); $i += 2) {
            if (isset($winners[$i + 1])) {
                $mainRound->matches()->create([
                    'tournament_id' => $tournament->id,
                    'pair1_id' => $winners[$i],
                    'pair2_id' => $winners[$i + 1],
                    'round_number' => 1,
                ]);
            } else {
                // Odd number of winners - give bye to last winner
                $mainRound->matches()->create([
                    'tournament_id' => $tournament->id,
                    'pair1_id' => $winners[$i],
                    'pair2_id' => null,
                    'round_number' => 1,
                    'status' => 'Bye',
                    'score' => 'BYE',
                    'result' => 'bye',
                ]);
            }
        }

        // Handle third place if this creates a final
        if ($roundInfo['playoff_level'] === 'final' && count($winners) === 2) {
            $this->createThirdPlaceMatch($tournament, $previousRound);
        }
    }

    /**
     * Create third place match from semi-final losers
     */
    private function createThirdPlaceMatch(Tournament $tournament, TournamentRound $semiFinalRound): void
    {
        $losers = [];

        foreach ($semiFinalRound->matches as $match) {
            if (! empty($match->score) && str_contains($match->score, '-')) {
                $scoreParts = explode('-', $match->score);
                if (count($scoreParts) === 2) {
                    [$score1, $score2] = array_map('intval', $scoreParts);
                    $losers[] = ($score1 < $score2) ? $match->pair1_id : $match->pair2_id;
                }
            }
        }

        if (count($losers) >= 2) {
            $thirdPlaceRound = TournamentRound::create([
                'tournament_id' => $tournament->id,
                'name' => 'Tranh Hạng Ba',
                'type' => 'playoff',
                'playoff_level' => 'third_place',
                'parent_round_id' => $semiFinalRound->id,
                'meta_data' => [
                    'playoff_level' => 'third_place',
                    'source_round_id' => $semiFinalRound->id,
                ],
            ]);

            $thirdPlaceRound->matches()->create([
                'tournament_id' => $tournament->id,
                'pair1_id' => $losers[0],
                'pair2_id' => $losers[1],
                'round_number' => 1,
            ]);
        }
    }

    /**
     * Generate tournament groups with automatic seeding - Performance optimized
     */
    public function generateGroups(Tournament $tournament, int $numberOfGroups): void
    {
        // Increase memory limit for large tournaments
        ini_set('memory_limit', '512M');

        // Increase time limit for large tournaments
        $this->setTimeLimit(300);

        // Validation
        $stateCheck = $this->stateService->canGenerateGroups($tournament);
        if (! $stateCheck['can_proceed']) {
            throw new \Exception(implode("\n", $stateCheck['errors']));
        }

        // Get approved registrations count first to avoid loading unnecessary data
        $registrationCount = $tournament->registrations()
            ->where('status', 'approved')
            ->count();

        if ($registrationCount < 4) {
            throw new \Exception('Cần ít nhất 4 cặp đấu được duyệt để tạo bảng.');
        }

        if ($numberOfGroups > $registrationCount) {
            throw new \Exception('Số bảng không thể lớn hơn số cặp đấu.');
        }

        // Performance monitoring
        $startTime = microtime(true);
        Log::info("Starting group generation for tournament {$tournament->id} with {$registrationCount} pairs into {$numberOfGroups} groups");

        // Load registrations only after validation
        $approvedRegistrations = $tournament->registrations()
            ->where('status', 'approved')
            ->select('id', 'tournament_id', 'player1_id', 'player2_id') // Only necessary fields
            ->get();

        // Use shorter transaction with optimized operations
        DB::transaction(function () use ($tournament, $numberOfGroups, $approvedRegistrations) {
            // Clear existing groups efficiently
            $this->clearGroupsOptimized($tournament);

            // Clear tournament-specific cache keys (database cache doesn't support tagging)
            cache()->forget("tournament_standings_{$tournament->id}");
            cache()->forget("tournament_schedule_{$tournament->id}");
            cache()->forget("tournament_groups_{$tournament->id}");

            // Create groups
            $groups = $this->createTournamentGroups($tournament, $numberOfGroups);

            // Distribute pairs using seeding logic
            $this->distributePairsToGroupsOptimized($approvedRegistrations, $groups);
        }, 3); // Retry up to 3 times on deadlock

        $endTime = microtime(true);
        $executionTime = round(($endTime - $startTime) * 1000, 2);
        Log::info("Group generation completed in {$executionTime}ms for tournament {$tournament->id}");
    }

    /**
     * Create tournament group records
     */
    private function createTournamentGroups(Tournament $tournament, int $numberOfGroups): Collection
    {
        $groups = collect();

        for ($i = 0; $i < $numberOfGroups; $i++) {
            $groupName = chr(65 + $i); // A, B, C, D...

            $group = $tournament->groups()->create([
                'name' => $groupName,
                'tournament_id' => $tournament->id,
            ]);

            $groups->push($group);
        }

        return $groups;
    }

    /**
     * Distribute pairs to groups using snake seeding (1-2-3-4-4-3-2-1)
     */
    private function distributePairsToGroups(Collection $registrations, Collection $groups): void
    {
        // Sort pairs by skill level (if available) or randomly
        $sortedPairs = $this->seedPairsBySkill($registrations);

        $numberOfGroups = $groups->count();
        $pairIndex = 0;

        // Snake distribution: 1-2-3-4-4-3-2-1-1-2...
        while ($pairIndex < $sortedPairs->count()) {
            // Forward pass: 0, 1, 2, 3...
            for ($groupIndex = 0; $groupIndex < $numberOfGroups && $pairIndex < $sortedPairs->count(); $groupIndex++) {
                $this->assignPairToGroup($sortedPairs[$pairIndex], $groups[$groupIndex]);
                $pairIndex++;
            }

            // Backward pass: 3, 2, 1, 0...
            for ($groupIndex = $numberOfGroups - 1; $groupIndex >= 0 && $pairIndex < $sortedPairs->count(); $groupIndex--) {
                $this->assignPairToGroup($sortedPairs[$pairIndex], $groups[$groupIndex]);
                $pairIndex++;
            }
        }
    }

    /**
     * Seed pairs by skill level or random if no skill data
     */
    private function seedPairsBySkill(Collection $registrations): Collection
    {
        return $registrations->shuffle(); // For now, random distribution

        // TODO: Implement skill-based seeding when player ratings are available
        // return $registrations->sortByDesc(function ($registration) {
        //     return ($registration->player1->skill_level ?? 0) + ($registration->player2->skill_level ?? 0);
        // });
    }

    /**
     * Assign a registration pair to a group
     */
    private function assignPairToGroup($registration, $group): void
    {
        $group->pairs()->create([
            'group_id' => $group->id,
            'player1_id' => $registration->player1_id,
            'player2_id' => $registration->player2_id,
        ]);
    }

    /**
     * Generate round-robin matches for all groups - Performance optimized
     */
    public function generateGroupRounds(Tournament $tournament): void
    {
        // Increase memory limit for large tournaments
        ini_set('memory_limit', '512M');

        // Increase time limit for large tournaments
        $this->setTimeLimit(300);

        // Validation
        $stateCheck = $this->stateService->canGenerateGroupRounds($tournament);
        if (! $stateCheck['can_proceed']) {
            throw new \Exception(implode("\n", $stateCheck['errors']));
        }

        // Check groups count first for early validation
        $groupCount = $tournament->groups()->count();
        if ($groupCount === 0) {
            throw new \Exception('Chưa có bảng đấu nào. Vui lòng tạo bảng đấu trước.');
        }

        // Load groups with selective fields and chunked processing for large datasets
        $groups = $tournament->groups()
            ->with(['pairs' => function ($query) {
                $query->select('id', 'group_id', 'player1_id', 'player2_id');
            }])
            ->select('id', 'name', 'tournament_id')
            ->get();

        // Calculate estimated matches to prevent memory issues
        $totalPairs = $groups->sum(fn ($g) => $g->pairs->count());
        $estimatedMatches = $groups->sum(function ($group) {
            $pairCount = $group->pairs->count();

            return $pairCount > 1 ? ($pairCount * ($pairCount - 1)) / 2 : 0;
        });

        if ($estimatedMatches > 10000) {
            throw new \Exception('Quá nhiều trận đấu sẽ được tạo ('.$estimatedMatches.'). Vui lòng giảm số nhóm hoặc số cặp.');
        }

        // Performance monitoring
        $startTime = microtime(true);
        Log::info("Starting round generation for tournament {$tournament->id} with {$totalPairs} total pairs in {$groups->count()} groups, estimated {$estimatedMatches} matches");

        DB::transaction(function () use ($tournament, $groups) {
            // Clear existing group rounds efficiently
            $this->clearGroupRoundsOptimized($tournament);

            // Clear tournament-specific cache keys
            cache()->forget("tournament_standings_{$tournament->id}");
            cache()->forget("tournament_schedule_{$tournament->id}");
            cache()->forget("tournament_matches_{$tournament->id}");

            // Create round-robin round
            $round = $tournament->rounds()->create([
                'name' => 'Vòng bảng',
                'type' => 'round_robin',
            ]);

            // Generate matches for each group using batch insert with memory management
            $this->generateRoundRobinMatchesBatch($tournament, $round, $groups);
        }, 3); // Retry up to 3 times on deadlock

        // Sau khi tạo rounds, tính toán standings mới để đảm bảo luôn tạo mới
        $this->recalculateStandingsAfterRoundGeneration($tournament);

        $endTime = microtime(true);
        $executionTime = round(($endTime - $startTime) * 1000, 2);
        Log::info("Round generation completed in {$executionTime}ms for tournament {$tournament->id}");
    }

    /**
     * Tính toán lại standings sau khi tạo rounds để đảm bảo luôn tạo mới
     */
    private function recalculateStandingsAfterRoundGeneration(Tournament $tournament): void
    {
        try {
            // Load groups và rounds mới
            $groups = $tournament->groups()
                ->with(['pairs' => function ($query) {
                    $query->select('id', 'group_id', 'player1_id', 'player2_id');
                }])
                ->select('id', 'name', 'tournament_id')
                ->get();

            $rounds = $tournament->rounds()->where('type', 'round_robin')->get();

            if ($groups->isEmpty() || $rounds->isEmpty()) {
                return;
            }

            // Tính toán standings cho từng group và cache lại
            foreach ($groups as $group) {
                try {
                    $standings = $this->standingsService->calculateForGroup($group, $rounds);
                    
                    // Cache standings với key mới để đảm bảo tạo mới
                    $cacheKey = "standings_group_{$group->id}_".md5(now()->timestamp);
                    cache()->put($cacheKey, $standings, 300); // 5 phút
                    
                    Log::info("Recalculated standings for group {$group->name} with " . count($standings) . " teams");
                } catch (\Exception $e) {
                    Log::warning("Failed to recalculate standings for group {$group->name}: {$e->getMessage()}");
                }
            }

            // Clear lại cache tournament để đảm bảo consistency
            cache()->forget("tournament_standings_{$tournament->id}");
            
        } catch (\Exception $e) {
            Log::error("Failed to recalculate standings after round generation: {$e->getMessage()}");
        }
    }

    /**
     * Set execution time limit for long operations
     */
    private function setTimeLimit(int $seconds = 300): void
    {
        if (function_exists('set_time_limit')) {
            set_time_limit($seconds); // 5 minutes for large tournaments
        }
    }

    /**
     * Dynamic batch sizing based on system load and data size
     */
    private function getDynamicBatchSize(int $totalItems): int
    {
        $memoryUsage = memory_get_usage(true) / 1024 / 1024; // MB

        // Adjust batch size based on current memory usage
        if ($memoryUsage > 400) {
            return 25;
        }  // High memory usage - small batches
        if ($memoryUsage > 200) {
            return 50;
        }  // Medium memory usage

        // Adjust based on total items
        if ($totalItems > 10000) {
            return 50;
        }
        if ($totalItems > 5000) {
            return 100;
        }
        if ($totalItems > 1000) {
            return 150;
        }

        return 200; // Default
    }

    /**
     * Enhanced memory monitoring and cleanup
     */
    private function monitorAndCleanupMemory(string $operation, int $processed = 0): void
    {
        $memoryMB = round(memory_get_usage(true) / 1024 / 1024, 2);
        $peakMemoryMB = round(memory_get_peak_usage(true) / 1024 / 1024, 2);

        // Log every 500 operations or when memory is high
        if ($processed % 500 === 0 || $memoryMB > 300) {
            Log::info("{$operation}: Memory {$memoryMB}MB, Peak {$peakMemoryMB}MB, Processed {$processed}");
        }

        // Force cleanup if memory usage is high (limit to 500MB as requested)
        if ($memoryMB > 400) {
            for ($i = 0; $i < 5; $i++) {
                gc_collect_cycles();
            }

            $newMemoryMB = round(memory_get_usage(true) / 1024 / 1024, 2);
            Log::info("Memory cleanup: {$memoryMB}MB -> {$newMemoryMB}MB");

            // If still high after cleanup, throw warning
            if ($newMemoryMB > 450) {
                Log::warning("High memory usage after cleanup: {$newMemoryMB}MB");
            }
        }
    }

    /**
     * Optimized pair distribution using batch operations
     */
    private function distributePairsToGroupsOptimized(Collection $registrations, Collection $groups): void
    {
        $sortedPairs = $this->seedPairsBySkill($registrations);
        $numberOfGroups = $groups->count();
        $pairIndex = 0;

        // Prepare batch data for insert
        $pairData = [];
        $timestamp = now();

        // Snake distribution: 1-2-3-4-4-3-2-1-1-2...
        while ($pairIndex < $sortedPairs->count()) {
            // Forward pass: 0, 1, 2, 3...
            for ($groupIndex = 0; $groupIndex < $numberOfGroups && $pairIndex < $sortedPairs->count(); $groupIndex++) {
                $registration = $sortedPairs[$pairIndex];
                $group = $groups[$groupIndex];

                $pairData[] = [
                    'group_id' => $group->id,
                    'player1_id' => $registration->player1_id,
                    'player2_id' => $registration->player2_id,
                    'created_at' => $timestamp,
                    'updated_at' => $timestamp,
                ];
                $pairIndex++;
            }

            // Backward pass: 3, 2, 1, 0...
            for ($groupIndex = $numberOfGroups - 1; $groupIndex >= 0 && $pairIndex < $sortedPairs->count(); $groupIndex--) {
                $registration = $sortedPairs[$pairIndex];
                $group = $groups[$groupIndex];

                $pairData[] = [
                    'group_id' => $group->id,
                    'player1_id' => $registration->player1_id,
                    'player2_id' => $registration->player2_id,
                    'created_at' => $timestamp,
                    'updated_at' => $timestamp,
                ];
                $pairIndex++;
            }
        }

        // Batch insert all pairs at once (process in chunks for memory efficiency)
        if (! empty($pairData)) {
            $batchSize = $this->getDynamicBatchSize(count($pairData));
            $chunks = array_chunk($pairData, $batchSize);
            foreach ($chunks as $chunk) {
                DB::table('group_pairs')->insert($chunk);
            }

            // Free memory
            unset($pairData, $chunks);
        }
    }

    /**
     * Generate all round-robin matches using ultra-efficient memory management
     */
    private function generateRoundRobinMatchesBatch(Tournament $tournament, TournamentRound $round, Collection $groups): void
    {
        // Emergency memory settings
        ini_set('memory_limit', '1G');
        gc_collect_cycles();

        $timestamp = now();
        $totalMatches = 0;
        $processedMatches = 0;

        // Process one group at a time with immediate cleanup
        foreach ($groups as $groupIndex => $group) {
            try {
                // Get only the essential data for this group
                $pairIds = DB::table('group_pairs')
                    ->where('group_id', $group->id)
                    ->pluck('id')
                    ->toArray();

                $pairCount = count($pairIds);

                if ($pairCount < 2) {
                    \Log::warning("Group {$group->name} has only {$pairCount} pairs, skipping match generation");

                    continue;
                }

                // Calculate matches for this group and insert immediately
                $groupMatchCount = ($pairCount * ($pairCount - 1)) / 2;
                $totalMatches += $groupMatchCount;

                Log::info("Processing {$groupMatchCount} matches for group {$group->name}");

                // Generate and insert matches with dynamic batching
                $batchSize = $this->getDynamicBatchSize($groupMatchCount);
                $currentBatch = [];

                for ($i = 0; $i < $pairCount; $i++) {
                    for ($j = $i + 1; $j < $pairCount; $j++) {
                        $currentBatch[] = [
                            'tournament_id' => $tournament->id,
                            'tournament_round_id' => $round->id,
                            'pair1_id' => $pairIds[$i],
                            'pair2_id' => $pairIds[$j],
                            'round_number' => 1,
                            'status' => 'Chưa bắt đầu',
                            'created_at' => $timestamp,
                            'updated_at' => $timestamp,
                        ];

                        // Insert when batch is full
                        if (count($currentBatch) >= $batchSize) {
                            DB::table('matches')->insert($currentBatch);
                            $processedMatches += count($currentBatch);

                            // Clear and restart batch
                            $currentBatch = [];

                            // Enhanced memory monitoring and cleanup
                            $this->monitorAndCleanupMemory('batch_insert', $processedMatches);
                        }
                    }
                }

                // Insert remaining matches
                if (! empty($currentBatch)) {
                    DB::table('matches')->insert($currentBatch);
                    $processedMatches += count($currentBatch);
                    unset($currentBatch);
                }

                // Enhanced cleanup after each group
                unset($pairIds);
                $this->monitorAndCleanupMemory('group_processing', $groupIndex);

            } catch (\Exception $e) {
                Log::error("Error processing group {$group->name}: ".$e->getMessage());

                // Emergency cleanup on error
                gc_collect_cycles();

                continue;
            }
        }

        Log::info("Completed inserting {$totalMatches} matches for tournament {$tournament->id}");

        // Final cleanup
        for ($i = 0; $i < 5; $i++) {
            gc_collect_cycles();
        }
    }

    /**
     * Optimized group clearing using bulk operations
     */
    private function clearGroupsOptimized(Tournament $tournament): void
    {
        // Get group IDs first
        $groupIds = $tournament->groups()->pluck('id')->toArray();

        if (! empty($groupIds)) {
            // Bulk delete pairs first (foreign key constraint)
            DB::table('group_pairs')->whereIn('group_id', $groupIds)->delete();

            // Then delete groups
            DB::table('tournament_groups')->whereIn('id', $groupIds)->delete();
            
            // Touch tournament to invalidate cache
            $tournament->touch();
            
            // Clear specific cache keys using TournamentCacheService
            if (app()->bound(\App\Services\TournamentCacheService::class)) {
                $cacheService = app(\App\Services\TournamentCacheService::class);
                $cacheService->clearStandingsCache($tournament->id);
            }
        }
    }

    /**
     * Optimized group rounds clearing using bulk operations
     */
    private function clearGroupRoundsOptimized(Tournament $tournament): void
    {
        // Get round IDs for group rounds
        $roundIds = $tournament->rounds()
            ->where('type', 'round_robin')
            ->pluck('id')
            ->toArray();

        if (! empty($roundIds)) {
            // Bulk delete matches first (foreign key constraint)
            DB::table('matches')->whereIn('tournament_round_id', $roundIds)->delete();

            // Then delete rounds
            DB::table('tournament_rounds')->whereIn('id', $roundIds)->delete();
            
            // Touch tournament to invalidate cache
            $tournament->touch();
            
            // Clear specific cache keys using TournamentCacheService
            if (app()->bound(\App\Services\TournamentCacheService::class)) {
                $cacheService = app(\App\Services\TournamentCacheService::class);
                $cacheService->clearStandingsCache($tournament->id);
            }
        }
    }

    // Keep existing methods for compatibility but mark as deprecated

    /**
     * @deprecated Use clearGroupsOptimized instead
     */
    public function clearGroups(Tournament $tournament): void
    {
        $this->clearGroupsOptimized($tournament);
    }

    /**
     * @deprecated Use clearGroupRoundsOptimized instead
     */
    public function clearGroupRounds(Tournament $tournament): void
    {
        $this->clearGroupRoundsOptimized($tournament);
    }

    public function clearPlayoffs(Tournament $tournament): void
    {
        // Get round IDs for playoff rounds
        $roundIds = $tournament->rounds()
            ->where('type', 'playoff')
            ->pluck('id')
            ->toArray();

        if (! empty($roundIds)) {
            // Bulk delete matches first (foreign key constraint)
            DB::table('matches')->whereIn('tournament_round_id', $roundIds)->delete();

            // Then delete rounds
            DB::table('tournament_rounds')->whereIn('id', $roundIds)->delete();
        }
    }

    /**
     * Alias for generatePlayoffsImproved for backward compatibility
     */
    public function generatePlayoffs(Tournament $tournament, int $teamsToAdvancePerGroup = 2): string
    {
        return $this->generatePlayoffsImproved($tournament, $teamsToAdvancePerGroup);
    }

    /**
     * Alias for generateNextPlayoffRoundImproved for backward compatibility
     */
    public function generateNextPlayoffRound(Tournament $tournament, string $previousRoundName, array $nextRoundNames = [], int $expectedWinners = 0): string
    {
        return $this->generateNextPlayoffRoundImproved($tournament, $previousRoundName);
    }

    /**
     * Improved group generation with skill-based seeding
     */
    public function generateGroupsWithSeeding(Tournament $tournament, int $numberOfGroups): void
    {
        $this->validateTournamentForGroupGeneration($tournament, $numberOfGroups);

        // Get approved registrations with skill data
        $registrations = $this->getApprovedRegistrationsWithSkill($tournament);

        if ($registrations->count() < 4) {
            throw new \Exception('Cần ít nhất 4 cặp đấu được duyệt để tạo bảng.');
        }

        DB::transaction(function () use ($tournament, $numberOfGroups, $registrations) {
            // Clear existing groups efficiently
            $this->clearExistingGroups($tournament);

            // Create groups
            $groups = $this->createGroups($tournament, $numberOfGroups);

            // Seed teams into groups using snake draft method
            $this->seedTeamsIntoGroups($registrations, $groups);

            // Clear cache
            $this->clearTournamentCache($tournament);
        });

        Log::info("Groups generated successfully for tournament {$tournament->id}");
    }

    /**
     * Get approved registrations with skill data for seeding
     */
    private function getApprovedRegistrationsWithSkill(Tournament $tournament): Collection
    {
        return $tournament->registrations()
            ->where('status', 'approved')
            ->with([
                'player1:id,name,doubles_score,singles_score',
                'player2:id,name,doubles_score,singles_score',
            ])
            ->get()
            ->map(function ($registration) use ($tournament) {
                $skillSum = $tournament->type === 'doubles'
                    ? $registration->player1->doubles_score + ($registration->player2->doubles_score ?? 0)
                    : $registration->player1->singles_score;

                $registration->skill_sum = $skillSum;

                return $registration;
            })
            ->sortByDesc('skill_sum'); // Sort by skill level
    }

    /**
     * Snake draft seeding: 1-2-3-4-4-3-2-1-1-2-3-4...
     */
    private function seedTeamsIntoGroups(Collection $registrations, Collection $groups): void
    {
        $numberOfGroups = $groups->count();
        $groupIndex = 0;
        $direction = 1; // 1 for forward, -1 for backward

        foreach ($registrations as $registration) {
            $groups[$groupIndex]->pairs()->create([
                'player1_id' => $registration->player1_id,
                'player2_id' => $registration->player2_id,
            ]);

            // Snake pattern logic
            if ($direction === 1) {
                $groupIndex++;
                if ($groupIndex >= $numberOfGroups) {
                    $groupIndex = $numberOfGroups - 1;
                    $direction = -1;
                }
            } else {
                $groupIndex--;
                if ($groupIndex < 0) {
                    $groupIndex = 0;
                    $direction = 1;
                }
            }
        }
    }

    /**
     * Improved round-robin generation with memory optimization
     */
    public function generateRoundRobinOptimized(Tournament $tournament): void
    {
        $this->validateTournamentForRoundGeneration($tournament);

        $groups = $tournament->groups()->with('pairs:id,group_id,player1_id,player2_id')->get();

        DB::transaction(function () use ($tournament, $groups) {
            // Clear existing rounds
            $this->clearExistingRounds($tournament);

            // Create round-robin round
            $round = $tournament->rounds()->create([
                'name' => 'Vòng bảng',
                'type' => 'round_robin',
            ]);

            // Generate matches efficiently
            $this->generateMatchesOptimized($tournament, $round, $groups);

            // Clear cache once after creating all
            $this->clearTournamentCache($tournament);
        });

        // Sau khi tạo rounds, tính toán standings mới để đảm bảo luôn tạo mới
        $this->recalculateStandingsAfterRoundGeneration($tournament);

        Log::info("Round-robin matches generated for tournament {$tournament->id}");
    }

    /**
     * Generate matches with proper memory management
     */
    private function generateMatchesOptimized(Tournament $tournament, TournamentRound $round, Collection $groups): void
    {
        $totalMatches = 0;
        $batchData = [];

        foreach ($groups as $group) {
            $pairs = $group->pairs->toArray();
            $pairCount = count($pairs);

            if ($pairCount < 2) {
                continue;
            }

            // Generate all combinations for this group
            for ($i = 0; $i < $pairCount; $i++) {
                for ($j = $i + 1; $j < $pairCount; $j++) {
                    $batchData[] = [
                        'tournament_id' => $tournament->id,
                        'tournament_round_id' => $round->id,
                        'pair1_id' => $pairs[$i]['id'],
                        'pair2_id' => $pairs[$j]['id'],
                        'round_number' => 1,
                        'status' => 'Chưa bắt đầu',
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];

                    $totalMatches++;

                    // Batch insert when reaching limit
                    if (count($batchData) >= self::BATCH_SIZE) {
                        DB::table('matches')->insert($batchData);
                        $batchData = [];

                        // Memory check
                        $this->checkMemoryUsage();
                    }
                }
            }
        }

        // Insert remaining matches
        if (! empty($batchData)) {
            DB::table('matches')->insert($batchData);
        }

        Log::info("Generated {$totalMatches} matches for tournament {$tournament->id}");
    }

    // Backward-compat: generate matches for a single group collection (1 group)
    private function generateMatchesForGroup(Tournament $tournament, TournamentRound $round, Collection $groups): void
    {
        $this->generateMatchesOptimized($tournament, $round, $groups);
    }

    // Wrapper expected by some tests
    public function generateRounds(Tournament $tournament): array
    {
        $this->generateRoundRobinOptimized($tournament);
        return ['success' => true];
    }

    /**
     * Validation methods
     */
    private function validateTournamentForGroupGeneration(Tournament $tournament, int $numberOfGroups): void
    {
        if ($numberOfGroups < 1) {
            throw new \Exception('Số bảng đấu phải lớn hơn 0');
        }

        $approvedCount = $tournament->registrations()->where('status', 'approved')->count();
        if ($approvedCount < 4) {
            throw new \Exception('Cần ít nhất 4 cặp đấu được duyệt');
        }

        if ($numberOfGroups > $approvedCount) {
            throw new \Exception('Số bảng không thể lớn hơn số cặp đấu');
        }
    }

    private function validateTournamentForRoundGeneration(Tournament $tournament): void
    {
        if ($tournament->groups()->count() === 0) {
            throw new \Exception('Chưa có bảng đấu nào. Vui lòng tạo bảng đấu trước.');
        }
    }

    /**
     * Utility methods
     */
    private function createGroups(Tournament $tournament, int $numberOfGroups): Collection
    {
        $groups = collect();

        for ($i = 0; $i < $numberOfGroups; $i++) {
            $group = $tournament->groups()->create([
                'name' => chr(65 + $i), // A, B, C, D...
            ]);
            $groups->push($group);
        }

        return $groups;
    }

    private function clearExistingGroups(Tournament $tournament): void
    {
        $groupIds = $tournament->groups()->pluck('id');
        if ($groupIds->isNotEmpty()) {
            DB::table('group_pairs')->whereIn('group_id', $groupIds)->delete();
            DB::table('tournament_groups')->whereIn('id', $groupIds)->delete();
        }
    }

    private function clearExistingRounds(Tournament $tournament): void
    {
        $roundIds = $tournament->rounds()->where('type', 'round_robin')->pluck('id');
        if ($roundIds->isNotEmpty()) {
            DB::table('matches')->whereIn('tournament_round_id', $roundIds)->delete();
            DB::table('tournament_rounds')->whereIn('id', $roundIds)->delete();
        }
    }

    private function clearTournamentCache(Tournament $tournament): void
    {
        $keys = [
            "tournament_standings_{$tournament->id}",
            "tournament_schedule_{$tournament->id}",
            "tournament_groups_{$tournament->id}",
            "tournament_matches_{$tournament->id}",
        ];

        foreach ($keys as $key) {
            Cache::forget($key);
        }
    }

    private function checkMemoryUsage(): void
    {
        if (memory_get_usage(true) > self::MAX_MEMORY_USAGE) {
            gc_collect_cycles();

            if (memory_get_usage(true) > self::MAX_MEMORY_USAGE) {
                throw new \Exception('Memory limit exceeded during tournament generation');
            }
        }
    }

    /**
     * Analyze optimal bracket structure with enhanced efficiency analysis
     * This method provides comprehensive analysis for tournament organizers
     */
    public function analyzeOptimalBracketStructure(array $teams): array
    {
        $totalTeams = count($teams);

        if ($totalTeams < 2) {
            throw new \InvalidArgumentException('Cần ít nhất 2 đội để tạo playoffs');
        }

        // Get optimal bracket structure
        $bracketInfo = $this->calculateOptimalBracket($totalTeams);

        // Enhanced team distribution for small tournaments
        $distribution = $this->distributeByes($teams, $bracketInfo['number_of_byes']);

        return [
            'round_name' => $bracketInfo['round_name'],
            'bracket_size' => $bracketInfo['bracket_size'],
            'total_teams' => $totalTeams,
            'teams_with_byes' => $distribution['teams_with_byes'],
            'teams_without_byes' => $distribution['teams_without_byes'],
            'first_round_matches' => $distribution['first_round_matchups'],
            'number_of_byes' => $bracketInfo['number_of_byes'],
            'first_round_match_count' => count($distribution['first_round_matchups']),
            'optimized' => $bracketInfo['optimized'] ?? false,
            'strategic_distribution' => $distribution['strategic_distribution'] ?? false,
            'efficiency_data' => [
                'bye_ratio' => round(($bracketInfo['number_of_byes'] / $bracketInfo['bracket_size']) * 100, 1),
                'efficiency_rating' => $this->calculateEfficiencyRating($bracketInfo['number_of_byes'], $bracketInfo['bracket_size']),
                'optimal_for_size' => $totalTeams <= 16 ? 'Small tournament optimized' : 'Standard tournament logic',
            ],
        ];
    }

    /**
     * Calculate efficiency rating based on bye ratio
     */
    private function calculateEfficiencyRating(int $byes, int $bracketSize): string
    {
        $byeRatio = $byes / $bracketSize;

        if ($byeRatio <= 0.25) {
            return 'Excellent';
        } elseif ($byeRatio <= 0.50) {
            return 'Good';
        } else {
            return 'Acceptable';
        }
    }
}