<?php

namespace App\Services;

use App\Models\Tournament;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;

class TournamentCacheService
{
    private const CACHE_PREFIX = 'tournament';

    private const DEFAULT_TTL = 300; // 5 minutes

    private const LONG_TTL = 3600; // 1 hour

    private const SHORT_TTL = 60; // 1 minute

    /**
     * Get cached tournament data with automatic invalidation
     */
    public function getTournamentData(int $tournamentId, string $dataType): mixed
    {
        $cacheKey = $this->getCacheKey($tournamentId, $dataType);

        return Cache::remember($cacheKey, $this->getTTL($dataType), function () use ($tournamentId, $dataType) {
            return $this->fetchTournamentData($tournamentId, $dataType);
        });
    }

    /**
     * Cache tournament standings with dependency tracking
     */
    public function cacheStandings(int $tournamentId, array $standings): void
    {
        $cacheKey = $this->getCacheKey($tournamentId, 'standings');
        $dependencyKey = $this->getCacheKey($tournamentId, 'standings_deps');

        // Get current match update timestamps for dependency tracking
        $dependencies = $this->getStandingsDependencies($tournamentId);

        Cache::put($cacheKey, $standings, self::DEFAULT_TTL);
        Cache::put($dependencyKey, $dependencies, self::DEFAULT_TTL);

        // Tag for easy invalidation
        if (config('cache.default') === 'redis') {
            Cache::tags(["tournament_{$tournamentId}", 'standings'])->put($cacheKey, $standings, self::DEFAULT_TTL);
        }
    }

    /**
     * Check if standings cache is still valid
     */
    public function isStandingsCacheValid(int $tournamentId): bool
    {
        $dependencyKey = $this->getCacheKey($tournamentId, 'standings_deps');
        $cachedDeps = Cache::get($dependencyKey);

        if (! $cachedDeps) {
            return false;
        }

        $currentDeps = $this->getStandingsDependencies($tournamentId);

        return $cachedDeps === $currentDeps;
    }

    /**
     * Cache match data with smart expiration
     */
    public function cacheMatches(int $tournamentId, $matches, string $roundType = 'all'): void
    {
        $cacheKey = $this->getCacheKey($tournamentId, "matches_{$roundType}");

        // Shorter TTL for ongoing matches
        $ttl = $this->hasOngoingMatches($matches) ? self::SHORT_TTL : self::DEFAULT_TTL;

        Cache::put($cacheKey, $matches, $ttl);

        // Set up automatic invalidation based on match status changes
        $this->setupMatchInvalidation($tournamentId, $matches);
    }

    /**
     * Invalidate specific tournament cache
     */
    public function invalidateTournamentCache(int $tournamentId, array $cacheTypes = []): void
    {
        if (empty($cacheTypes)) {
            $cacheTypes = ['standings', 'matches', 'schedule', 'groups', 'overview'];
        }

        foreach ($cacheTypes as $type) {
            $pattern = $this->getCacheKey($tournamentId, $type.'*');
            $this->invalidateByPattern($pattern);
        }

        // Invalidate using tags if Redis
        if (config('cache.default') === 'redis') {
            Cache::tags(["tournament_{$tournamentId}"])->flush();
        }

        Log::info('Tournament cache invalidated', [
            'tournament_id' => $tournamentId,
            'cache_types' => $cacheTypes,
        ]);
    }

    /**
     * Smart cache warming for frequently accessed data
     */
    public function warmTournamentCache(Tournament $tournament): void
    {
        Log::info("Warming cache for tournament {$tournament->id}");

        // Warm standings cache
        if ($tournament->groups()->exists()) {
            foreach ($tournament->groups as $group) {
                $this->getTournamentData($tournament->id, "standings_group_{$group->id}");
            }
        }

        // Warm matches cache
        $this->getTournamentData($tournament->id, 'matches_round_robin');
        $this->getTournamentData($tournament->id, 'matches_playoff');

        // Warm schedule cache
        $this->getTournamentData($tournament->id, 'schedule');

        Log::info("Cache warming completed for tournament {$tournament->id}");
    }

    /**
     * Get cache statistics for monitoring
     */
    public function getCacheStats(int $tournamentId): array
    {
        $stats = [
            'tournament_id' => $tournamentId,
            'cache_hits' => 0,
            'cache_misses' => 0,
            'cached_items' => [],
            'memory_usage' => 0,
        ];

        $pattern = $this->getCacheKey($tournamentId, '*');
        $keys = $this->getKeysByPattern($pattern);

        foreach ($keys as $key) {
            $ttl = Cache::getRedis()->ttl($key);
            $size = strlen(serialize(Cache::get($key)));

            $stats['cached_items'][] = [
                'key' => $key,
                'ttl' => $ttl,
                'size' => $size,
            ];

            $stats['memory_usage'] += $size;
        }

        $stats['cached_items_count'] = count($stats['cached_items']);

        return $stats;
    }

    /**
     * Automatic cache cleanup for old tournaments
     */
    public function cleanupExpiredCache(): int
    {
        $cleanedCount = 0;

        // Find tournaments older than 30 days
        $expiredTournaments = Tournament::where('end_date', '<', now()->subDays(30))->pluck('id');

        foreach ($expiredTournaments as $tournamentId) {
            $this->invalidateTournamentCache($tournamentId);
            $cleanedCount++;
        }

        Log::info('Cache cleanup completed', ['cleaned_tournaments' => $cleanedCount]);

        return $cleanedCount;
    }

    /**
     * Preload cache for upcoming tournaments
     */
    public function preloadUpcomingTournaments(): void
    {
        $upcomingTournaments = Tournament::where('start_date', '>', now())
            ->where('start_date', '<=', now()->addDays(7))
            ->with(['groups', 'registrations'])
            ->get();

        foreach ($upcomingTournaments as $tournament) {
            $this->warmTournamentCache($tournament);
        }
    }

    /**
     * Real-time cache invalidation on data changes
     */
    public function handleDataChange(string $entityType, int $entityId, array $changes = []): void
    {
        switch ($entityType) {
            case 'match':
                $this->handleMatchChange($entityId, $changes);
                break;
            case 'registration':
                $this->handleRegistrationChange($entityId, $changes);
                break;
            case 'group_pair':
                $this->handleGroupPairChange($entityId, $changes);
                break;
        }
    }

    /**
     * Private helper methods
     */
    private function getCacheKey(int $tournamentId, string $dataType): string
    {
        return self::CACHE_PREFIX."_{$tournamentId}_{$dataType}";
    }

    private function getTTL(string $dataType): int
    {
        return match ($dataType) {
            'overview', 'groups' => self::LONG_TTL,
            'standings', 'schedule' => self::DEFAULT_TTL,
            'matches_ongoing' => self::SHORT_TTL,
            default => self::DEFAULT_TTL
        };
    }

    private function fetchTournamentData(int $tournamentId, string $dataType): mixed
    {
        $tournament = Tournament::find($tournamentId);
        if (! $tournament) {
            return null;
        }

        return match ($dataType) {
            'overview' => $this->fetchOverviewData($tournament),
            'standings' => $this->fetchStandingsData($tournament),
            'matches_round_robin' => $this->fetchMatchesData($tournament, 'round_robin'),
            'matches_playoff' => $this->fetchMatchesData($tournament, 'playoff'),
            'schedule' => $this->fetchScheduleData($tournament),
            'groups' => $this->fetchGroupsData($tournament),
            default => null
        };
    }

    private function getStandingsDependencies(int $tournamentId): array
    {
        return Tournament::find($tournamentId)
            ?->matches()
            ->whereHas('round', fn ($q) => $q->where('type', 'round_robin'))
            ->pluck('updated_at', 'id')
            ->toArray() ?? [];
    }

    private function hasOngoingMatches($matches): bool
    {
        if (is_array($matches) || is_object($matches)) {
            foreach ($matches as $match) {
                if (isset($match->status) && $match->status === 'Đang diễn ra') {
                    return true;
                }
            }
        }

        return false;
    }

    private function setupMatchInvalidation(int $tournamentId, $matches): void
    {
        // Set up Redis expiration callbacks or schedule cleanup jobs
        if (config('cache.default') === 'redis') {
            foreach ($matches as $match) {
                if (isset($match->id)) {
                    $key = "match_invalidation_{$match->id}";
                    Redis::setex($key, 3600, $tournamentId); // 1 hour
                }
            }
        }
    }

    private function invalidateByPattern(string $pattern): void
    {
        if (config('cache.default') === 'redis') {
            $keys = Redis::keys($pattern);
            if (! empty($keys)) {
                Redis::del($keys);
            }
        } else {
            // For file/database cache, we need to track keys manually
            $keys = $this->getTrackedKeys($pattern);
            foreach ($keys as $key) {
                Cache::forget($key);
            }
        }
    }

    private function getKeysByPattern(string $pattern): array
    {
        if (config('cache.default') === 'redis') {
            return Redis::keys($pattern) ?: [];
        }

        // Fallback for non-Redis caches
        return $this->getTrackedKeys($pattern);
    }

    private function getTrackedKeys(string $pattern): array
    {
        // This would require implementing key tracking for non-Redis caches
        // For now, return empty array
        return [];
    }

    private function handleMatchChange(int $matchId, array $changes): void
    {
        $match = \App\Models\MatchModel::find($matchId);
        if (! $match) {
            return;
        }

        $tournamentId = $match->tournament_id;

        // Invalidate standings if score changed
        if (isset($changes['score']) || isset($changes['status'])) {
            $this->invalidateTournamentCache($tournamentId, ['standings', 'matches']);
        }
    }

    private function handleRegistrationChange(int $registrationId, array $changes): void
    {
        $registration = \App\Models\Registration::find($registrationId);
        if (! $registration) {
            return;
        }

        $tournamentId = $registration->tournament_id;

        // Invalidate relevant caches
        if (isset($changes['status'])) {
            $this->invalidateTournamentCache($tournamentId, ['overview', 'groups']);
        }
    }

    private function handleGroupPairChange(int $pairId, array $changes): void
    {
        $pair = \App\Models\GroupPair::find($pairId);
        if (! $pair) {
            return;
        }

        $tournamentId = $pair->group->tournament_id;

        // Invalidate standings and matches
        $this->invalidateTournamentCache($tournamentId, ['standings', 'matches', 'groups']);
    }

    private function fetchOverviewData(Tournament $tournament): array
    {
        return [
            'tournament' => $tournament->only(['id', 'name', 'start_date', 'end_date', 'location', 'type']),
            'stats' => [
                'total_registrations' => $tournament->registrations()->count(),
                'approved_registrations' => $tournament->registrations()->where('status', 'approved')->count(),
                'total_matches' => $tournament->matches()->count(),
                'completed_matches' => $tournament->matches()->where('status', 'Hoàn thành')->count(),
            ],
        ];
    }

    private function fetchStandingsData(Tournament $tournament): array
    {
        // This would call the actual standings calculation service
        return [];
    }

    private function fetchMatchesData(Tournament $tournament, string $roundType): array
    {
        return $tournament->matches()
            ->whereHas('round', fn ($q) => $q->where('type', $roundType))
            ->with(['pair1.player1', 'pair1.player2', 'pair2.player1', 'pair2.player2'])
            ->get()
            ->toArray();
    }

    private function fetchScheduleData(Tournament $tournament): array
    {
        // Fetch schedule data with minimal queries
        return [];
    }

    private function fetchGroupsData(Tournament $tournament): array
    {
        return $tournament->groups()
            ->with(['pairs.player1', 'pairs.player2'])
            ->get()
            ->toArray();
    }

    /**
     * Clear all standings-related cache for a tournament
     */
    public function clearStandingsCache(int $tournamentId): void
    {
        try {
            // Clear tournament standings cache (used by Admin Controller)
            $tournament = \App\Models\Tournament::find($tournamentId);
            if ($tournament) {
                $cacheKey = "tournament_standings_{$tournamentId}_" . md5($tournament->updated_at);
                cache()->forget($cacheKey);
                
                // Clear dashboard cache too
                $dashboardKey = "tournament_dashboard_{$tournamentId}_" . md5($tournament->updated_at);
                cache()->forget($dashboardKey);
            }
            
            // Clear group standings cache (used by StandingsService)
            $groups = \App\Models\TournamentGroup::where('tournament_id', $tournamentId)->get();
            foreach ($groups as $group) {
                // Get latest match time for this group to build the right cache key
                $latestMatchTime = \DB::table('matches as m')
                    ->join('tournament_rounds as tr', 'm.tournament_round_id', '=', 'tr.id')
                    ->join('group_pairs as gp1', 'm.pair1_id', '=', 'gp1.id')
                    ->where('gp1.group_id', $group->id)
                    ->where('tr.type', 'round_robin')
                    ->max('m.updated_at');
                    
                $groupCacheKey = "standings_group_{$group->id}_" . md5($latestMatchTime ?? '');
                cache()->forget($groupCacheKey);
            }
            
            // Also clear any cached standings data
            $this->invalidateTournamentCache($tournamentId, ['standings']);
            
            \Log::info("Cleared standings cache for tournament {$tournamentId}");
            
        } catch (\Exception $e) {
            \Log::error("Error clearing standings cache for tournament {$tournamentId}: " . $e->getMessage());
        }
    }
}