<?php

namespace App\Services;

use App\Models\Tournament;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;

class TournamentAuthService
{
    private const SESSION_DURATION = 8; // hours
    private const SESSION_EXTEND_THRESHOLD = 30; // minutes
    
    /**
     * Authenticate user for tournament access
     */
    public function authenticate(Tournament $tournament, string $password, string $userType = 'auto'): array
    {
        // Validate inputs
        if (empty($password)) {
            return ['success' => false, 'message' => 'Vui lòng nhập mật khẩu'];
        }
        
        // Check passwords based on type
        $role = $this->validatePassword($tournament, $password, $userType);
        
        if (!$role) {
            Log::warning("Failed tournament authentication attempt", [
                'tournament_id' => $tournament->id,
                'user_type' => $userType,
                'ip' => request()->ip()
            ]);
            
            return ['success' => false, 'message' => 'Mật khẩu không chính xác'];
        }
        
        // Create session
        $this->createSession($tournament, $role);
        
        Log::info("Tournament authentication successful", [
            'tournament_id' => $tournament->id,
            'role' => $role,
            'user_id' => auth()->id()
        ]);
        
        return [
            'success' => true, 
            'role' => $role,
            'message' => "Đăng nhập thành công với vai trò " . $this->getRoleDisplayName($role)
        ];
    }
    
    /**
     * Validate password and determine role
     */
    private function validatePassword(Tournament $tournament, string $password, string $userType): ?string
    {
        // For organizer
        if (($userType === 'auto' || $userType === 'organizer') && 
            $tournament->password && $password === $tournament->password) {
            return 'organizer';
        }
        
        // For referee  
        if (($userType === 'auto' || $userType === 'referee') && 
            $tournament->referee_password && $password === $tournament->referee_password) {
            return 'referee';
        }
        
        return null;
    }
    
    /**
     * Create authenticated session
     */
    private function createSession(Tournament $tournament, string $role): void
    {
        $sessionData = [
            'tournament_id' => $tournament->id,
            'role' => $role,
            'authenticated_at' => now()->timestamp,
            'expires_at' => now()->addHours(self::SESSION_DURATION)->timestamp,
            'user_id' => auth()->id(),
            'ip' => request()->ip()
        ];
        
        // Store session data
        Session::put("tournament_auth_{$tournament->id}", $sessionData);
        Session::put("tournament_role_{$tournament->id}", $role);
        
        // Legacy compatibility
        Session::put("tournament_pw_{$tournament->id}", true);
    }
    
    /**
     * Check if user is authenticated for tournament
     */
    public function isAuthenticated(Tournament $tournament, ?string $requiredRole = null): bool
    {
        $sessionKey = "tournament_auth_{$tournament->id}";
        $sessionData = Session::get($sessionKey);
        
        if (!$sessionData) {
            return false;
        }
        
        // Check expiration
        if (now()->timestamp > $sessionData['expires_at']) {
            $this->logout($tournament);
            return false;
        }
        
        // Check role if specified
        if ($requiredRole && $sessionData['role'] !== $requiredRole) {
            return false;
        }
        
        // Auto-extend session if close to expiry
        $this->autoExtendSession($tournament, $sessionData);
        
        return true;
    }
    
    /**
     * Get user's role for tournament
     */
    public function getUserRole(Tournament $tournament): ?string
    {
        if (!$this->isAuthenticated($tournament)) {
            return null;
        }
        
        $sessionData = Session::get("tournament_auth_{$tournament->id}");
        return $sessionData['role'] ?? null;
    }
    
    /**
     * Check if user has permission for specific action
     */
    public function hasPermission(Tournament $tournament, string $permission): bool
    {
        $user = auth()->user();
        
        // Admin/Mod always have all permissions
        if ($user && ($user->hasRole('admin') || $user->hasRole('mod'))) {
            return true;
        }
        
        $role = $this->getUserRole($tournament);
        if (!$role) {
            return false;
        }
        
        return $this->checkRolePermission($role, $permission);
    }
    
    /**
     * Check role-specific permissions
     */
    private function checkRolePermission(string $role, string $permission): bool
    {
        $permissions = [
            'organizer' => [
                'view', 'manage_groups', 'manage_rounds', 'manage_matches', 
                'approve_registrations', 'generate_brackets', 'update_scores',
                'export_data', 'view_activity_logs', 'swap_players'
            ],
            'referee' => [
                'view', 'update_scores', 'export_data'
            ]
        ];
        
        return in_array($permission, $permissions[$role] ?? []);
    }
    
    /**
     * Auto-extend session before expiry
     */
    private function autoExtendSession(Tournament $tournament, array $sessionData): void
    {
        $timeLeft = $sessionData['expires_at'] - now()->timestamp;
        $extendThreshold = self::SESSION_EXTEND_THRESHOLD * 60; // Convert to seconds
        
        // Extend if less than threshold but more than 5 minutes left
        if ($timeLeft < $extendThreshold && $timeLeft > 300) {
            $sessionData['expires_at'] = now()->addHours(self::SESSION_DURATION)->timestamp;
            Session::put("tournament_auth_{$tournament->id}", $sessionData);
            
            Log::info("Tournament session auto-extended", [
                'tournament_id' => $tournament->id,
                'role' => $sessionData['role'],
                'new_expiry' => date('Y-m-d H:i:s', $sessionData['expires_at'])
            ]);
        }
    }
    
    /**
     * Get session time remaining in minutes
     */
    public function getSessionTimeRemaining(Tournament $tournament): ?int
    {
        $sessionData = Session::get("tournament_auth_{$tournament->id}");
        
        if (!$sessionData) {
            return null;
        }
        
        $timeLeft = $sessionData['expires_at'] - now()->timestamp;
        return max(0, intval($timeLeft / 60));
    }
    
    /**
     * Logout from tournament
     */
    public function logout(Tournament $tournament): void
    {
        $sessionKeys = [
            "tournament_auth_{$tournament->id}",
            "tournament_role_{$tournament->id}",
            "tournament_pw_{$tournament->id}" // Legacy compatibility
        ];
        
        foreach ($sessionKeys as $key) {
            Session::forget($key);
        }
        
        Log::info("Tournament logout", [
            'tournament_id' => $tournament->id,
            'user_id' => auth()->id()
        ]);
    }
    
    /**
     * Get comprehensive user permissions for tournament
     */
    public function getUserPermissions(Tournament $tournament): array
    {
        $user = auth()->user();
        $role = 'guest';
        $permissions = [];
        
        // Check authenticated user roles first
        if ($user) {
            if ($user->hasRole('admin')) {
                $role = 'admin';
                $permissions = ['all'];
            } elseif ($user->hasRole('mod')) {
                $role = 'mod'; 
                $permissions = ['manage', 'approve', 'export', 'view_logs'];
            }
        }
        
        // Check tournament-specific authentication
        if (!$user || (!$user->hasRole('admin') && !$user->hasRole('mod'))) {
            $tournamentRole = $this->getUserRole($tournament);
            if ($tournamentRole) {
                $role = $tournamentRole;
                $permissions = $this->getRolePermissions($tournamentRole);
            }
        }
        
        return [
            'role' => $role,
            'permissions' => $permissions,
            'can_manage' => in_array($role, ['admin', 'mod', 'organizer']),
            'can_approve' => in_array($role, ['admin', 'mod', 'organizer']),
            'can_update_scores' => in_array($role, ['admin', 'mod', 'organizer', 'referee']),
            'can_export' => in_array($role, ['admin', 'mod', 'organizer', 'referee']),
            'session_remaining' => $this->getSessionTimeRemaining($tournament)
        ];
    }
    
    /**
     * Get permissions for a specific role
     */
    private function getRolePermissions(string $role): array
    {
        return match($role) {
            'organizer' => ['manage', 'approve', 'export', 'view_logs'],
            'referee' => ['view', 'update_scores', 'export'],
            default => []
        };
    }
    
    /**
     * Get display name for role
     */
    private function getRoleDisplayName(string $role): string
    {
        return match($role) {
            'organizer' => 'Ban tổ chức',
            'referee' => 'Trọng tài',
            'admin' => 'Quản trị viên',
            'mod' => 'Điều hành viên', 
            default => 'Khách'
        };
    }
    
    /**
     * Validate session security
     */
    public function validateSessionSecurity(Tournament $tournament): bool
    {
        $sessionData = Session::get("tournament_auth_{$tournament->id}");
        
        if (!$sessionData) {
            return false;
        }
        
        // Check IP consistency (optional, might cause issues with mobile)
        $currentIp = request()->ip();
        $sessionIp = $sessionData['ip'] ?? null;
        
        // For now, we'll log IP changes but not invalidate session
        if ($sessionIp && $sessionIp !== $currentIp) {
            Log::info("Tournament session IP change detected", [
                'tournament_id' => $tournament->id,
                'old_ip' => $sessionIp,
                'new_ip' => $currentIp,
                'role' => $sessionData['role']
            ]);
        }
        
        return true;
    }
    
    /**
     * Get active sessions for tournament (for admin monitoring)
     */
    public function getActiveSessions(Tournament $tournament): array
    {
        // This would require storing sessions in database or cache for full implementation
        // For now, just return current session if exists
        $sessionData = Session::get("tournament_auth_{$tournament->id}");
        
        if (!$sessionData) {
            return [];
        }
        
        return [
            [
                'role' => $sessionData['role'],
                'authenticated_at' => date('Y-m-d H:i:s', $sessionData['authenticated_at']),
                'expires_at' => date('Y-m-d H:i:s', $sessionData['expires_at']),
                'user_id' => $sessionData['user_id'],
                'ip' => $sessionData['ip']
            ]
        ];
    }
}