<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;

class TelegramService
{
    protected $client;

    protected $botToken;

    protected $chatId;

    public function __construct()
    {
        // Configure Guzzle with aggressive timeouts to avoid blocking requests
        $this->client = new Client([
            'timeout' => 2.0,          // total timeout seconds
            'connect_timeout' => 1.0,  // connection timeout seconds
            'http_errors' => false,    // don't throw on 4xx/5xx
            // 'verify' => true,       // keep default; can be overridden via env if needed
            'headers' => [
                'User-Agent' => 'PickleballVNA/1.0',
            ],
        ]);
        $this->botToken = config('services.telegram.bot_token');
        $this->chatId = config('services.telegram.chat_id');
    }

    /**
     * Gửi thông báo đăng ký thành viên mới
     */
    public function sendNewMemberNotification($memberData)
    {
        if (empty($this->botToken) || empty($this->chatId)) {
            Log::warning('Telegram bot token hoặc chat ID chưa được cấu hình');

            return false;
        }

        $message = $this->formatMemberMessage($memberData);

        return $this->sendMessage($message);
    }

    /**
     * Format thông tin thành viên thành message
     */
    private function formatMemberMessage($memberData)
    {
        $message = "🎾 *ĐĂNG KÝ THÀNH VIÊN MỚI* 🎾\n\n";
        $message .= '👤 *Họ tên:* '.$memberData['name']."\n";
        $message .= '📧 *Email:* '.$memberData['email']."\n";
        $message .= '📱 *Điện thoại:* '.($memberData['phone'] ?? 'Không có')."\n";
        $message .= '📍 *Địa điểm:* '.$memberData['location']."\n";
        $message .= '🏆 *Trình độ:* '.$memberData['initial_level']."\n";
        $message .= '⚧ *Giới tính:* '.$memberData['gender']."\n";
        $message .= '⏰ *Thời gian:* '.now()->format('d/m/Y H:i:s')."\n\n";
        $message .= '✅ Vui lòng kiểm tra và duyệt thành viên mới!';

        return $message;
    }

    /**
     * Gửi message tới Telegram
     */
    public function sendMessage($message)
    {
        try {
            $response = $this->client->post("https://api.telegram.org/bot{$this->botToken}/sendMessage", [
                'json' => [
                    'chat_id' => $this->chatId,
                    'text' => $message,
                    'parse_mode' => 'Markdown',
                    'disable_web_page_preview' => true,
                    'disable_notification' => true,
                ],
            ]);

            $result = json_decode($response->getBody()->getContents(), true);

            if ($result['ok']) {
                Log::info('Gửi thông báo Telegram thành công');

                return true;
            } else {
                Log::error('Lỗi gửi thông báo Telegram: '.$result['description']);

                return false;
            }
        } catch (RequestException $e) {
            Log::error('Lỗi kết nối Telegram API: '.$e->getMessage());

            return false;
        } catch (\Exception $e) {
            Log::error('Lỗi không xác định khi gửi Telegram: '.$e->getMessage());

            return false;
        }
    }

    /**
     * Kiểm tra cấu hình Telegram
     */
    public function isConfigured()
    {
        return ! empty($this->botToken) && ! empty($this->chatId);
    }

    /**
     * Test gửi message
     */
    public function testConnection()
    {
        if (! $this->isConfigured()) {
            return ['success' => false, 'message' => 'Telegram chưa được cấu hình'];
        }

        $testMessage = "🤖 Test kết nối Telegram Bot thành công!\n⏰ ".now()->format('d/m/Y H:i:s');

        if ($this->sendMessage($testMessage)) {
            return ['success' => true, 'message' => 'Kết nối Telegram thành công'];
        } else {
            return ['success' => false, 'message' => 'Không thể gửi message tới Telegram'];
        }
    }
}