<?php

namespace App\Services;

use App\Models\Tournament;
use App\Models\TournamentGroup;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class StandingsService
{
    /**
     * Tính toán và trả về bảng xếp hạng cho một bảng đấu với memory-optimized caching.
     * Service này nhận vào các collection đã được eager-loaded để tối ưu hóa truy vấn.
     *
     * @param  Collection  $allRounds  Tất cả các vòng đấu của giải (đã load `matches`).
     * @return array Bảng xếp hạng đã được sắp xếp.
     */
    public function calculateForGroup(TournamentGroup $group, Collection $allRounds): array
    {
        // Emergency memory management
        ini_set('memory_limit', '1G');
        gc_collect_cycles();

        // Tạo cache key dựa trên group ID và timestamp của match cuối cùng
        // Chỉ tính những trận vòng bảng (round_robin) đã HOÀN THÀNH và cả hai cặp đều thuộc cùng một bảng
        $latestMatchTime = DB::table('matches as m')
            ->join('tournament_rounds as tr', 'm.tournament_round_id', '=', 'tr.id')
            ->join('group_pairs as gp1', 'm.pair1_id', '=', 'gp1.id')
            ->join('group_pairs as gp2', 'm.pair2_id', '=', 'gp2.id')
            ->where('gp1.group_id', $group->id)
            ->where('gp2.group_id', $group->id)
            ->where('tr.type', 'round_robin')
            ->where('m.status', 'Hoàn thành')
            ->max('m.updated_at');

        // Add version tag to bust cache when tie-breaker logic changes
        // v4: Only include completed (status = 'Hoàn thành') round_robin matches in standings and cache invalidation
        $cacheKey = "standings_group_v4_{$group->id}_".md5($latestMatchTime ?? '');

        return cache()->remember($cacheKey, 300, function () use ($group, $allRounds) {
            return $this->calculateStandingsForGroupOptimized($group, $allRounds);
        });
    }

    /**
     * Thực hiện tính toán bảng xếp hạng với memory optimization
     */
    private function calculateStandingsForGroupOptimized(TournamentGroup $group, Collection $allRounds): array
    {
        // Get only pair IDs to minimize memory usage
        $pairIds = DB::table('group_pairs')
            ->where('group_id', $group->id)
            ->pluck('id')
            ->toArray();

        // Early return if no pairs
        if (empty($pairIds)) {
            return [];
        }

        // Initialize standings with minimal data
        $standings = [];
        foreach ($pairIds as $pairId) {
            $standings[$pairId] = [
                'pair_id' => $pairId,
                'played' => 0,
                'wins' => 0,
                'losses' => 0,
                'points_for' => 0,
                'points_against' => 0,
                'points_diff' => 0,
            ];
        }

        // Free memory
        unset($pairIds);
        gc_collect_cycles();

        // 2. Get matches with minimal memory usage using direct DB query
        $pairIdsInGroup = array_keys($standings);

        $matchesForGroup = DB::table('matches as m')
            ->join('tournament_rounds as tr', 'm.tournament_round_id', '=', 'tr.id')
            ->whereIn('m.pair1_id', $pairIdsInGroup)
            ->whereIn('m.pair2_id', $pairIdsInGroup)
            ->where('tr.type', 'round_robin')
            // Chỉ tính các trận đã HOÀN THÀNH để không cập nhật bảng xếp hạng khi chưa chốt tỉ số
            ->where('m.status', 'Hoàn thành')
            ->whereNotNull('m.score')
            ->where('m.score', '!=', '')
            ->select('m.pair1_id', 'm.pair2_id', 'm.score')
            ->get();

        if ($matchesForGroup->isEmpty()) {
            // Load pair data only when returning results
            foreach ($standings as &$standing) {
                $pairData = DB::table('group_pairs')
                    ->join('players as p1', 'group_pairs.player1_id', '=', 'p1.id')
                    ->leftJoin('players as p2', 'group_pairs.player2_id', '=', 'p2.id')
                    ->where('group_pairs.id', $standing['pair_id'])
                    ->select('group_pairs.*', 'p1.name as player1_name', 'p2.name as player2_name')
                    ->first();

                // Create proper object structure that the view expects
                $pair = (object) [
                    'id' => $pairData->id,
                    'group_id' => $pairData->group_id,
                    'player1_id' => $pairData->player1_id,
                    'player2_id' => $pairData->player2_id,
                    'player1' => (object) [
                        'id' => $pairData->player1_id,
                        'name' => $pairData->player1_name ?? 'N/A',
                    ],
                    'player2' => (object) [
                        'id' => $pairData->player2_id,
                        'name' => $pairData->player2_name ?? 'N/A',
                    ],
                ];

                $standing['pair'] = $pair;
            }

            return array_values($standings);
        }

        // 3. Lặp qua các trận đấu để tính điểm (bao gồm cả trận có score đã lưu dù trạng thái chưa 'Hoàn thành')
        foreach ($matchesForGroup as $match) {
            // Chỉ tính các trận đã có tỷ số ở dạng 'x-y'
            if (empty($match->score) || ! str_contains($match->score, '-')) {
                continue;
            }

            [$score1, $score2] = array_map('intval', explode('-', $match->score));
            $pair1_id = $match->pair1_id;
            $pair2_id = $match->pair2_id;

            // Bảo vệ an toàn: nếu pair không thuộc standings (không nằm trong bảng), bỏ qua
            if (! isset($standings[$pair1_id]) || ! isset($standings[$pair2_id])) {
                continue;
            }

            // Cập nhật cho cặp 1
            $standings[$pair1_id]['played']++;
            $standings[$pair1_id]['points_for'] += $score1;
            $standings[$pair1_id]['points_against'] += $score2;

            // Cập nhật cho cặp 2
            $standings[$pair2_id]['played']++;
            $standings[$pair2_id]['points_for'] += $score2;
            $standings[$pair2_id]['points_against'] += $score1;

            // Xác định thắng/thua (trường hợp hòa: tính là không thắng không thua)
            if ($score1 > $score2) {
                $standings[$pair1_id]['wins']++;
                $standings[$pair2_id]['losses']++;
            } elseif ($score2 > $score1) {
                $standings[$pair2_id]['wins']++;
                $standings[$pair1_id]['losses']++;
            }
        }

        // Tính toán hiệu số trước khi sắp xếp
        foreach ($standings as &$standing) {
            $standing['points_diff'] = $standing['points_for'] - $standing['points_against'];
        }
        unset($standing); // Hủy tham chiếu

        // 4. Sắp xếp bảng xếp hạng
        // Tạo một map để lưu kết quả đối đầu, tránh tìm kiếm lặp lại
        // Sử dụng chính các trận đã lọc cho bảng này (matchesForGroup)
        $headToHeadResults = $this->buildHeadToHeadMap($matchesForGroup);

        usort($standings, function ($a, $b) use ($headToHeadResults) {
            // Ưu tiên 1: Số trận thắng (giảm dần)
            if ($a['wins'] !== $b['wins']) {
                return $b['wins'] <=> $a['wins'];
            }

            // Ưu tiên 2: Hiệu số tổng (giảm dần)
            if ($a['points_diff'] !== $b['points_diff']) {
                return $b['points_diff'] <=> $a['points_diff'];
            }

            // Ưu tiên 3: Đối đầu trực tiếp (chỉ hiệu lực nếu đã có trận và không hòa)
            $h2hResult = $this->getHeadToHeadResult($a['pair_id'], $b['pair_id'], $headToHeadResults);
            if ($h2hResult !== 0) {
                return $h2hResult;
            }

            // Ưu tiên 4: Tổng điểm ghi được (giảm dần)
            if ($a['points_for'] !== $b['points_for']) {
                return $b['points_for'] <=> $a['points_for'];
            }

            // Ưu tiên 5: Ổn định thứ tự theo ID (tăng dần)
            return $a['pair_id'] <=> $b['pair_id'];
        });

        // Load pair data for all standings
        foreach ($standings as &$standing) {
            $pairData = DB::table('group_pairs')
                ->join('players as p1', 'group_pairs.player1_id', '=', 'p1.id')
                ->leftJoin('players as p2', 'group_pairs.player2_id', '=', 'p2.id')
                ->where('group_pairs.id', $standing['pair_id'])
                ->select('group_pairs.*', 'p1.name as player1_name', 'p2.name as player2_name')
                ->first();

            // Create proper object structure that the view expects
            $pair = (object) [
                'id' => $pairData->id,
                'group_id' => $pairData->group_id,
                'player1_id' => $pairData->player1_id,
                'player2_id' => $pairData->player2_id,
                'player1' => (object) [
                    'id' => $pairData->player1_id,
                    'name' => $pairData->player1_name ?? 'N/A',
                ],
                'player2' => (object) [
                    'id' => $pairData->player2_id,
                    'name' => $pairData->player2_name ?? 'N/A',
                ],
            ];

            $standing['pair'] = $pair;
        }

        return $standings;
    }

    /**
     * Xây dựng một map lưu kết quả đối đầu để tra cứu nhanh.
     * Key là "pair1_id-pair2_id" (ID nhỏ hơn đứng trước), value là ID đội thắng.
     */
    private function buildHeadToHeadMap(Collection $matches): array
    {
        $map = [];
        foreach ($matches as $match) {
            if (empty($match->score) || ! str_contains($match->score, '-')) {
                continue;
            }
            [$score1, $score2] = array_map('intval', explode('-', $match->score));
            $p1 = $match->pair1_id;
            $p2 = $match->pair2_id;

            $key = min($p1, $p2).'-'.max($p1, $p2);
            $winnerId = null;
            if ($score1 > $score2) {
                $winnerId = $p1;
            } elseif ($score2 > $score1) {
                $winnerId = $p2;
            } // Nếu hòa, winnerId vẫn là null

            $map[$key] = [
                'winner_id' => $winnerId,
                'score_diff' => abs($score1 - $score2),
                'scores' => [$p1 => $score1, $p2 => $score2],
            ];
        }

        return $map;
    }

    /** Lấy kết quả đối đầu trực tiếp từ map. */
    private function getHeadToHeadResult(int $pairA_id, int $pairB_id, array $headToHeadMap): int
    {
        $key = min($pairA_id, $pairB_id).'-'.max($pairA_id, $pairB_id);
        if (! isset($headToHeadMap[$key])) {
            return 0;
        }
        $winnerId = $headToHeadMap[$key]['winner_id'];
        if ($winnerId == $pairA_id) {
            return -1;
        } // A thắng B
        if ($winnerId == $pairB_id) {
            return 1;
        } // B thắng A

        return 0;
    }

    /** Tính toán bảng xếp hạng phụ cho các đội bằng điểm. */
    private function calculateMiniStandings(array $tiedTeamIds, Collection $allMatchesForGroup): array
    {
        $miniStandings = [];
        foreach ($tiedTeamIds as $id) {
            $miniStandings[$id] = ['pair_id' => $id, 'wins' => 0, 'points_for' => 0, 'points_against' => 0, 'points_diff' => 0];
        }

        // Lọc các trận đấu chỉ giữa các đội bằng điểm
        $miniMatches = $allMatchesForGroup->filter(function ($match) use ($tiedTeamIds) {
            return in_array($match->pair1_id, $tiedTeamIds) && in_array($match->pair2_id, $tiedTeamIds);
        });

        foreach ($miniMatches as $match) {
            if (empty($match->score) || ! str_contains($match->score, '-')) {
                continue;
            }
            [$score1, $score2] = array_map('intval', explode('-', $match->score));
            $p1_id = $match->pair1_id;
            $p2_id = $match->pair2_id;

            // Cập nhật điểm cho bảng phụ
            $miniStandings[$p1_id]['points_for'] += $score1;
            $miniStandings[$p1_id]['points_against'] += $score2;
            $miniStandings[$p2_id]['points_for'] += $score2;
            $miniStandings[$p2_id]['points_against'] += $score1;

            if ($score1 > $score2) {
                $miniStandings[$p1_id]['wins']++;
            } elseif ($score2 > $score1) {
                $miniStandings[$p2_id]['wins']++;
            }
        }

        foreach ($miniStandings as &$standing) {
            $standing['points_diff'] = $standing['points_for'] - $standing['points_against'];
        }
        unset($standing);

        return $miniStandings;
    }

    /**
     * Get standings for all groups in a tournament
     */
    public function getGroupStandings(Tournament $tournament): array
    {
        $standingsByGroup = [];

        // Only calculate if tournament has groups AND round robin rounds
        if ($tournament->groups()->count() === 0 || $tournament->rounds()->where('type', 'round_robin')->count() === 0) {
            return $standingsByGroup;
        }

        $rounds = $tournament->rounds()->where('type', 'round_robin')->get();

        foreach ($tournament->groups as $group) {
            try {
                $standingsByGroup[$group->name] = $this->calculateForGroup($group, $rounds);
            } catch (\Exception $e) {
                \Log::warning("Failed to calculate standings for group {$group->name}: {$e->getMessage()}");
                $standingsByGroup[$group->name] = [];
            }
        }

        return $standingsByGroup;
    }
}
