<?php

namespace App\Services;

use App\Models\Player;
use App\Models\Registration;
use App\Models\Tournament;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class RegistrationService
{
    /**
     * Cập nhật trạng thái của một đơn đăng ký và xử lý các logic nghiệp vụ liên quan.
     *
     * @param  Registration  $registration  Đối tượng đăng ký cần cập nhật.
     * @param  string  $status  Trạng thái mới ('approved', 'rejected', 'pending').
     * @param  string|null  $reason  Lý do từ chối (nếu có).
     * @return Registration
     *
     * @throws \Exception Nếu có lỗi nghiệp vụ xảy ra.
     */
    public function updateStatus(Registration $registration, string $status, ?string $reason = null): Registration
    {
        // Chỉ kiểm tra khi chuyển sang trạng thái 'approved'
        if ($status === 'approved') {
            $tournament = $registration->tournament;

            // Kiểm tra xem giải đấu có giới hạn số cặp và đã đủ số lượng chưa.
            // Chúng ta không tính đăng ký hiện tại vào số lượng đã duyệt.
            if ($tournament->max_pairs > 0) {
                $approvedCount = Registration::where('tournament_id', $tournament->id)
                    ->where('status', 'approved')
                    ->where('id', '!=', $registration->id) // Loại trừ chính nó ra khỏi việc đếm
                    ->count();

                if ($approvedCount >= $tournament->max_pairs) {
                    // Ném ra một ngoại lệ để controller có thể bắt và hiển thị lỗi
                    throw new \Exception('Không thể duyệt. Giải đấu đã đủ số lượng cặp đăng ký tối đa ('.$tournament->max_pairs.' cặp).');
                }
            }
        }

        $registration->update([
            'status' => $status,
            'reason' => $reason,
        ]);

        // Invalidate cache after status update
        if (app()->bound('App\Services\TournamentCacheService')) {
            app('App\Services\TournamentCacheService')->invalidateTournamentCache($registration->tournament_id, ['overview']);
        }

        return $registration;
    }

    /**
     * Validate registration data before creating
     *
     * @param Tournament $tournament
     * @param array $data
     * @throws ValidationException
     */
    public function validateRegistration(Tournament $tournament, array $data): void
    {
        $player1 = Player::where('phone', $data['player1_phone'])->first();
        $player2 = null;

        if ($tournament->type === 'doubles' && !empty($data['player2_phone'])) {
            $player2 = Player::where('phone', $data['player2_phone'])->first();
        }

        // 1. Check duplicate registration (exclude rejected)
        $this->checkDuplicateRegistration($tournament, $player1, $player2);
        
        // 2. Check max pairs limit  
        $this->checkMaxPairsLimit($tournament);
        
        // 3. Check skill point requirements
        $this->checkSkillPointRequirements($tournament, $player1, $player2);
    }

    /**
     * Create a new registration with proper validation
     *
     * @param Tournament $tournament
     * @param array $data
     * @return Registration
     * @throws ValidationException|\Exception
     */
    public function createRegistration(Tournament $tournament, array $data): Registration
    {
        DB::beginTransaction();
        
        try {
            $this->validateRegistration($tournament, $data);
            
            $player1 = Player::where('phone', $data['player1_phone'])->first();
            $player2 = null;
            
            if ($tournament->type === 'doubles' && !empty($data['player2_phone'])) {
                $player2 = Player::where('phone', $data['player2_phone'])->first();
            }

            $scoreSum = $tournament->type === 'doubles'
                ? ($player1->doubles_score ?? 0) + ($player2->doubles_score ?? 0)
                : ($player1->singles_score ?? 0);

            $registration = Registration::create([
                'tournament_id' => $tournament->id,
                'player1_id' => $player1->id,
                'player2_id' => $player2?->id,
                'score_sum' => $scoreSum,
                'status' => 'pending',
                'reason' => $data['note'] ?? null,
            ]);

            // Invalidate caches
            if (app()->bound('App\Services\TournamentCacheService')) {
                app('App\Services\TournamentCacheService')->invalidateTournamentCache($tournament->id, ['overview']);
            }
            
            DB::commit();
            return $registration;
            
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Check for duplicate registrations
     */
    private function checkDuplicateRegistration(Tournament $tournament, Player $player1, ?Player $player2): void
    {
        if ($tournament->type === 'doubles' && $player2) {
            // Check pair exists (both directions, exclude rejected)
            $pairExists = Registration::where('tournament_id', $tournament->id)
                ->where('status', '!=', 'rejected')
                ->where(function ($q) use ($player1, $player2) {
                    $q->where(function ($q1) use ($player1, $player2) {
                        $q1->where('player1_id', $player1->id)->where('player2_id', $player2->id);
                    })->orWhere(function ($q2) use ($player1, $player2) {
                        $q2->where('player1_id', $player2->id)->where('player2_id', $player1->id);
                    });
                })->exists();

            if ($pairExists) {
                throw ValidationException::withMessages([
                    'player2_phone' => 'Cặp VĐV này đã đăng ký giải này!'
                ]);
            }

            // Check individual player participation
            $this->checkPlayerAlreadyRegistered($tournament, $player1, 'player1_phone');
            $this->checkPlayerAlreadyRegistered($tournament, $player2, 'player2_phone');
        } else {
            // Singles
            $this->checkPlayerAlreadyRegistered($tournament, $player1, 'player1_phone');
        }
    }

    /**
     * Check if individual player already registered
     */
    private function checkPlayerAlreadyRegistered(Tournament $tournament, Player $player, string $field): void
    {
        $exists = Registration::where('tournament_id', $tournament->id)
            ->where('status', '!=', 'rejected')
            ->where(function ($q) use ($player) {
                $q->where('player1_id', $player->id)->orWhere('player2_id', $player->id);
            })->exists();

        if ($exists) {
            $message = $field === 'player1_phone' ? 'Người chơi 1 đã tham gia đăng ký trong giải này!' : 'Người chơi 2 đã tham gia đăng ký trong giải này!';
            throw ValidationException::withMessages([
                $field => $message
            ]);
        }
    }

    /**
     * Check max pairs limit
     */
    private function checkMaxPairsLimit(Tournament $tournament): void
    {
        if ($tournament->max_pairs > 0) {
            $currentCount = Registration::where('tournament_id', $tournament->id)
                ->where('status', '!=', 'rejected')
                ->count();

            if ($currentCount >= $tournament->max_pairs) {
                throw ValidationException::withMessages([
                    'max_pairs' => 'Số lượng cặp/đội đã đủ cho giải này!'
                ]);
            }
        }
    }

    /**
     * Check skill point requirements
     */
    private function checkSkillPointRequirements(Tournament $tournament, Player $player1, ?Player $player2): void
    {
        if (!$tournament->skill_point || !$tournament->skill_tolerance) {
            return; // No skill restrictions
        }

        $scoreSum = $tournament->type === 'doubles'
            ? ($player1->doubles_score ?? 0) + ($player2?->doubles_score ?? 0)
            : ($player1->singles_score ?? 0);

        $maxSkill = $tournament->skill_point + $tournament->skill_tolerance;

        if ($scoreSum > $maxSkill) {
            throw ValidationException::withMessages([
                'score_sum' => "Tổng điểm trình của " . ($tournament->type === 'doubles' ? 'cặp' : 'VĐV') . " đăng ký là " . number_format($scoreSum, 2) . ", vượt quá giới hạn cho phép là " . number_format($maxSkill, 2) . "."
            ]);
        }
    }
}