<?php

namespace App\Services;

use App\Models\Player;
use App\Models\User;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;

class PlayerService
{
    /**
     * Tạo mới Player và User liên quan.
     *
     * @param  array  $data  Dữ liệu đã được validate từ request.
     * @return array Trả về một mảng chứa ['player' => Player, 'password' => string|null]
     *
     * @throws \Exception
     */
    public function createPlayer(array $data): array
    {
        return DB::transaction(function () use ($data) {
            // Xử lý avatar
            if (isset($data['avatar']) && $data['avatar'] instanceof UploadedFile) {
                $data['avatar'] = $this->handleAvatarUpload($data['avatar']);
            }

            // Sinh player_id duy nhất
            $playerId = $this->generateUniquePlayerId();

            // Tạo user và lấy mật khẩu (nếu có)
            $userCreationResult = $this->createAssociatedUser($data);
            $user = $userCreationResult['user'];
            $password = $userCreationResult['password'];

            // Tạo player
            $player = Player::create([
                'user_id' => $user->id,
                'player_id' => $playerId,
                'name' => $data['name'],
                'doubles_score' => $data['initial_level'],
                'singles_score' => $data['initial_level'],
                'max_score' => $data['initial_level'],
                'lock_score' => $data['initial_level'],
                'location' => $data['location'] ?? null,
                'phone' => $data['phone'] ?? null,
                'gender' => $data['gender'] ?? null,
                'avatar' => $data['avatar'] ?? null,
                'status' => $data['status'] ?? 0,
                'created_by' => $data['created_by'] ?? null,
            ]);

            return [
                'player' => $player,
                'password' => $password,
            ];
        });
    }

    /**
     * Cập nhật thông tin Player và User liên quan.
     *
     * @param  array  $data  Dữ liệu đã được validate từ request.
     */
    public function updatePlayer(Player $player, array $data): Player
    {
        return DB::transaction(function () use ($player, $data) {
            // Xử lý avatar mới
            if (isset($data['avatar']) && $data['avatar'] instanceof UploadedFile) {
                $data['avatar'] = $this->handleAvatarUpload($data['avatar'], $player->avatar);
            }

            // Cập nhật User liên kết
            if ($player->user) {
                $player->user->update([
                    'name' => $data['name'],
                    'email' => $data['email'],
                    'phone' => $data['phone'],
                    'location' => $data['location'],
                    'gender' => $data['gender'],
                    'avatar' => $data['avatar'] ?? $player->user->avatar,
                ]);
            }

            // Cập nhật Player
            $player->update($data);

            return $player;
        });
    }

    private function handleAvatarUpload(UploadedFile $file, ?string $oldAvatarPath = null): string
    {
        // Sử dụng ImageOptimizationService để tối ưu hóa
        $imageService = app(ImageOptimizationService::class);
        $optimizedImages = $imageService->optimizeAndStore($file, 'avatar', $oldAvatarPath);
        
        // Trả về path của medium size (mặc định)
        return $optimizedImages['medium']['jpeg'] ?? $this->fallbackUpload($file, $oldAvatarPath);
    }
    
    /**
     * Fallback upload method (legacy support)
     */
    private function fallbackUpload(UploadedFile $file, ?string $oldAvatarPath = null): string
    {
        if ($oldAvatarPath) {
            // Handle both old storage format and new assets format
            if (str_starts_with($oldAvatarPath, 'avatars/')) {
                Storage::disk('public')->delete($oldAvatarPath);
            } elseif (str_starts_with($oldAvatarPath, 'assets/avatars/')) {
                $avatarPath = public_path($oldAvatarPath);
                if (file_exists($avatarPath)) {
                    unlink($avatarPath);
                }
            }
        }

        // Upload avatar mới vào public/assets/avatars
        $filename = time() . '_' . $file->getClientOriginalName();
        $file->move(public_path('assets/avatars'), $filename);
        return 'assets/avatars/' . $filename;
    }

    private function generateUniquePlayerId(): string
    {
        do {
            $playerId = strtoupper(Str::random(6));
        } while (Player::where('player_id', $playerId)->exists());

        return $playerId;
    }

    private function createAssociatedUser(array $data): array
    {
        $password = $data['password'] ?? null;
        $hashedPassword = null;

        if ($password) {
            $hashedPassword = Hash::make($password);
        } else {
            // Nếu không có password, tạo một mật khẩu ngẫu nhiên
            $password = Str::random(10);
            $hashedPassword = Hash::make($password);
        }

        $user = User::create(array_merge($data, ['password' => $hashedPassword]));
        Role::findOrCreate('user', 'web');
        $user->assignRole('user');

        return ['user' => $user, 'password' => $password];
    }
}