<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PerformanceMonitorService
{
    /**
     * Monitor and log performance metrics
     */
    public function logPerformanceMetrics(string $operation, float $executionTime, int $queryCount, array $context = []): void
    {
        $metrics = [
            'operation' => $operation,
            'execution_time_ms' => round($executionTime * 1000, 2),
            'query_count' => $queryCount,
            'memory_usage_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2),
            'context' => $context,
            'timestamp' => now()->toISOString()
        ];

        // Log to application log
        Log::info('Performance Metrics', $metrics);

        // Store in cache for dashboard display
        $this->storeMetricsInCache($metrics);

        // Alert if performance is poor
        $this->checkPerformanceThresholds($metrics);
    }

    /**
     * Store metrics in cache for dashboard
     */
    private function storeMetricsInCache(array $metrics): void
    {
        $key = 'performance_metrics_' . date('Y-m-d-H');
        $cached = Cache::get($key, []);
        
        $cached[] = $metrics;
        
        // Keep only last 100 entries per hour
        if (count($cached) > 100) {
            $cached = array_slice($cached, -100);
        }
        
        Cache::put($key, $cached, 3600); // Store for 1 hour
    }

    /**
     * Check performance thresholds and alert
     */
    private function checkPerformanceThresholds(array $metrics): void
    {
        // Alert thresholds
        $executionThreshold = 5000; // 5 seconds
        $queryThreshold = 50;
        $memoryThreshold = 256; // 256MB

        $alerts = [];

        if ($metrics['execution_time_ms'] > $executionThreshold) {
            $alerts[] = "High execution time: {$metrics['execution_time_ms']}ms";
        }

        if ($metrics['query_count'] > $queryThreshold) {
            $alerts[] = "High query count: {$metrics['query_count']} queries";
        }

        if ($metrics['memory_usage_mb'] > $memoryThreshold) {
            $alerts[] = "High memory usage: {$metrics['memory_usage_mb']}MB";
        }

        if (!empty($alerts)) {
            Log::warning('Performance Alert', [
                'operation' => $metrics['operation'],
                'alerts' => $alerts,
                'metrics' => $metrics
            ]);
        }
    }

    /**
     * Get performance metrics for dashboard
     */
    public function getRecentMetrics(int $hours = 24): array
    {
        $metrics = [];
        
        for ($i = 0; $i < $hours; $i++) {
            $hour = now()->subHours($i);
            $key = 'performance_metrics_' . $hour->format('Y-m-d-H');
            $hourlyMetrics = Cache::get($key, []);
            
            $metrics = array_merge($metrics, $hourlyMetrics);
        }

        return array_reverse($metrics); // Most recent first
    }

    /**
     * Clear old performance data
     */
    public function clearOldMetrics(): void
    {
        // Clear metrics older than 24 hours
        for ($i = 25; $i < 72; $i++) { // Clear 25-72 hours ago
            $hour = now()->subHours($i);
            $key = 'performance_metrics_' . $hour->format('Y-m-d-H');
            Cache::forget($key);
        }
    }

    /**
     * Get database health metrics
     */
    public function getDatabaseHealth(): array
    {
        try {
            $health = [
                'status' => 'healthy',
                'connection_count' => $this->getConnectionCount(),
                'slow_queries' => $this->getSlowQueryCount(),
                'table_sizes' => $this->getTableSizes(),
                'last_checked' => now()->toISOString()
            ];

            // Check for issues
            if ($health['connection_count'] > 50) {
                $health['status'] = 'warning';
                $health['warnings'][] = 'High connection count';
            }

            if ($health['slow_queries'] > 10) {
                $health['status'] = 'warning';
                $health['warnings'][] = 'High slow query count';
            }

            return $health;
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'error' => $e->getMessage(),
                'last_checked' => now()->toISOString()
            ];
        }
    }

    /**
     * Get database connection count
     */
    private function getConnectionCount(): int
    {
        try {
            $result = DB::select('SHOW STATUS LIKE "Threads_connected"');
            return (int) $result[0]->Value;
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * Get slow query count
     */
    private function getSlowQueryCount(): int
    {
        try {
            $result = DB::select('SHOW STATUS LIKE "Slow_queries"');
            return (int) $result[0]->Value;
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * Get table sizes
     */
    private function getTableSizes(): array
    {
        try {
            $tables = ['tournaments', 'registrations', 'matches', 'tournament_groups', 'group_pairs'];
            $sizes = [];

            foreach ($tables as $table) {
                $result = DB::select("
                    SELECT 
                        table_name,
                        round(((data_length + index_length) / 1024 / 1024), 2) AS 'size_mb'
                    FROM information_schema.TABLES 
                    WHERE table_schema = DATABASE() AND table_name = ?
                ", [$table]);

                if (!empty($result)) {
                    $sizes[$table] = $result[0]->size_mb;
                }
            }

            return $sizes;
        } catch (\Exception $e) {
            return [];
        }
    }
}