<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Intervention\Image\Drivers\Gd\Driver as GdDriver;
use Intervention\Image\Drivers\Imagick\Driver as ImagickDriver;
use Intervention\Image\ImageManager;

class ImageOptimizationService
{
    private ImageManager $manager;

    public function __construct()
    {
        // Prefer Imagick driver if available (faster/more memory efficient), fallback to GD
        // Note: On Windows/XAMPP, Imagick may not be installed; this will gracefully use GD.
        $driver = \extension_loaded('imagick') ? new ImagickDriver : new GdDriver;
        $this->manager = new ImageManager($driver);
    }

    /**
     * Tối ưu hóa và lưu trữ hình ảnh với multiple formats và sizes
     */
    public function optimizeAndStore(UploadedFile $file, string $type = 'avatar', ?string $oldPath = null): array
    {
        // Xóa file cũ nếu có
        if ($oldPath) {
            $this->deleteOldImage($oldPath);
        }

        // Avatar path: optimize for speed. Generate a single square image quickly.
        if ($type === 'avatar' || $type === 'avatar_pending') {
            $pending = ($type === 'avatar_pending');
            $quality = 80; // slightly lower for speed/size
            $target = $pending ? 'assets/avatars/pending' : 'assets/avatars';

            // Ensure directory exists
            $this->ensureDirectory(public_path($target));

            // Generate filename
            $filename = time().'_'.uniqid().'.jpg';
            $savePath = $target.'/'.$filename;

            // Read once, crop to square, and resize to 256x256 (good tradeoff)
            $image = $this->manager->read($file);
            // cover() will crop center and scale to requested size
            $image->cover(256, 256);

            // Save JPEG only for maximum compatibility and speed
            $image->save(public_path($savePath), quality: $quality, format: 'jpeg');

            // Optionally also save a WebP version if desired; skip for maximum speed
            // $image->save(public_path(str_replace('.jpg', '.webp', $savePath)), quality: $quality, format: 'webp');

            // Return structure compatible with callers expecting ['medium']['jpeg']
            return [
                'medium' => [
                    // 'webp' => str_replace('.jpg', '.webp', $savePath),
                    'jpeg' => $savePath,
                ],
            ];
        }

        // Cấu hình theo loại hình ảnh (venue/tournament): multi-size but more efficient
        $config = $this->getImageConfig($type);

        // Tạo tên file unique
        $filename = time().'_'.uniqid();
        $basePath = "assets/{$type}s/";
        $results = [];

        // Read the original once
        $original = $this->manager->read($file);

        // Tạo multiple sizes cho responsive loading
        foreach ($config['sizes'] as $size => $dimensions) {
            $dir = public_path($basePath.$size);
            $this->ensureDirectory($dir);

            $processedImage = clone $original;
            if (! empty($dimensions['crop'])) {
                $processedImage->cover($dimensions['width'], $dimensions['height']);
            } else {
                $processedImage->scale($dimensions['width'], $dimensions['height']);
            }

            // Lưu WebP (format tối ưu nhất)
            $webpPath = $basePath.$size.'/'.$filename.'.webp';
            $processedImage->save(public_path($webpPath), quality: $config['quality'], format: 'webp');

            // Lưu fallback JPEG
            $jpegPath = $basePath.$size.'/'.$filename.'.jpg';
            $processedImage->save(public_path($jpegPath), quality: $config['quality'], format: 'jpeg');

            $results[$size] = [
                'webp' => $webpPath,
                'jpeg' => $jpegPath,
            ];
        }

        return $results;
    }

    /**
     * Xử lý hình ảnh: resize, crop, optimize
     */
    private function processImage(UploadedFile $file, array $dimensions, int $quality)
    {
        $image = $this->manager->read($file);

        // Resize maintaining aspect ratio
        if (isset($dimensions['crop']) && $dimensions['crop']) {
            // Crop to exact dimensions (for avatars)
            $image->cover($dimensions['width'], $dimensions['height']);
        } else {
            // Resize maintaining ratio (for venues, tournaments)
            $image->scale($dimensions['width'], $dimensions['height']);
        }

        return $image;
    }

    /**
     * Cấu hình cho từng loại hình ảnh
     */
    private function getImageConfig(string $type): array
    {
        $configs = [
            'avatar' => [
                'sizes' => [
                    'thumb' => ['width' => 64, 'height' => 64, 'crop' => true],
                    'small' => ['width' => 128, 'height' => 128, 'crop' => true],
                    'medium' => ['width' => 256, 'height' => 256, 'crop' => true],
                    'large' => ['width' => 512, 'height' => 512, 'crop' => true],
                ],
                'quality' => 85,
            ],
            'venue' => [
                'sizes' => [
                    'thumb' => ['width' => 300, 'height' => 200, 'crop' => true],
                    'medium' => ['width' => 800, 'height' => 600, 'crop' => false],
                    'large' => ['width' => 1200, 'height' => 900, 'crop' => false],
                ],
                'quality' => 80,
            ],
            'tournament' => [
                'sizes' => [
                    'thumb' => ['width' => 400, 'height' => 300, 'crop' => true],
                    'medium' => ['width' => 800, 'height' => 600, 'crop' => false],
                    'large' => ['width' => 1200, 'height' => 900, 'crop' => false],
                ],
                'quality' => 80,
            ],
        ];

        return $configs[$type] ?? $configs['avatar'];
    }

    /**
     * Xóa hình ảnh cũ (tất cả sizes)
     */
    private function deleteOldImage(string $oldPath): void
    {
        // Xử lý path cũ (có thể là single file hoặc array)
        if (str_starts_with($oldPath, 'assets/')) {
            $fullPath = public_path($oldPath);
            if (file_exists($fullPath)) {
                unlink($fullPath);
            }
        }

        // Xóa các variant sizes nếu có
        $pathInfo = pathinfo($oldPath);
        $baseName = $pathInfo['filename'];
        $baseDir = dirname(public_path($oldPath));

        foreach (['thumb', 'small', 'medium', 'large'] as $size) {
            $sizeDir = $baseDir.'/'.$size;
            if (is_dir($sizeDir)) {
                foreach (['webp', 'jpg'] as $ext) {
                    $filePath = $sizeDir.'/'.$baseName.'.'.$ext;
                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }
                }
            }
        }
    }

    /**
     * Ensure a directory exists (recursive)
     */
    private function ensureDirectory(string $path): void
    {
        if (! is_dir($path)) {
            @mkdir($path, 0755, true);
        }
    }

    /**
     * Lấy đường dẫn tối ưu cho hiển thị
     */
    public function getOptimalImagePath(array $imagePaths, string $size = 'medium'): array
    {
        if (isset($imagePaths[$size]) && is_array($imagePaths[$size])) {
            $entry = $imagePaths[$size];
            $jpeg = $entry['jpeg'] ?? null;
            $webp = $entry['webp'] ?? null;

            $primary = $webp ?? $jpeg;
            $fallback = $jpeg ?? $webp;

            if ($primary || $fallback) {
                return [
                    'webp' => $webp,
                    'jpeg' => $jpeg,
                    'alt' => $fallback ?? $primary,
                ];
            }
        }

        // Return default if not found
        return [
            'webp' => 'assets/images/default.webp',
            'jpeg' => 'assets/images/default.jpg',
            'alt' => 'assets/images/default.jpg',
        ];
    }

    /**
     * Batch optimize existing images
     */
    public function batchOptimizeExisting(string $type): array
    {
        $path = public_path("assets/{$type}s/");
        $results = ['processed' => 0, 'errors' => 0];

        if (! is_dir($path)) {
            return $results;
        }

        $files = glob($path.'*.{jpg,jpeg,png,gif,webp}', GLOB_BRACE);

        foreach ($files as $filePath) {
            try {
                $filename = pathinfo($filePath, PATHINFO_FILENAME);
                $config = $this->getImageConfig($type);

                // Skip if already optimized
                if (is_dir($path.'thumb/') && file_exists($path.'thumb/'.$filename.'.webp')) {
                    continue;
                }

                $image = $this->manager->read($filePath);

                foreach ($config['sizes'] as $size => $dimensions) {
                    $sizeDir = $path.$size.'/';
                    if (! is_dir($sizeDir)) {
                        mkdir($sizeDir, 0755, true);
                    }

                    $processedImage = clone $image;

                    if (isset($dimensions['crop']) && $dimensions['crop']) {
                        $processedImage->cover($dimensions['width'], $dimensions['height']);
                    } else {
                        $processedImage->scale($dimensions['width'], $dimensions['height']);
                    }

                    // Save WebP and JPEG
                    $processedImage->save($sizeDir.$filename.'.webp', quality: $config['quality'], format: 'webp');
                    $processedImage->save($sizeDir.$filename.'.jpg', quality: $config['quality'], format: 'jpeg');
                }

                $results['processed']++;

            } catch (\Exception $e) {
                $results['errors']++;
                \Log::error("Failed to optimize image {$filePath}: ".$e->getMessage());
            }
        }

        return $results;
    }
}
