<?php

namespace App\Services;

use App\Models\MatchModel;
use App\Models\Player;

class EloService
{
    const K_FACTOR = 0.03; // Hệ số K, có thể điều chỉnh

    /**
     * Cập nhật điểm Elo cho tất cả người chơi trong một trận đấu.
     *
     * @param  MatchModel  $match  Trận đấu đã hoàn thành với đầy đủ thông tin.
     */
    public function updateEloForMatch(MatchModel $match): void
    {
        // Chỉ cập nhật khi trận đấu có tỷ số và có đủ 2 cặp
        if (empty($match->score) || ! $match->pair1 || ! $match->pair2) {
            return;
        }

        // Eager load thông tin người chơi để tối ưu
        $match->load('pair1.player1', 'pair1.player2', 'pair2.player1', 'pair2.player2');

        if ($match->type === 'doubles' || $match->type === 'round_robin' || $match->type === 'playoff') {
            $this->updateEloForDoublesMatch($match);
        } elseif ($match->type === 'singles') {
            $this->updateEloForSinglesMatch($match);
        }
    }

    /**
     * Cập nhật Elo cho trận đấu đôi.
     */
    private function updateEloForDoublesMatch(MatchModel $match): void
    {
        $player1_pair1 = $match->pair1->player1;
        $player2_pair1 = $match->pair1->player2;
        $player1_pair2 = $match->pair2->player1;
        $player2_pair2 = $match->pair2->player2;

        if (! $player1_pair1 || ! $player2_pair1 || ! $player1_pair2 || ! $player2_pair2) {
            return; // Cần đủ 4 người chơi
        }

        // Xác định kết quả thực tế
        $scoreParts = explode('-', $match->score);
        if (count($scoreParts) !== 2) {
            return; // Tỷ số không hợp lệ
        }
        [$score1, $score2] = array_map('intval', $scoreParts);
        $pair1_won = $score1 > $score2;

        // Tính điểm trung bình của mỗi đội
        $avg_rating_pair1 = ($player1_pair1->doubles_score + $player2_pair1->doubles_score) / 2;
        $avg_rating_pair2 = ($player1_pair2->doubles_score + $player2_pair2->doubles_score) / 2;

        // Tính toán và cập nhật Elo cho từng người chơi trong cặp 1
        $expected_win_p1p1 = 1 / (1 + pow(10, ($avg_rating_pair2 - $player1_pair1->doubles_score) / 400));
        $scoreChange_p1p1 = self::K_FACTOR * (($pair1_won ? 1 : 0) - $expected_win_p1p1);
        $this->updatePlayerDoublesStats($player1_pair1, $scoreChange_p1p1);

        $expected_win_p2p1 = 1 / (1 + pow(10, ($avg_rating_pair2 - $player2_pair1->doubles_score) / 400));
        $scoreChange_p2p1 = self::K_FACTOR * (($pair1_won ? 1 : 0) - $expected_win_p2p1);
        $this->updatePlayerDoublesStats($player2_pair1, $scoreChange_p2p1);

        // Tính toán và cập nhật Elo cho từng người chơi trong cặp 2
        $expected_win_p1p2 = 1 / (1 + pow(10, ($avg_rating_pair1 - $player1_pair2->doubles_score) / 400));
        $scoreChange_p1p2 = self::K_FACTOR * ((! $pair1_won ? 1 : 0) - $expected_win_p1p2);
        $this->updatePlayerDoublesStats($player1_pair2, $scoreChange_p1p2);

        $expected_win_p2p2 = 1 / (1 + pow(10, ($avg_rating_pair1 - $player2_pair2->doubles_score) / 400));
        $scoreChange_p2p2 = self::K_FACTOR * ((! $pair1_won ? 1 : 0) - $expected_win_p2p2);
        $this->updatePlayerDoublesStats($player2_pair2, $scoreChange_p2p2);

        // Lưu lại sự thay đổi điểm vào trận đấu để tham khảo
        // Lưu giá trị trung bình của sự thay đổi điểm của đội 1 (có thể âm hoặc dương)
        $avg_change_pair1 = ($scoreChange_p1p1 + $scoreChange_p2p1) / 2;
        $match->score_change = $avg_change_pair1;
        $match->saveQuietly(); // Lưu không trigger event
    }

    /**
     * Cập nhật chỉ số (Elo, wins, losses) cho một người chơi.
     */
    private function updatePlayerDoublesStats(Player $player, float $scoreChange): void
    {
        $player->doubles_score += $scoreChange;

        // Giới hạn điểm tối thiểu
        $minScore = ($player->gender === 'Nữ') ? 2.0 : 2.5;
        if ($player->doubles_score < $minScore) {
            $player->doubles_score = $minScore;
        }

        $player->doubles_matches += 1;

        if ($scoreChange > 0) {
            $player->doubles_wins += 1;
        } else {
            $player->doubles_losses += 1;
        }

        if ($player->doubles_score > $player->max_score) {
            $player->max_score = $player->doubles_score;
        }

        $player->save();
    }

    /**
     * Cập nhật Elo cho trận đấu đơn.
     */
    private function updateEloForSinglesMatch(MatchModel $match): void
    {
        // Giả định trận đơn, mỗi cặp chỉ có player1
        $player1 = $match->pair1->player1;
        $player2 = $match->pair2->player1;

        if (! $player1 || ! $player2) {
            return; // Cần đủ 2 người chơi
        }

        $rating1 = $player1->singles_score;
        $rating2 = $player2->singles_score;

        // Tính toán tỷ lệ thắng dự kiến
        $expected1 = 1 / (1 + pow(10, ($rating2 - $rating1) / 400));

        // Xác định kết quả thực tế
        $scoreParts = explode('-', $match->score);
        if (count($scoreParts) !== 2) {
            return; // Tỷ số không hợp lệ
        }
        [$score1, $score2] = array_map('intval', $scoreParts);
        $actual1 = ($score1 > $score2) ? 1 : 0;

        // Tính toán sự thay đổi điểm Elo
        $scoreChange = self::K_FACTOR * ($actual1 - $expected1);

        // Cập nhật Elo cho từng người chơi
        $this->updatePlayerSinglesStats($player1, $scoreChange);
        $this->updatePlayerSinglesStats($player2, -$scoreChange);

        // Lưu lại sự thay đổi điểm vào trận đấu để tham khảo
        $match->score_change = $scoreChange;
        $match->saveQuietly();
    }

    /**
     * Cập nhật chỉ số (Elo, wins, losses) cho một người chơi trong trận đơn.
     */
    private function updatePlayerSinglesStats(Player $player, float $scoreChange): void
    {
        $player->singles_score += $scoreChange;

        // Giới hạn điểm tối thiểu cho trận đơn (có thể khác trận đôi)
        $minScore = 2.0; // Ví dụ
        if ($player->singles_score < $minScore) {
            $player->singles_score = $minScore;
        }

        $player->singles_matches += 1;

        if ($scoreChange > 0) {
            $player->singles_win += 1;
        } else {
            $player->singles_lose += 1;
        }

        if ($player->singles_score > $player->max_score) {
            $player->max_score = $player->singles_score;
        }

        $player->save();
    }
}