<?php

namespace App\Rules;

use Closure;
use Illuminate\Contracts\Validation\ValidationRule;
use Illuminate\Http\UploadedFile;

class ValidImageContent implements ValidationRule
{
    private array $allowedMimes;

    public function __construct(array $allowedMimes = null)
    {
        $this->allowedMimes = $allowedMimes ?? [
            'image/jpeg',
            'image/pjpeg',
            'image/png', 
            'image/gif',
            'image/webp'
        ];
    }

    /**
     * Run the validation rule.
     *
     * @param  \Closure(string): \Illuminate\Translation\PotentiallyTranslatedString  $fail
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        if (!$value instanceof UploadedFile) {
            $fail('File không hợp lệ.');
            return;
        }

        if (!$value->isValid()) {
            $fail('File upload không thành công.');
            return;
        }

        // Check file content using magic bytes
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $contentMimeType = finfo_file($finfo, $value->getPathname());
        finfo_close($finfo);

        if (!$contentMimeType || !in_array($contentMimeType, $this->allowedMimes)) {
            $fail('File không phải là hình ảnh hợp lệ hoặc đã bị hỏng.');
            return;
        }

        // Additional security check: verify file header
        if (!$this->validateImageHeader($value->getPathname())) {
            $fail('Định dạng hình ảnh không được hỗ trợ hoặc file bị lỗi.');
            return;
        }
    }

    /**
     * Validate image file header bytes
     */
    private function validateImageHeader(string $filePath): bool
    {
        $handle = fopen($filePath, 'rb');
        if (!$handle) {
            return false;
        }

        $header = fread($handle, 12);
        fclose($handle);

        // JPEG: FF D8 FF
        if (substr($header, 0, 3) === "\xFF\xD8\xFF") {
            return true;
        }

        // PNG: 89 50 4E 47 0D 0A 1A 0A  
        if (substr($header, 0, 8) === "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A") {
            return true;
        }

        // GIF: GIF87a or GIF89a
        if (substr($header, 0, 6) === "GIF87a" || substr($header, 0, 6) === "GIF89a") {
            return true;
        }

        // WebP: RIFF....WEBP
        if (substr($header, 0, 4) === "RIFF" && substr($header, 8, 4) === "WEBP") {
            return true;
        }

        return false;
    }
}