<?php

namespace App\Rules;

/**
 * Centralized image validation rules to ensure consistency
 * across all image upload functionality
 */
class ImageValidationRules
{
    /**
     * Avatar validation rules for user profiles and players
     * Max size: 2MB (2048KB)
     * Formats: JPEG, JPG, PNG, GIF, WebP
     */
    public static function avatar(): array
    {
        return [
            'nullable',
            'file',
            'image',
            'mimes:jpeg,jpg,png,gif,webp,pjpeg', // Include pjpeg for IE compatibility
            'max:2048', // 2MB in kilobytes
            'dimensions:min_width=32,min_height=32,max_width=4096,max_height=4096',
        ];
    }

    /**
     * Venue images validation rules
     * Max size: 5MB (5120KB)
     * Max count: 10 images
     */
    public static function venueImages(): array
    {
        return [
            'nullable',
            'array',
            'max:10', // Maximum 10 images
        ];
    }

    /**
     * Individual venue image validation
     */
    public static function venueImage(): array
    {
        return [
            'image',
            'mimes:jpeg,jpg,png,gif,webp,pjpeg',
            'max:5120', // 5MB in kilobytes
            'dimensions:min_width=100,min_height=100,max_width=4096,max_height=4096',
        ];
    }

    /**
     * Tournament image validation rules
     * Max size: 2MB (2048KB)
     */
    public static function tournamentImage(): array
    {
        return [
            'nullable',
            'image',
            'mimes:jpeg,jpg,png,gif,webp,pjpeg',
            'max:2048', // 2MB in kilobytes
            'dimensions:min_width=200,min_height=150,max_width=4096,max_height=4096',
        ];
    }

    /**
     * Match images validation (multiple)
     */
    public static function matchImages(): array
    {
        return [
            'nullable',
            'array',
            'max:5', // Maximum 5 images per match
        ];
    }

    /**
     * Individual match image validation
     */
    public static function matchImage(): array
    {
        return [
            'image',
            'mimes:jpeg,jpg,png,gif,webp,pjpeg',
            'max:2048', // 2MB in kilobytes
            'dimensions:min_width=100,min_height=100,max_width=4096,max_height=4096',
        ];
    }

    /**
     * Get standardized error messages for image validation
     */
    public static function messages(): array
    {
        return [
            'image' => 'File phải là hình ảnh hợp lệ.',
            'mimes' => 'Hình ảnh phải có định dạng: JPEG, JPG, PNG, GIF hoặc WebP.',
            'max' => 'Kích thước file quá lớn.',
            'dimensions.min_width' => 'Hình ảnh phải có chiều rộng tối thiểu :min_width pixels.',
            'dimensions.min_height' => 'Hình ảnh phải có chiều cao tối thiểu :min_height pixels.',
            'dimensions.max_width' => 'Hình ảnh không được có chiều rộng vượt quá :max_width pixels.',
            'dimensions.max_height' => 'Hình ảnh không được có chiều cao vượt quá :max_height pixels.',
            'file' => 'Phải chọn một file hợp lệ.',
            'array' => 'Danh sách hình ảnh không hợp lệ.',
            'max_images' => 'Không được chọn quá :max hình ảnh.',
        ];
    }

    /**
     * Custom error messages with file size in MB for better UX
     */
    public static function customMessages(string $field, int $maxKB): array
    {
        $maxMB = round($maxKB / 1024, 1);

        return [
            $field.'.image' => 'File phải là hình ảnh hợp lệ.',
            $field.'.mimes' => 'Hình ảnh phải có định dạng: JPEG, JPG, PNG, GIF hoặc WebP.',
            $field.'.max' => "Kích thước file không được vượt quá {$maxMB}MB.",
            $field.'.file' => 'Phải chọn một file hợp lệ.',
            $field.'.dimensions' => 'Kích thước hình ảnh không phù hợp.',
        ];
    }

    /**
     * Validate file content using magic bytes for security
     *
     * @param  mixed  $file
     */
    public static function validateFileContent($file): bool
    {
        if (! $file || ! $file->isValid()) {
            return false;
        }

        $path = $file->getPathname();
        if (! file_exists($path)) {
            return false;
        }

        // Use ValidImageContent for consistent validation
        $validator = new ValidImageContent();
        $isValid = true;
        
        $validator->validate('file', $file, function($message) use (&$isValid) {
            $isValid = false;
        });
        
        return $isValid;
    }

    /**
     * Get file size in human readable format
     */
    public static function formatFileSize(int $bytes): string
    {
        if ($bytes >= 1048576) {
            return round($bytes / 1048576, 2).' MB';
        } elseif ($bytes >= 1024) {
            return round($bytes / 1024, 2).' KB';
        } else {
            return $bytes.' bytes';
        }
    }

    /**
     * Convert bytes to kilobytes for Laravel validation
     */
    public static function bytesToKilobytes(int $bytes): int
    {
        return (int) ceil($bytes / 1024);
    }

    /**
     * Debug validation for a specific file
     *
     * @param  mixed  $file
     */
    public static function debugFile($file): array
    {
        if (! $file) {
            return ['error' => 'No file provided'];
        }

        $debug = [
            'original_name' => $file->getClientOriginalName(),
            'size_bytes' => $file->getSize(),
            'size_kb' => self::bytesToKilobytes($file->getSize()),
            'size_formatted' => self::formatFileSize($file->getSize()),
            'mime_type' => $file->getMimeType(),
            'client_extension' => $file->getClientOriginalExtension(),
            'is_valid' => $file->isValid(),
            'error_code' => $file->getError(),
        ];

        // Additional MIME check using file content
        if ($file->isValid()) {
            $debug['content_mime'] = self::getContentMimeType($file);
            $debug['content_valid'] = self::validateFileContent($file);
        }

        return $debug;
    }

    /**
     * Get MIME type from file content
     *
     * @param  mixed  $file
     */
    private static function getContentMimeType($file): ?string
    {
        if (! $file->isValid()) {
            return null;
        }

        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file->getPathname());
        finfo_close($finfo);

        return $mimeType ?: null;
    }

    /**
     * Get allowed MIME types for validation consistency
     */
    public static function getAllowedMimeTypes(): array
    {
        return [
            'image/jpeg',
            'image/pjpeg', // IE legacy
            'image/png',
            'image/gif',
            'image/webp',
        ];
    }
}