<?php

namespace App\Policies;

use App\Models\Player;
use App\Models\User;

class PlayerPolicy
{
    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return true;
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Player $player): bool
    {
        return true;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasRole('admin') || $user->hasRole('mod');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Player $player): bool
    {
        // Admin can update all players
        if ($user->hasRole('admin')) {
            return true;
        }

        // Mods can update players (manage) but cannot delete — allow full management here
        if ($user->hasRole('mod')) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Player $player): bool
    {
        // Only admin can delete players
        return $user->hasRole('admin');
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Player $player): bool
    {
        if ($user->hasRole('admin')) {
            return true;
        }

        if ($user->hasRole('mod')) {
            return true; // mods can restore players
        }

        return false;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Player $player): bool
    {
        // Only admin can force delete
        return $user->hasRole('admin');
    }

    /**
     * Determine whether the user can approve player avatar.
     */
    public function approveAvatar(User $user, Player $player): bool
    {
        // Admin can approve all players
        if ($user->hasRole('admin')) {
            return true;
        }

        // Mod can only approve their own players
        if ($user->hasRole('mod')) {
            return true; // allow mods to approve any player avatar
        }

        return false;
    }

    /**
     * Determine whether the user can reject player avatar.
     */
    public function rejectAvatar(User $user, Player $player): bool
    {
        // Admin can reject all players
        if ($user->hasRole('admin')) {
            return true;
        }

        // Mods can reject any player avatar
        if ($user->hasRole('mod')) {
            return true;
        }

        return false;
    }
}
