<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;
use App\Models\Player;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasApiTokens, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone', 
        'location', 
        'gender',
        'avatar',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    public function player()
    {
    return $this->hasOne(Player::class);
    }
    
    // Encrypt phone
    public function getPhoneAttribute($value)
    {
        if (is_null($value) || $value === '') {
            return null;
        }

        try {
            return decrypt($value);
        } catch (\Illuminate\Contracts\Encryption\DecryptException | \Illuminate\Encryption\DecryptException $e) {
            // Nếu không thể giải mã (dữ liệu không phải payload đã mã hóa), trả về nguyên bản
            return $value;
        }
    }
    
    // Encrypt phone before saving
    public function setPhoneAttribute($value)
    {
        if (is_null($value) || $value === '') {
            $this->attributes['phone'] = null;
        } else {
            $this->attributes['phone'] = encrypt($value);
        }
    }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    /**
     * Scope để tìm kiếm tổng hợp (name, email, phone)
     * Lưu ý: Phone được mã hóa nên cần load về memory để search
     */
    public function scopeSearchAll($query, $search)
    {
        if (empty($search)) {
            return $query;
        }

        // Tìm kiếm name và email trực tiếp trong database (hiệu suất cao)
        $nameEmailIds = User::where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
                ->orWhere('email', 'like', "%{$search}%");
        })->pluck('id')->toArray();

        // Tìm kiếm phone trong memory (cần thiết vì phone được mã hóa)
        $phoneIds = [];
        if (is_numeric(str_replace([' ', '-', '+'], '', $search))) {
            // Chỉ tìm phone nếu search term có chứa số
            $usersWithPhone = User::whereNotNull('phone')->get(['id', 'phone']);
            $phoneIds = $usersWithPhone->filter(function ($user) use ($search) {
                return $user->phone && stripos($user->phone, $search) !== false;
            })->pluck('id')->toArray();
        }

        // Kết hợp IDs
        $allIds = array_unique(array_merge($nameEmailIds, $phoneIds));
        
        return $query->whereIn('id', $allIds);
    }
}