<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class Tournament extends Model
{
    use HasFactory, LogsActivity;

    protected $fillable = [
        'name', 'slug', 'start_date', 'end_date', 'description', 'location', 'type', 'max_pairs', 'fee', 'prizes', 'rules', 'image', 'password', 'referee_password', 'bracket_generated_at',
        'skill_point', 'skill_tolerance', 'venue_id', 'status', 'created_by',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'bracket_generated_at' => 'datetime',
        'skill_point' => 'double',
        'skill_tolerance' => 'double',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['name', 'start_date', 'end_date', 'location'])
            ->setDescriptionForEvent(fn (string $eventName) => 'Giải đấu đã được '.trans("activitylog.{$eventName}"))
            ->useLogName('tournament');
    }

    public function matches()
    {
        return $this->hasMany(MatchModel::class);
    }

    public function registrations()
    {
        return $this->hasMany(Registration::class);
    }

    public function venue()
    {
        return $this->belongsTo(Venue::class);
    }

    // Relationship: Tournament belongs to User (creator)
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Relationship: Tournament has many TournamentGroups
    public function groups()
    {
        return $this->hasMany(TournamentGroup::class);
    }

    // Relationship: Tournament has many TournamentRounds
    public function rounds()
    {
        return $this->hasMany(TournamentRound::class);
    }

    // In app/Models/Tournament.php
    public function groupRounds()
    {
        return $this->hasMany(TournamentRound::class)->where('type', 'round_robin');
    }

    // Accessor methods for consistent registration counting
    public function getActiveRegistrationsCountAttribute(): int
    {
        return $this->registrations()->where('status', '!=', 'rejected')->count();
    }

    // Query scope: Filter tournaments by creator (for mods)
    public function scopeOwnedBy($query, $userId)
    {
        return $query->where('created_by', $userId);
    }

    // Query scope: Filter tournaments accessible by user role
    public function scopeAccessibleBy($query, $user)
    {
        // Admin sees all tournaments
        if ($user->hasRole('admin')) {
            return $query;
        }

        // Mod only sees their own tournaments
        if ($user->hasRole('mod')) {
            return $query->where('created_by', $user->id);
        }

        // Other roles see nothing by default (or implement custom logic)
        return $query->whereNull('id'); // Returns empty result
    }

    public function getApprovedRegistrationsCountAttribute(): int
    {
        return $this->registrations()->where('status', 'approved')->count();
    }

    public function getPendingRegistrationsCountAttribute(): int
    {
        return $this->registrations()->where('status', 'pending')->count();
    }

    public function getRejectedRegistrationsCountAttribute(): int
    {
        return $this->registrations()->where('status', 'rejected')->count();
    }

    // Chia bảng đấu tự động từ registrations đã duyệt
    public function getGroups($groupCount = 4)
    {
        $pairs = $this->registrations()->where('status', 'approved')->get();
        if ($this->max_pairs && $pairs->count() > $this->max_pairs) {
            $pairs = $pairs->take($this->max_pairs);
        }
        $groups = [];
        $groupNames = range('A', chr(64 + $groupCount));
        foreach ($groupNames as $g) {
            $groups[$g] = [];
        }
        // Chia đều các bảng (round-robin)
        $idx = 0;
        foreach ($pairs as $reg) {
            $name = $this->type == 'singles'
                ? $reg->player1->name
                : ($reg->player1->name.($reg->player2 ? ' & '.$reg->player2->name : ''));
            $groups[$groupNames[$idx % $groupCount]][] = $name;
            $idx++;
        }

        return $groups;
    }

    // Render lịch thi đấu vòng tròn cho từng bảng
    public function getSchedule($groupCount = 4)
    {
        $groups = $this->getGroups($groupCount);
        $schedule = [];
        foreach ($groups as $groupName => $pairs) {
            $schedule[$groupName] = [];
            $n = count($pairs);
            for ($i = 0; $i < $n; $i++) {
                for ($j = $i + 1; $j < $n; $j++) {
                    $schedule[$groupName][] = [
                        'pair1' => $pairs[$i],
                        'pair2' => $pairs[$j],
                    ];
                }
            }
        }

        return $schedule;
    }
}
