<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Player extends Model
{
    // Quan hệ với bảng registrations
    public function registrations()
    {
        return $this->hasMany(\App\Models\Registration::class, 'player1_id');
    }

    protected $appends = ['match_history'];

    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id', 'player_id', 'name',
        'doubles_score', 'doubles_matches', 'doubles_wins', 'doubles_losses',
        'singles_score', 'singles_matches', 'singles_win', 'singles_lose',
        'max_score', 'lock_score', 'location', 'phone', 'gender',
        'avatar', 'status', 'created_by',
    ];

    public function user()
    {
        return $this->belongsTo(\App\Models\User::class);
    }

    // Relationship: Player belongs to User (creator)
    public function creator()
    {
        return $this->belongsTo(\App\Models\User::class, 'created_by');
    }

    // Query scope: Filter players by creator (for mods)
    public function scopeOwnedBy($query, $userId)
    {
        return $query->where('created_by', $userId);
    }

    // Query scope: Filter players accessible by user role
    public function scopeAccessibleBy($query, $user)
    {
        // Admin sees all players
        if ($user->hasRole('admin')) {
            return $query;
        }

        // Mod only sees their own players
        if ($user->hasRole('mod')) {
            return $query->where('created_by', $user->id);
        }

        // Other roles see nothing by default (or implement custom logic)
        return $query->whereNull('id'); // Returns empty result
    }

    public function getMatchHistoryAttribute()
    {
        // Lấy lịch sử trận đấu bằng cách join qua các cặp đấu (group_pairs)
        return \App\Models\MatchModel::with(['tournament', 'pair1.player1', 'pair1.player2', 'pair2.player1', 'pair2.player2'])
            ->where(function ($query) {
                $query->whereHas('pair1', function ($q) {
                    $q->where('player1_id', $this->id)->orWhere('player2_id', $this->id);
                })->orWhereHas('pair2', function ($q) {
                    $q->where('player1_id', $this->id)->orWhere('player2_id', $this->id);
                });
            })
            ->orderByDesc('date')
            ->get()
            ->map(function ($match) {
                // Gắn điểm đúng loại trận cho player
                if ($match->type === 'singles') {
                    $match->score_display = $match->singles_score;
                } else {
                    $match->score_display = $match->doubles_score;
                }

                return $match;
            });
    }

    // Logic cập nhật Elo (gọi sau match)
    public function updateElo(float $scoreChange)
    {
        $this->doubles_score += $scoreChange;
        // Giới hạn doubles_score không âm, không dưới minScore
        $minScore = ($this->gender === 'Nữ') ? 2.0 : 2.5;
        if ($this->doubles_score < $minScore) {
            $this->doubles_score = $minScore;
        }
        if ($this->doubles_score < 0) {
            $this->doubles_score = $minScore;
        }
        $this->doubles_matches += 1;
        $this->doubles_wins += $scoreChange > 0 ? 1 : 0;
        $this->doubles_losses += $scoreChange < 0 ? 1 : 0;
        if ($this->doubles_score > $this->max_score) {
            $this->max_score = $this->doubles_score;
        }
        $this->save();
    }
}
