<?php

namespace App\Jobs;

use App\Models\Tournament;
use App\Services\TournamentGenerationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class GenerateTournamentRoundsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 900; // 15 minutes timeout
    public $tries = 3;
    public $backoff = 60;

    protected $tournament;
    protected $userId;

    /**
     * Create a new job instance.
     */
    public function __construct(Tournament $tournament, int $userId = null)
    {
        $this->tournament = $tournament;
        $this->userId = $userId;
        
        $this->setJobStatus('queued');
    }

    /**
     * Execute the job.
     */
    public function handle(TournamentGenerationService $generationService): void
    {
        try {
            Log::info("Starting rounds generation job for tournament {$this->tournament->id}");
            
            $this->setJobStatus('processing', 'Đang tạo vòng đấu...');

            // Generate rounds
            $generationService->generateGroupRounds($this->tournament);

            $this->setJobStatus('completed', 'Đã tạo vòng đấu thành công!');
            
            Log::info("Rounds generation job completed for tournament {$this->tournament->id}");

        } catch (\Exception $e) {
            Log::error("Rounds generation job failed for tournament {$this->tournament->id}: " . $e->getMessage());
            
            $this->setJobStatus('failed', 'Lỗi: ' . $e->getMessage());
            
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("Rounds generation job permanently failed for tournament {$this->tournament->id}: " . $exception->getMessage());
        
        $this->setJobStatus('failed', 'Tạo vòng đấu thất bại: ' . $exception->getMessage());
    }

    /**
     * Set job status in cache
     */
    private function setJobStatus(string $status, string $message = ''): void
    {
        $key = "tournament_job_{$this->tournament->id}_rounds";
        
        Cache::put($key, [
            'status' => $status,
            'message' => $message,
            'progress' => $this->getProgressByStatus($status),
            'updated_at' => now()->toISOString(),
        ], 3600);
    }

    /**
     * Get progress percentage by status
     */
    private function getProgressByStatus(string $status): int
    {
        return match($status) {
            'queued' => 0,
            'processing' => 50,
            'completed' => 100,
            'failed' => 0,
            default => 0
        };
    }
}