<?php

namespace App\Jobs;

use App\Models\Tournament;
use App\Services\TournamentGenerationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class GenerateTournamentGroupsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 600; // 10 minutes timeout
    public $tries = 3; // Retry 3 times
    public $backoff = 60; // Wait 60 seconds between retries

    protected $tournament;
    protected $numberOfGroups;
    protected $userId;

    /**
     * Create a new job instance.
     */
    public function __construct(Tournament $tournament, int $numberOfGroups, int $userId = null)
    {
        $this->tournament = $tournament;
        $this->numberOfGroups = $numberOfGroups;
        $this->userId = $userId;
        
        // Set job progress
        $this->setJobStatus('queued');
    }

    /**
     * Execute the job.
     */
    public function handle(TournamentGenerationService $generationService): void
    {
        try {
            Log::info("Starting group generation job for tournament {$this->tournament->id}");
            
            $this->setJobStatus('processing', 'Đang tạo bảng đấu...');

            // Generate groups
            $generationService->generateGroups($this->tournament, $this->numberOfGroups);

            $this->setJobStatus('completed', 'Đã tạo bảng đấu thành công!');
            
            Log::info("Group generation job completed for tournament {$this->tournament->id}");

        } catch (\Exception $e) {
            Log::error("Group generation job failed for tournament {$this->tournament->id}: " . $e->getMessage());
            
            $this->setJobStatus('failed', 'Lỗi: ' . $e->getMessage());
            
            throw $e; // Re-throw to trigger retry mechanism
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("Group generation job permanently failed for tournament {$this->tournament->id}: " . $exception->getMessage());
        
        $this->setJobStatus('failed', 'Tạo bảng đấu thất bại: ' . $exception->getMessage());
    }

    /**
     * Set job status in cache for frontend tracking
     */
    private function setJobStatus(string $status, string $message = ''): void
    {
        $key = "tournament_job_{$this->tournament->id}_groups";
        
        Cache::put($key, [
            'status' => $status,
            'message' => $message,
            'progress' => $this->getProgressByStatus($status),
            'updated_at' => now()->toISOString(),
        ], 3600); // Cache for 1 hour
    }

    /**
     * Get progress percentage by status
     */
    private function getProgressByStatus(string $status): int
    {
        return match($status) {
            'queued' => 0,
            'processing' => 50,
            'completed' => 100,
            'failed' => 0,
            default => 0
        };
    }
}