<?php

namespace App\Jobs;

use App\Models\Tournament;
use App\Services\TournamentGenerationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class GeneratePlayoffsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 600; // 10 minutes timeout
    public $tries = 3;
    public $backoff = 60;

    protected $tournament;
    protected $teamsToAdvance;
    protected $userId;

    /**
     * Create a new job instance.
     */
    public function __construct(Tournament $tournament, int $teamsToAdvance = 2, int $userId = null)
    {
        $this->tournament = $tournament;
        $this->teamsToAdvance = $teamsToAdvance;
        $this->userId = $userId;
        
        $this->setJobStatus('queued');
    }

    /**
     * Execute the job.
     */
    public function handle(TournamentGenerationService $generationService): void
    {
        try {
            Log::info("Starting playoffs generation job for tournament {$this->tournament->id}");
            
            $this->setJobStatus('processing', 'Đang tạo vòng playoffs...');

            // Generate playoffs
            $roundName = $generationService->generatePlayoffsImproved($this->tournament, $this->teamsToAdvance);

            $this->setJobStatus('completed', "Đã tạo thành công {$roundName}!");
            
            Log::info("Playoffs generation job completed for tournament {$this->tournament->id}");

        } catch (\Exception $e) {
            Log::error("Playoffs generation job failed for tournament {$this->tournament->id}: " . $e->getMessage());
            
            $this->setJobStatus('failed', 'Lỗi: ' . $e->getMessage());
            
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("Playoffs generation job permanently failed for tournament {$this->tournament->id}: " . $exception->getMessage());
        
        $this->setJobStatus('failed', 'Tạo playoffs thất bại: ' . $exception->getMessage());
    }

    /**
     * Set job status in cache
     */
    private function setJobStatus(string $status, string $message = ''): void
    {
        $key = "tournament_job_{$this->tournament->id}_playoffs";
        
        Cache::put($key, [
            'status' => $status,
            'message' => $message,
            'progress' => $this->getProgressByStatus($status),
            'updated_at' => now()->toISOString(),
        ], 3600);
    }

    /**
     * Get progress percentage by status
     */
    private function getProgressByStatus(string $status): int
    {
        return match($status) {
            'queued' => 0,
            'processing' => 50,
            'completed' => 100,
            'failed' => 0,
            default => 0
        };
    }
}