<?php
namespace App\Jobs;

use App\Models\Player;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Carbon\Carbon;

class ApprovePendingAvatarsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function handle()
    {
        $now = Carbon::now();
        $players = Player::whereNotNull('avatar_pending')
            ->whereNotNull('avatar_pending_requested_at')
            ->where('avatar_pending_requested_at', '<=', $now->subHours(24))
            ->get();
            
        foreach ($players as $player) {
            try {
                $pendingPath = $player->avatar_pending;
                $pendingFilePath = public_path($pendingPath);
                
                // Check if pending file exists
                if (!file_exists($pendingFilePath)) {
                    // Clean up DB if file doesn't exist
                    $player->update([
                        'avatar_pending' => null,
                        'avatar_pending_requested_at' => null,
                    ]);
                    continue;
                }

                // Create new filename for approved avatar
                $extension = pathinfo($pendingFilePath, PATHINFO_EXTENSION);
                $newFilename = 'player_'.$player->id.'_'.time().'.'.$extension;
                $newPath = 'assets/avatars/'.$newFilename;
                $newFilePath = public_path($newPath);

                // Move file from pending to main avatar folder
                if (rename($pendingFilePath, $newFilePath)) {
                    // Delete old avatar if exists
                    if ($player->avatar) {
                        $oldAvatarPath = public_path($player->avatar);
                        if (file_exists($oldAvatarPath)) {
                            @unlink($oldAvatarPath);
                        }
                    }

                    // Update player record
                    $player->avatar = $newPath;
                    $player->avatar_pending = null;
                    $player->avatar_pending_requested_at = null;
                    $player->save();

                    // Update user record if linked
                    if ($player->user) {
                        $player->user->avatar = $newPath;
                        $player->user->save();
                    }
                }
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error("Failed to auto-approve avatar for player #{$player->id}: ".$e->getMessage());
            }
        }
    }
}