<?php

namespace App\Http\Requests;

use App\Services\RegistrationService;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;

class StoreRegistrationRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true; // Anyone can register for tournaments
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $tournament = $this->route('tournament');

        return [
            'player1_phone' => 'required|exists:players,phone',
            'player2_phone' => $tournament && $tournament->type === 'doubles'
                ? 'required|exists:players,phone|different:player1_phone'
                : 'nullable',
            'note' => 'nullable|string|max:255',
        ];
    }

    /**
     * Get custom error messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'player1_phone.required' => 'Vui lòng nhập số điện thoại người chơi 1',
            'player1_phone.exists' => 'Người chơi 1 không tồn tại trong hệ thống',
            'player2_phone.required' => 'Vui lòng nhập số điện thoại người chơi 2',
            'player2_phone.exists' => 'Người chơi 2 không tồn tại trong hệ thống',
            'player2_phone.different' => 'Hai người chơi không được trùng nhau',
            'note.max' => 'Ghi chú không được vượt quá 255 ký tự',
        ];
    }

    /**
     * Configure the validator instance.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $tournament = $this->route('tournament');

            if ($tournament && ! $validator->errors()->any()) {
                try {
                    // Use RegistrationService for business logic validation
                    $registrationService = app(RegistrationService::class);
                    $registrationService->validateRegistration($tournament, $this->validated());
                } catch (ValidationException $e) {
                    // Add validation errors from the service
                    foreach ($e->errors() as $field => $messages) {
                        foreach ($messages as $message) {
                            $validator->errors()->add($field, $message);
                        }
                    }
                }
            }
        });
    }
}
