<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use App\Rules\ImageValidationRules;
use App\Rules\ValidImageContent;

class MatchesRequest extends FormRequest
{
    public function authorize(): bool
    {
        return auth()->user()->hasRole('admin') || auth()->user()->hasRole('mod');
    }

    public function rules(): array
    {
        return [
            'date' => ['required', 'date'],
            'type' => ['required', Rule::in(['tournament'])],
            'result' => ['required', Rule::in(['win', 'loss'])],
            'score' => ['required', 'string', 'max:50', 'regex:/^\d-\d\s+\d+-\d+$/'], // e.g., "1-0 15-8"
            'player1_id' => ['required', Rule::exists('players', 'id')],
            'player2_id' => ['required', Rule::exists('players', 'id'), 'different:player1_id'],
            'opponent1_id' => [
                'required',
                Rule::exists('players', 'id'),
                'different:player1_id',
                'different:player2_id'
            ],
            'opponent2_id' => [
                'required',
                Rule::exists('players', 'id'),
                'different:player1_id',
                'different:player2_id',
                'different:opponent1_id'
            ],
            'images' => ImageValidationRules::matchImages(),
            'images.*' => array_merge(ImageValidationRules::matchImage(), [new ValidImageContent()]),
        ];
    }

    public function messages(): array
    {
        return array_merge([
            'score.regex' => 'Score must be in the format "X-Y A-B" (e.g., "1-0 15-8").',
            'player1_id.exists' => 'Player 1 does not exist.',
            'player2_id.different' => 'Player 2 must be different from Player 1.',
            'opponent1_id.different' => 'Opponent 1 must be different from both players.',
            'opponent2_id.different' => 'Opponent 2 must be different from both players and Opponent 1.',
            'images.array' => 'Images must be an array.',
            'images.max' => 'Cannot upload more than 5 images per match.',
        ], ImageValidationRules::customMessages('images.*', 2048));
    }
}