<?php

namespace App\Http\Requests\Admin;

use Illuminate\Foundation\Http\FormRequest;
use App\Rules\ImageValidationRules;
use App\Rules\ValidImageContent;

class UpdateTournamentRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'name' => ['bail', 'required', 'string', 'max:255'],
            'description' => ['bail', 'required', 'string', 'min:10', 'max:10000'],
            'location' => ['bail', 'required', 'string', 'max:255'],
            'type' => ['bail', 'required', 'in:singles,doubles'],
            'start_date' => ['bail', 'required', 'date'], // cho phép chỉnh về quá khứ khi cập nhật
            'end_date' => ['bail', 'required', 'date', 'after_or_equal:start_date'],
            'skill_point' => ['bail', 'required', 'numeric', 'min:0', 'max:15'],
            'skill_tolerance' => ['bail', 'required', 'numeric', 'min:0', 'max:5'],
            'max_pairs' => ['bail', 'required', 'integer', 'min:4', 'max:128'],
            'fee' => ['bail', 'required', 'numeric', 'min:0'],
            'prizes' => ['bail', 'required', 'string', 'max:255'],
            'rules' => ['bail', 'required', 'string', 'min:10', 'max:20000'],
            'image' => array_merge(ImageValidationRules::tournamentImage(), [new ValidImageContent]),
            'password' => ['sometimes'], // không cho đổi qua form này
            'referee_password' => ['nullable', 'digits:6'],
        ];
    }

    public function attributes(): array
    {
        return [
            'name' => 'Tên giải đấu',
            'description' => 'Mô tả giải đấu',
            'location' => 'Địa điểm',
            'type' => 'Loại giải đấu',
            'start_date' => 'Ngày bắt đầu',
            'end_date' => 'Ngày kết thúc',
            'skill_point' => 'Điểm trình',
            'skill_tolerance' => 'Dung sai điểm trình',
            'max_pairs' => 'Số cặp/VĐV tối đa',
            'fee' => 'Lệ phí',
            'prizes' => 'Giải thưởng',
            'rules' => 'Thể lệ giải đấu',
            'image' => 'Hình ảnh giải đấu',
            'password' => 'Mật khẩu quản lý',
            'referee_password' => 'Mật khẩu trọng tài',
        ];
    }

    public function messages(): array
    {
        return array_merge([
            'required' => ':attribute là bắt buộc.',
            'string' => ':attribute phải là chuỗi ký tự hợp lệ.',
            'max' => ':attribute không được vượt quá :max ký tự.',
            'max.string' => ':attribute không được vượt quá :max ký tự.',
            'in' => ':attribute không hợp lệ.',
            'date' => ':attribute không đúng định dạng ngày.',
            'after_or_equal' => ':attribute phải sau hoặc bằng :date.',
            'numeric' => ':attribute phải là số.',
            'integer' => ':attribute phải là số nguyên.',
            'min' => ':attribute phải lớn hơn hoặc bằng :min.',
            'min.string' => ':attribute phải có ít nhất :min ký tự.',
            'max.numeric' => ':attribute không được lớn hơn :max.',
            'digits' => ':attribute phải gồm đúng :digits chữ số.',
        ], ImageValidationRules::customMessages('image', 2048));
    }

    protected function prepareForValidation(): void
    {
        $normalizeDecimal = function ($value) {
            if ($value === null || $value === '') return $value;
            $value = str_replace([' ', ','], ['', '.'], (string) $value);
            return is_numeric($value) ? $value : $value;
        };

        $this->merge([
            'skill_point' => $normalizeDecimal($this->input('skill_point')),
            'skill_tolerance' => $normalizeDecimal($this->input('skill_tolerance')),
            'fee' => $normalizeDecimal($this->input('fee')),
        ]);

        foreach (['name','location','prizes'] as $field) {
            if ($this->has($field)) {
                $this->merge([$field => trim((string) $this->input($field))]);
            }
        }
    }
}