<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class RuntimeMemoryManager
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Get memory config
        $memoryConfig = config('memory');
        
        // Determine memory limit based on route
        $memoryLimit = $this->getMemoryLimitForRoute($request, $memoryConfig);
        
        // Set PHP configuration
        ini_set('memory_limit', $memoryLimit);
        ini_set('max_execution_time', $memoryConfig['max_execution_time']);
        
        // Set additional PHP settings for optimization
        ini_set('max_input_vars', '10000');
        ini_set('post_max_size', '512M');
        ini_set('upload_max_filesize', '256M');
        
        // Force garbage collection before heavy operations
        if ($this->isHeavyOperation($request)) {
            gc_collect_cycles();
            
            // Set even higher limits for tournament operations
            ini_set('memory_limit', $memoryConfig['tournament_operations']);
            ini_set('max_execution_time', $memoryConfig['tournament_generation_time']);
        }
        
        return $next($request);
    }
    
    private function getMemoryLimitForRoute(Request $request, array $config): string
    {
        $route = $request->route();
        if (!$route) {
            return $config['default_limit'];
        }
        
        $routeName = $route->getName() ?? '';
        $uri = $request->getRequestUri();
        
        // Tournament generation routes
        if (str_contains($uri, 'tournament') && str_contains($uri, 'generate')) {
            return $config['tournament_operations'];
        }
        
        // Export routes
        if (str_contains($uri, 'export') || str_contains($routeName, 'export')) {
            return $config['export_operations'];
        }
        
        // Tournament setting/health check routes
        if (str_contains($uri, 'tournament-setting') || str_contains($uri, 'health-check')) {
            return $config['tournament_operations'];
        }
        
        return $config['general_operations'];
    }
    
    private function isHeavyOperation(Request $request): bool
    {
        $uri = $request->getRequestUri();
        $heavyRoutes = [
            'tournament-setting',
            'generate',
            'health-check',
            'standings',
            'export',
            'pairs'
        ];
        
        foreach ($heavyRoutes as $route) {
            if (str_contains($uri, $route)) {
                return true;
            }
        }
        
        return false;
    }
}