<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class OptimizeStaticAssets
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Only apply to static assets
        if ($this->isStaticAsset($request)) {
            $this->addCacheHeaders($response);
            $this->addCompressionHeaders($response);
            $this->addSecurityHeaders($response);
        }

        return $response;
    }

    /**
     * Check if request is for static asset
     */
    private function isStaticAsset(Request $request): bool
    {
        $path = $request->path();

        // Check for asset paths
        if (str_starts_with($path, 'assets/')) {
            return true;
        }

        // Check for file extensions
        $extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'css', 'js', 'woff', 'woff2', 'ttf', 'eot'];
        $extension = pathinfo($path, PATHINFO_EXTENSION);

        return in_array(strtolower($extension), $extensions);
    }

    /**
     * Add cache headers for better performance
     */
    private function addCacheHeaders(Response $response): void
    {
        $maxAge = 31536000; // 1 year for static assets

        $response->headers->set('Cache-Control', "public, max-age={$maxAge}, immutable");
        $response->headers->set('Expires', gmdate('D, d M Y H:i:s', time() + $maxAge).' GMT');

        // ETag for cache validation
        $etag = md5($response->getContent());
        $response->headers->set('ETag', '"'.$etag.'"');

        // Last-Modified
        $response->headers->set('Last-Modified', gmdate('D, d M Y H:i:s', filemtime($_SERVER['SCRIPT_FILENAME'])).' GMT');
    }

    /**
     * Add compression hints
     */
    private function addCompressionHeaders(Response $response): void
    {
        // Tell browsers we accept various compressions
        $response->headers->set('Vary', 'Accept-Encoding');

        // Add content type optimization hints
        $contentType = $response->headers->get('Content-Type');

        if (str_contains($contentType, 'image/')) {
            // Suggest modern image formats
            $response->headers->set('Accept-CH', 'Viewport-Width, Width');
        }
    }

    /**
     * Add security headers for assets
     */
    private function addSecurityHeaders(Response $response): void
    {
        // Prevent content sniffing
        $response->headers->set('X-Content-Type-Options', 'nosniff');

        // Cross-origin resource sharing for fonts and images
        $response->headers->set('Access-Control-Allow-Origin', '*');
        $response->headers->set('Cross-Origin-Resource-Policy', 'cross-origin');
    }
}
