<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class AggressiveMemoryManagement
{
    /**
     * Handle an incoming request with aggressive memory management
     */
    public function handle(Request $request, Closure $next)
    {
        // Set aggressive memory settings
        ini_set('memory_limit', '1G'); // Increase to 1GB
        ini_set('max_execution_time', 600); // 10 minutes
        
        // Clear opcache if available
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }
        
        // Force garbage collection at start
        gc_collect_cycles();
        
        $initialMemory = memory_get_usage(true);
        $peakMemory = memory_get_peak_usage(true);
        
        Log::info('Request started', [
            'url' => $request->url(),
            'method' => $request->method(),
            'initial_memory_mb' => round($initialMemory / 1024 / 1024, 2),
            'peak_memory_mb' => round($peakMemory / 1024 / 1024, 2)
        ]);

        // Disable query logging to save memory
        DB::connection()->disableQueryLog();
        
        try {
            $response = $next($request);
        } catch (\Exception $e) {
            $this->handleMemoryError($e, $request);
            throw $e;
        }
        
        // Aggressive cleanup after request
        $this->performAggressiveCleanup();
        
        $finalMemory = memory_get_usage(true);
        $finalPeak = memory_get_peak_usage(true);
        
        Log::info('Request completed', [
            'url' => $request->url(),
            'final_memory_mb' => round($finalMemory / 1024 / 1024, 2),
            'peak_memory_mb' => round($finalPeak / 1024 / 1024, 2),
            'memory_increase_mb' => round(($finalMemory - $initialMemory) / 1024 / 1024, 2)
        ]);
        
        return $response;
    }
    
    /**
     * Handle memory-related errors
     */
    private function handleMemoryError(\Exception $e, Request $request): void
    {
        if (strpos($e->getMessage(), 'memory') !== false || 
            strpos($e->getMessage(), 'exhausted') !== false) {
            
            Log::critical('Memory exhaustion detected', [
                'url' => $request->url(),
                'method' => $request->method(),
                'memory_usage_mb' => round(memory_get_usage(true) / 1024 / 1024, 2),
                'memory_peak_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2),
                'error' => $e->getMessage()
            ]);
            
            // Emergency cleanup
            $this->emergencyCleanup();
        }
    }
    
    /**
     * Perform aggressive memory cleanup
     */
    private function performAggressiveCleanup(): void
    {
        // Clear all possible caches
        if (class_exists('\Illuminate\Support\Facades\Cache')) {
            try {
                cache()->flush();
            } catch (\Exception $e) {
                // Ignore cache errors during cleanup
            }
        }
        
        // Clear session data if not needed
        if (session()->isStarted() && !session()->has('user_id')) {
            session()->flush();
        }
        
        // Force garbage collection
        gc_collect_cycles();
        
        // Clear any static variables in Laravel
        if (class_exists('\Illuminate\Support\Facades\DB')) {
            DB::connection()->getPdo()->setAttribute(\PDO::ATTR_EMULATE_PREPARES, true);
        }
    }
    
    /**
     * Emergency cleanup when memory is critically low
     */
    private function emergencyCleanup(): void
    {
        // Clear everything possible
        if (function_exists('gc_collect_cycles')) {
            for ($i = 0; $i < 3; $i++) {
                gc_collect_cycles();
            }
        }
        
        // Clear opcache if available
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }
        
        // Disable query logging
        if (class_exists('\Illuminate\Support\Facades\DB')) {
            DB::connection()->disableQueryLog();
        }
    }
}