<?php

namespace App\Http\Controllers;

use App\Models\Venue;
use Illuminate\Http\Request;

class VenueController extends Controller
{
    public function index(Request $request)
    {
        $query = $this->buildVenueQuery($request);
        $venues = $query->paginate(12);
        $locations = Venue::select('location')->distinct()->orderBy('location')->pluck('location');

        return view('venues.index', compact('venues', 'locations'));
    }

    public function search(Request $request)
    {
        try {
            $query = $this->buildVenueQuery($request);
            $venues = $query->paginate(12);

            return view('venues.partials._venues_grid', compact('venues'))->render();
        } catch (\Exception $e) {
            \Log::error('Public venues search error: ' . $e->getMessage(), [
                'search' => $request->get('search'),
                'location' => $request->get('location'),
                'trace' => $e->getTraceAsString()
            ]);
            
            if ($request->ajax()) {
                return response()->json([
                    'error' => 'Có lỗi xảy ra khi tìm kiếm: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Có lỗi xảy ra khi tìm kiếm.');
        }
    }

    private function buildVenueQuery(Request $request)
    {
        $query = Venue::query();

        // Xử lý tìm kiếm
        if ($request->filled('search')) {
            $search = trim($request->search);
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%')
                    ->orWhere('address', 'like', '%'.$search.'%')
                    ->orWhere('location', 'like', '%'.$search.'%')
                    ->orWhere('contact_phone', 'like', '%'.$search.'%')
                    ->orWhere('contact_email', 'like', '%'.$search.'%');
            });
        }

        // Xử lý lọc theo địa điểm
        if ($request->filled('location')) {
            $query->where('location', $request->location);
        }

        // Sắp xếp mặc định theo ngày tạo mới nhất
        $query->latest();

        return $query;
    }

    /**
     * Display the specified resource.
     */
    public function show(Venue $venue)
    {
        // Eager load reviews and the user for each review
        $venue->load(['reviews.user']);

        // Laravel sẽ tự động tìm Venue dựa trên ID hoặc slug từ URL
        return view('venues.show', compact('venue'));
    }
}
