<?php

namespace App\Http\Controllers;

use App\Exports\RegistrationsExport;
use App\Exports\ScheduleExport;
use App\Exports\StandingsExport;
use App\Exports\PairsExport;
use App\Models\GroupPair;
use App\Models\MatchModel as Matches;
use App\Models\Player;
use App\Models\Registration;
use App\Models\Tournament;
use App\Models\TournamentGroup;
use App\Models\TournamentRound;
use App\Services\EloService;
use App\Services\RegistrationService;
use App\Services\StandingsService;
use App\Services\TournamentAuthService;
use App\Services\TournamentCacheService;
use App\Services\TournamentGenerationService;
use App\Services\TournamentStateService;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class TournamentSettingController extends Controller
{
    protected $generationService;

    protected $standingsService;

    protected $eloService;

    protected $registrationService;

    protected $stateService;

    protected $authService;

    protected $cacheService;

    public function __construct(
        TournamentGenerationService $generationService,
        StandingsService $standingsService,
        EloService $eloService,
        RegistrationService $registrationService,
        TournamentStateService $stateService,
        TournamentAuthService $authService,
        TournamentCacheService $cacheService
    ) {
        $this->generationService = $generationService;
        $this->standingsService = $standingsService;
        $this->eloService = $eloService;
        $this->registrationService = $registrationService;
        $this->stateService = $stateService;
        $this->authService = $authService;
        $this->cacheService = $cacheService;
    }

    // Trang setting: hiển thị giao diện quản lý
    public function index(Request $request, $tournamentId)
    {
        // Kiểm tra session xác thực cho cả ban tổ chức và trọng tài
        $sessionKeyOrg = 'tournament_pw_'.$tournamentId;
        $sessionKeyRole = 'tournament_role_'.$tournamentId;
        $role = session()->get($sessionKeyRole);
        $isOrganizer = session()->get($sessionKeyOrg);
        $isReferee = $role === 'referee';
        if (! $isOrganizer && ! $isReferee) {
            // Nếu chưa xác thực, trả về view nhập password
            return view('tournament_setting_password', ['tournamentId' => $tournamentId]);
        }
        // Load tournament with relations similar to admin show
        $tournament = Tournament::findOrFail($tournamentId);
        $tournament->load([
            'groups.pairs.player1',
            'groups.pairs.player2',
            'rounds' => fn ($query) => $query->orderBy('created_at'),
            'rounds.matches' => fn ($query) => $query->with([
                'pair1.player1', 'pair1.player2', 'pair2.player1', 'pair2.player2',
            ]),
            'registrations', // Tải tất cả các đăng ký
        ]);

        $playoffRounds = $tournament->rounds->where('type', 'playoff');

        // Find different types of playoff rounds using both playoff_level and name matching
        $roundOf16 = $playoffRounds->first(function ($round) {
            return $round->playoff_level === 'round_of_16' || str_contains($round->name, 'Vòng 1/16');
        });
        $quarterFinals = $playoffRounds->first(function ($round) {
            return $round->playoff_level === 'quarter_final' || str_contains($round->name, 'Tứ kết');
        });
        $semiFinals = $playoffRounds->first(function ($round) {
            return $round->playoff_level === 'semi_final' || str_contains($round->name, 'Bán kết');
        });
        $final = $playoffRounds->first(function ($round) {
            return $round->playoff_level === 'final' || str_contains($round->name, 'Chung kết');
        });
        $thirdPlace = $playoffRounds->first(function ($round) {
            return $round->playoff_level === 'third_place' || str_contains($round->name, 'Tranh Hạng Ba');
        });

        $standingsByGroup = [];
        foreach ($tournament->groups as $group) {
            $standingsByGroup[$group->name] = $this->standingsService->calculateForGroup($group, $tournament->rounds);
        }

        // Organize playoff rounds for the view
        $playoffRounds = $this->organizePlayoffRounds($tournament->rounds->where('type', 'playoff'));

        // Lấy trạng thái và thống kê giải đấu
        $tournamentState = $this->stateService->getCurrentPhase($tournament);
        $tournamentStats = $this->stateService->getTournamentStats($tournament);
        $validation = $this->stateService->validateTournamentState($tournament);
        $recommendedActions = $this->stateService->getRecommendedActions($tournament);
        $userRole = $this->getUserTournamentRole($tournament);

        return view('admin.tournaments.show', compact(
            'tournament',
            'standingsByGroup',
            'roundOf16',
            'quarterFinals',
            'semiFinals',
            'final',
            'thirdPlace',
            'playoffRounds',
            'tournamentState',
            'tournamentStats',
            'validation',
            'recommendedActions',
            'userRole'
        ));
    }

    // Xác thực mật khẩu tournament
    public function checkPassword(Request $request, $tournamentId)
    {
        $tournament = Tournament::findOrFail($tournamentId);
        $password = $request->input('password');
        $userType = $request->input('user_type', 'auto');

        $result = $this->authService->authenticate($tournament, $password, $userType);

        if ($result['success']) {
            if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
                return response()->json($result);
            }

            return redirect()->intended(route('tournament.setting.index', ['tournament' => $tournamentId]))
                ->with('success', $result['message']);
        }

        if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
            return response()->json(['error' => $result['message']], 401);
        }

        return back()->with('error', $result['message']);
    }

    /**
     * Helper to ensure the current session is authorized for this tournament.
     * Throws a 403 JSON response for API callers if unauthorized.
     */
    protected function ensureTournamentSession($tournamentId)
    {
        $tournament = Tournament::findOrFail($tournamentId);

        if (! $this->authService->isAuthenticated($tournament)) {
            if (request()->expectsJson() || request()->isJson() || request()->wantsJson()) {
                throw new HttpResponseException(response()->json(['error' => 'Authentication required'], 403));
            }

            session()->put('url.intended', url()->current());
            throw new HttpResponseException(redirect()->route('tournament.setting.index', ['tournament' => $tournamentId]));
        }
    }

    // Đăng xuất khỏi tournament-setting
    public function logout(Request $request)
    {
        $tournamentId = $request->input('tournamentId');
        if ($tournamentId) {
            session()->forget('tournament_pw_'.$tournamentId);
            session()->forget('tournament_role_'.$tournamentId);
            if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
                return response()->json(['success' => true]);
            }

            return redirect()->route('tournament.setting.index', ['tournament' => $tournamentId]);
        }
        if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
            return response()->json(['error' => 'Missing tournamentId'], 400);
        }

        return back()->with('error', 'Missing tournamentId');
    }

    /**
     * Show the registration management page for a tournament (Organizer).
     *
     * @return \Illuminate\View\View
     */
    public function registrations(Tournament $tournament)
    {
        $this->ensureTournamentSession($tournament->id);
        $registrations = $tournament->registrations()
            ->with(['player1', 'player2'])
            ->latest()
            ->paginate(20);

        $stats = [
            'total' => $tournament->registrations->count(),
            'approved' => $tournament->registrations->where('status', 'approved')->count(),
            'pending' => $tournament->registrations->where('status', 'pending')->count(),
            'rejected' => $tournament->registrations->where('status', 'rejected')->count(),
        ];

        return view('admin.tournaments.registrations', compact('tournament', 'registrations', 'stats'));
    }

    /**
     * Update the status of a registration (Organizer).
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateRegistrationStatus(Request $request, Registration $registration)
    {
        $this->ensureTournamentSession($registration->tournament_id);

        $data = $request->validate([
            'status' => 'required|in:approved,rejected,pending',
            'reason' => 'nullable|string|max:255',
        ]);

        try {
            $this->registrationService->updateStatus($registration, $data['status'], $data['reason'] ?? null);

            return back()->with('success', 'Cập nhật trạng thái đăng ký thành công!');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    // Lấy danh sách bảng đấu
    public function getGroups($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $groups = TournamentGroup::where('tournament_id', $tournamentId)
            ->with(['pairs.player1', 'pairs.player2'])
            ->get();

        return response()->json($groups);
    }

    // Thêm bảng đấu
    public function addGroup(Request $request, $tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $request->validate([
            'name' => 'required|string|max:255',
        ]);
        $group = TournamentGroup::create([
            'tournament_id' => $tournamentId,
            'name' => $request->input('name'),
        ]);

        // If the caller expects JSON (AJAX), return JSON as before.
        if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
            return response()->json($group);
        }

        // For normal browser form submissions, redirect back to the tournament setting
        // page and request the UI open the "Bảng Đấu & Cặp Đấu" tab.
        return redirect()
            ->route('tournament.setting.index', ['tournament' => $tournamentId])
            ->with('success', 'Đã thêm bảng thành công.')
            ->with('open_tab', 'groups-pairs');
    }

    // Xóa bảng đấu
    public function deleteGroup(Request $request, $groupId)
    {
        // Accept either POST with _method=DELETE (admin-style) or plain POST from organizer UI
        $isDelete = $request->input('_method') === 'DELETE' || $request->isMethod('post');
        if ($isDelete) {
            $group = TournamentGroup::findOrFail($groupId);
            // Ensure session authorized for this tournament
            $this->ensureTournamentSession($group->tournament_id);

            $group->delete();

            // For browser requests, redirect back to tournament setting UI
            if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
                return response()->json(['success' => true]);
            }

            return redirect()->route('tournament.setting.index', ['tournament' => $group->tournament_id])->with('success', 'Đã xóa bảng thành công.')->with('open_tab', 'groups-pairs');
        }

        return response()->json(['error' => 'Invalid method'], 405);
    }

    // Thêm cặp vào bảng (chỉ lấy player đã đăng ký và duyệt)
    // Route: POST /{tournament}/groups/{group}/pairs
    public function addPair(Request $request, Tournament $tournament, TournamentGroup $group)
    {
        // Ensure organizer session is for this tournament before doing validation
        $this->ensureTournamentSession($tournament->id);

        // Validate inputs similarly to admin store
        $data = $request->validate([
            'player1_id' => 'required|exists:players,id|different:player2_id',
            'player2_id' => 'required|exists:players,id',
        ], [
            'player1_id.different' => 'Hai vận động viên không được trùng nhau.',
        ]);

        // Prevent adding a pair that already exists in any group of this tournament
        $player1_id = $data['player1_id'];
        $player2_id = $data['player2_id'];

        $existingPair = GroupPair::whereHas('group', function ($query) use ($tournament) {
            $query->where('tournament_id', $tournament->id);
        })->where(function ($query) use ($player1_id, $player2_id) {
            $query->where(function ($q) use ($player1_id, $player2_id) {
                $q->where('player1_id', $player1_id)->where('player2_id', $player2_id);
            })->orWhere(function ($q) use ($player1_id, $player2_id) {
                $q->where('player1_id', $player2_id)->where('player2_id', $player1_id);
            });
        })->first();

        if ($existingPair) {
            // Redirect back to organizer create form with error
            return redirect()->route('tournament.setting.groups.pairs.create', ['tournament' => $tournament->id, 'group' => $group->id])
                ->withErrors(['player1_id' => 'Cặp đấu này đã tồn tại ở Bảng '.$existingPair->group->name.'.'])
                ->withInput();
        }

        $pair = GroupPair::create([
            'group_id' => $group->id,
            'player1_id' => $player1_id,
            'player2_id' => $player2_id,
        ]);
        $pair->load(['player1', 'player2']);

        // If caller expects JSON (AJAX), return JSON
        if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
            return response()->json($pair);
        }

        // Otherwise redirect back to the tournament setting UI (groups tab)
        return redirect()
            ->route('tournament.setting.index', ['tournament' => $group->tournament_id])
            ->with('success', 'Đã thêm cặp đấu thành công.')->with('open_tab', 'groups-pairs');
    }

    // Organizer: show edit form for a pair
    public function editPair($tournamentId, $groupId, $pairId)
    {
        $this->ensureTournamentSession($tournamentId);

        $group = TournamentGroup::findOrFail($groupId);
        $pair = GroupPair::findOrFail($pairId);

        // Gather available players similar to admin edit logic
        $registeredPlayers = \Illuminate\Support\Facades\DB::table('registrations')
            ->where('tournament_id', $group->tournament_id)
            ->where('status', 'approved')
            ->select('player1_id', 'player2_id')
            ->get();
        $registeredPlayerIds = $registeredPlayers->pluck('player1_id')->merge($registeredPlayers->pluck('player2_id'))->unique()->filter();

        $otherPaired = \Illuminate\Support\Facades\DB::table('group_pairs')
            ->join('tournament_groups', 'group_pairs.group_id', '=', 'tournament_groups.id')
            ->where('tournament_groups.tournament_id', $group->tournament_id)
            ->where('group_pairs.id', '!=', $pair->id)
            ->select('group_pairs.player1_id', 'group_pairs.player2_id')
            ->get()
            ->pluck('player1_id')->merge(\Illuminate\Support\Facades\DB::table('group_pairs')->where('group_pairs.id', '!=', $pair->id)->pluck('player2_id'))
            ->unique()->filter();

        $players = Player::whereIn('id', $registeredPlayerIds)->whereNotIn('id', $otherPaired)->get();
        $players = $players->merge(Player::whereIn('id', [$pair->player1_id, $pair->player2_id])->get())->unique('id');

        // Reuse admin edit view but set organizer form action/method and cancel URL
        $formAction = route('tournament.setting.groups.pairs.update', ['tournament' => $group->tournament_id, 'group' => $group->id, 'pair' => $pair->id]);
        $formMethod = 'POST';
        $cancelUrl = route('tournament.setting.index', ['tournament' => $group->tournament_id]);

        return view('admin.pairs.edit', compact('group', 'pair', 'players', 'formAction', 'formMethod', 'cancelUrl'))->with('open_tab', 'groups-pairs');
    }

    // Organizer: update a pair
    public function updatePair(Request $request, $tournamentId, $groupId, $pairId)
    {
        $this->ensureTournamentSession($tournamentId);

        $group = TournamentGroup::findOrFail($groupId);
        $pair = GroupPair::findOrFail($pairId);

        $data = $request->validate([
            'player1_id' => 'required|exists:players,id|different:player2_id',
            'player2_id' => 'required|exists:players,id',
        ]);

        $pair->update($data);

        // Redirect back to tournament setting UI
        return redirect()->route('tournament.setting.index', ['tournament' => $group->tournament_id])->with('success', 'Cập nhật cặp đấu thành công!')->with('open_tab', 'groups-pairs');
    }

    // Show the Add Pair form for organizers (reuse admin logic/view)
    public function showAddPairForm($tournamentId, $groupId)
    {
        $this->ensureTournamentSession($tournamentId);

        // Reuse the admin GroupPairController::create logic to gather available registrations
        // We'll replicate the necessary parts here to avoid cross-controller calls.
        $group = TournamentGroup::findOrFail($groupId);

        $registrations = \Illuminate\Support\Facades\DB::table('registrations')
            ->where('tournament_id', $group->tournament_id)
            ->where('status', 'approved')
            ->select('id', 'player1_id', 'player2_id')
            ->get();

        $existingPairs = \Illuminate\Support\Facades\DB::table('group_pairs')
            ->join('tournament_groups', 'group_pairs.group_id', '=', 'tournament_groups.id')
            ->where('tournament_groups.tournament_id', $group->tournament_id)
            ->select('group_pairs.player1_id', 'group_pairs.player2_id')
            ->get();

        $existingPairKeys = $existingPairs->map(function ($p) {
            $a = (int) $p->player1_id;
            $b = (int) $p->player2_id;
            if ($a > $b) {
                [$a, $b] = [$b, $a];
            }

            return $a.'-'.$b;
        })->unique()->toArray();

        $availableRegistrations = $registrations->filter(function ($r) use ($existingPairKeys) {
            $a = (int) $r->player1_id;
            $b = (int) $r->player2_id;
            if ($a > $b) {
                [$a, $b] = [$b, $a];
            }
            $key = $a.'-'.$b;

            return ! in_array($key, $existingPairKeys, true);
        })->values();

        if ($availableRegistrations->isEmpty()) {
            return redirect()->route('tournament.setting.index', ['tournament' => $group->tournament_id])->with('info', 'Không còn cặp đăng ký nào chưa được thêm vào bảng.')->with('open_tab', 'groups-pairs');
        }

        $playerIds = $availableRegistrations->pluck('player1_id')->merge($availableRegistrations->pluck('player2_id'))->unique()->filter();
        $playersMap = \App\Models\Player::whereIn('id', $playerIds)->get()->keyBy('id');

        $pairs = $availableRegistrations->map(function ($r) use ($playersMap) {
            return [
                'registration_id' => $r->id,
                'player1_id' => (int) $r->player1_id,
                'player2_id' => (int) $r->player2_id,
                'player1' => $playersMap->get($r->player1_id),
                'player2' => $playersMap->get($r->player2_id),
            ];
        })->all();

        $formAction = route('tournament.setting.groups.pairs.add', ['tournament' => $group->tournament_id, 'group' => $group->id]);
        $cancelUrl = route('tournament.setting.index', ['tournament' => $group->tournament_id]);

        return view('admin.pairs.create', compact('group', 'pairs', 'formAction', 'cancelUrl'));
    }

    // Xóa cặp khỏi bảng
    public function deletePair(Request $request, $pairId)
    {
        // Accept POST from organizer UI as delete as well as admin-style _method=DELETE
        $isDelete = $request->input('_method') === 'DELETE' || $request->isMethod('post');
        if ($isDelete) {
            $pair = GroupPair::findOrFail($pairId);
            $group = $pair->group()->first();
            $this->ensureTournamentSession($group->tournament_id);
            $pair->delete();

            if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
                return response()->json(['success' => true]);
            }

            return redirect()->route('tournament.setting.index', ['tournament' => $group->tournament_id])->with('success', 'Đã xóa cặp đấu!')->with('open_tab', 'groups-pairs');
        }

        return response()->json(['error' => 'Invalid method'], 405);
    }

    // Lưu bảng đấu (có thể là cập nhật lại danh sách cặp)
    public function saveGroup(Request $request, $groupId)
    {
        // Có thể cập nhật tên bảng hoặc các thông tin khác
        $group = TournamentGroup::findOrFail($groupId);
        $this->ensureTournamentSession($group->tournament_id);
        $group->name = $request->input('name', $group->name);
        $group->save();

        return response()->json(['success' => true, 'group' => $group]);
    }

    // Tự động tạo vòng đấu bảng
    public function autoCreateRounds($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        // Logic tự động tạo vòng đấu bảng
        // Ví dụ: tạo 1 vòng cho mỗi bảng
        $groups = TournamentGroup::where('tournament_id', $tournamentId)->get();
        foreach ($groups as $group) {
            TournamentRound::create([
                'tournament_id' => $tournamentId,
                'name' => 'Vòng bảng '.$group->name,
                'type' => 'normal',
            ]);
        }

        return response()->json(['success' => true]);
    }

    // Organizer: proxy to generate groups (seeding)
    public function autoGenerateGroups(Request $request, $tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $validated = $request->validate([
            'number_of_groups' => 'required|integer|min:1',
        ]);

        try {
            $tournament = Tournament::findOrFail($tournamentId);
            // Use improved generation service with skill-based seeding
            $this->generationService->generateGroupsWithSeeding($tournament, (int) $validated['number_of_groups']);

            // Clear cache after generation
            $this->cacheService->invalidateTournamentCache($tournamentId, ['groups', 'standings']);

            return back()->with('success', '✅ Đã tự động chia bảng theo trình độ thành công!')->with('open_tab', 'groups-pairs');
        } catch (\Exception $e) {
            return $this->handleTournamentError($e, 'generateGroups', Tournament::findOrFail($tournamentId));
        }
    }

    // Organizer: generate group rounds (round-robin)
    public function autoGenerateGroupRounds($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        try {
            $tournament = Tournament::findOrFail($tournamentId);
            // Use improved generation service with memory optimization
            $this->generationService->generateRoundRobinOptimized($tournament);

            // Clear cache after generation
            $this->cacheService->invalidateTournamentCache($tournamentId, ['matches', 'schedule']);

            return back()->with('success', '✅ Đã tự động tạo các trận đấu vòng bảng thành công!')->with('open_tab', 'schedule');
        } catch (\Exception $e) {
            return $this->handleTournamentError($e, 'generateGroupRounds', Tournament::findOrFail($tournamentId));
        }
    }

    // Organizer: create playoffs from group standings
    public function generatePlayoffs(Request $request, $tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $validated = $request->validate([
            'teams_to_advance' => 'required|integer|min:1',
        ]);

        try {
            $tournament = Tournament::findOrFail($tournamentId);
            // Use improved playoff generation with proper bracket handling
            $roundName = $this->generationService->generatePlayoffsImproved($tournament, (int) $validated['teams_to_advance']);

            // Clear cache after generation
            $this->cacheService->invalidateTournamentCache($tournamentId, ['matches', 'playoffs']);

            return back()->with('success', "🏆 Vòng {$roundName} đã được tạo thành công!")->with('open_tab', 'playoffs');
        } catch (\Exception $e) {
            return $this->handleTournamentError($e, 'generatePlayoffs', Tournament::findOrFail($tournamentId));
        }
    }

    public function generateRoundOf32($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        try {
            $tournament = Tournament::findOrFail($tournamentId);
            $roundOf64Round = $tournament->rounds()->where('playoff_level', 'round_of_64')->first();

            if (! $roundOf64Round) {
                return back()->with('error', 'Không tìm thấy vòng 1/64 để tạo vòng 1/32.');
            }

            $message = $this->generationService->generateNextPlayoffRound($tournament, $roundOf64Round->name, ['Vòng 1/32'], 32);

            return back()->with('success', $message)->with('open_tab', 'playoffs');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function generateRoundOf16($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        try {
            $tournament = Tournament::findOrFail($tournamentId);
            $roundOf32Round = $tournament->rounds()->where('playoff_level', 'round_of_32')->first();

            if (! $roundOf32Round) {
                return back()->with('error', 'Không tìm thấy vòng 1/32 để tạo vòng 1/16.');
            }

            $message = $this->generationService->generateNextPlayoffRound($tournament, $roundOf32Round->name, ['Vòng 1/16'], 16);

            return back()->with('success', $message)->with('open_tab', 'playoffs');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function generateQuarterFinals($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        try {
            $tournament = Tournament::findOrFail($tournamentId);
            $roundOf16Round = $tournament->rounds()->where('playoff_level', 'round_of_16')->first();

            if (! $roundOf16Round) {
                return back()->with('error', 'Không tìm thấy vòng 1/16 để tạo tứ kết.');
            }

            $message = $this->generationService->generateNextPlayoffRound($tournament, $roundOf16Round->name, ['Tứ kết'], 8);

            return back()->with('success', $message)->with('open_tab', 'playoffs');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function generateSemiFinals($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        try {
            // Only generate from Quarter Finals, not from Round of 16
            $tournament = Tournament::findOrFail($tournamentId);
            $quarterFinalsRound = $tournament->rounds()->where('playoff_level', 'quarter_final')->first();

            if (! $quarterFinalsRound) {
                return back()->with('error', 'Không tìm thấy vòng tứ kết để tạo bán kết. Vui lòng tạo tứ kết trước.');
            }

            $message = $this->generationService->generateNextPlayoffRound($tournament, $quarterFinalsRound->name, ['Bán kết'], 4);

            return back()->with('success', $message)->with('open_tab', 'playoffs');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function generateFinals($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        try {
            $message = $this->generationService->generateNextPlayoffRound(Tournament::findOrFail($tournamentId), 'Bán kết', ['Chung kết', 'Tranh Hạng Ba'], 2);

            return back()->with('success', $message)->with('open_tab', 'bracket');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function clearGroups($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);
        $this->generationService->clearGroups(Tournament::findOrFail($tournamentId));

        return back()->with('success', 'Đã xóa tất cả các bảng đấu.');
    }

    public function clearGroupRounds($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);
        $this->generationService->clearGroupRounds(Tournament::findOrFail($tournamentId));

        return back()->with('success', 'Đã xóa tất cả các vòng đấu bảng.');
    }

    public function clearPlayoffs($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);
        $this->generationService->clearPlayoffs(Tournament::findOrFail($tournamentId));

        return back()->with('success', 'Đã xóa tất cả các vòng đấu Play-off.');
    }

    /**
     * Hiển thị lịch sử hoạt động của giải đấu cho Ban Tổ Chức (tournament-setting)
     */
    public function activityLog($tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $tournament = Tournament::findOrFail($tournamentId);
        $query = $tournament->activities()->with('causer');

        // Apply filters to keep parity with admin view
        if (request()->filled('event')) {
            $event = request()->string('event');
            $query->where(function ($q) use ($event) {
                $q->where('description', 'like', '%'.$event.'%')
                    ->orWhere('properties->event', $event)
                    ->orWhere('properties->event_label', 'like', '%'.$event.'%');
            });
        }

        if (request()->filled('causer')) {
            $query->whereHas('causer', function ($q) {
                $q->where('name', 'like', '%'.request('causer').'%');
            });
        }

        if (request()->filled('date_from')) {
            $query->whereDate('created_at', '>=', request('date_from'));
        }

        if (request()->filled('date_to')) {
            $query->whereDate('created_at', '<=', request('date_to'));
        }

        $activities = $query->latest()->paginate(20);

        return view('admin.tournaments.activity_log', compact('tournament', 'activities'));
    }

    // Thêm vòng đấu
    public function addRound(Request $request, $tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:normal,round_tree',
        ]);
        $round = TournamentRound::create([
            'tournament_id' => $tournamentId,
            'name' => $request->input('name'),
            'type' => $request->input('type', 'normal'),
        ]);

        return response()->json($round);
    }

    // Lấy danh sách player đã đăng ký và duyệt cho tournament, kèm các cặp đã có trong bảng đấu
    // Lấy danh sách cặp player hợp lệ cho từng group (chỉ cặp chưa có trong group đó)
    public function getApprovedPlayers($tournamentId, Request $request)
    {
        $groupId = $request->input('groupId');

        // ensure session for this tournament
        $this->ensureTournamentSession($tournamentId);

        $players = Player::whereHas('registrations', function ($q) use ($tournamentId) {
            $q->where('tournament_id', $tournamentId)->where('status', 'approved');
        })->get(['id', 'name']);

        // Lấy các cặp đã có trong group hiện tại
        $existingPairs = GroupPair::where('group_id', $groupId)->get(['player1_id', 'player2_id']);
        $pairs = [];
        foreach ($players as $i => $p1) {
            for ($j = $i + 1; $j < count($players); $j++) {
                $p2 = $players[$j];
                // Nếu cặp đã tồn tại ở group này thì disabled
                $exists = $existingPairs->contains(function ($pair) use ($p1, $p2) {
                    return ($pair->player1_id == $p1->id && $pair->player2_id == $p2->id)
                        || ($pair->player1_id == $p2->id && $pair->player2_id == $p1->id);
                });
                $pairs[] = [
                    'player1' => ['id' => $p1->id, 'name' => $p1->name],
                    'player2' => ['id' => $p2->id, 'name' => $p2->name],
                    'disabled' => $exists,
                ];
            }
        }

        return response()->json(['pairs' => $pairs]);
    }

    // Lấy danh sách cặp đấu của giải đấu
    public function getTournamentPairs(Request $request, $tournamentId)
    {
        $this->ensureTournamentSession($tournamentId);

        $pairs = GroupPair::whereHas('group', function ($q) use ($tournamentId) {
            $q->where('tournament_id', $tournamentId);
        })->with(['player1', 'player2', 'group'])->get();

        if ($request->expectsJson() || $request->isJson() || $request->wantsJson()) {
            return response()->json($pairs);
        }
    }

    // Thêm trận đấu vào vòng
    public function addMatch(Request $request, TournamentRound $round)
    {
        // Ensure session for the tournament this round belongs to
        $this->ensureTournamentSession($round->tournament_id);

        $request->validate([
            'pair1_id' => 'required|integer|different:pair2_id',
            'pair2_id' => 'required|integer',
        ]);

        $pair1 = GroupPair::with('player1', 'player2')->findOrFail($request->pair1_id);
        $pair2 = GroupPair::with('player1', 'player2')->findOrFail($request->pair2_id);

        // Make sure both pairs belong to the same tournament as the round
        if ($pair1->group->tournament_id !== $round->tournament_id || $pair2->group->tournament_id !== $round->tournament_id) {
            return response()->json(['error' => 'One or both pairs do not belong to this tournament'], 403);
        }

        $match = Matches::create([
            'match_id' => 'M'.strtoupper(uniqid()),
            'tournament_id' => $round->tournament_id,
            'tournament_round_id' => $round->id,
            'date' => now(), // Placeholder, should probably be set from the UI
            'type' => 'doubles',
            'result' => '0-0', // Placeholder
            'score' => '', // Placeholder
            'player1_id' => $pair1->player1_id,
            'player2_id' => $pair1->player2_id,
            'opponent1_id' => $pair2->player1_id,
            'opponent2_id' => $pair2->player2_id,
        ]);

        // Load pair relations (each pair has player1 and player2)
        $match->load(['pair1.player1', 'pair1.player2', 'pair2.player1', 'pair2.player2']);

        return response()->json($match);
    }

    // Organizer: update score for a match (mirror admin behavior but protected by tournament password)
    public function updateScore(Request $request, $matchId)
    {
        $match = Matches::with(['round', 'pair1', 'pair2'])->findOrFail($matchId);

        // Ensure organizer session for this tournament
        $tournamentId = $match->round?->tournament_id;
        if ($tournamentId) {
            $this->ensureTournamentSession($tournamentId);
        }

        $validated = $request->validate([
            'score1' => ['sometimes', 'required', 'integer', 'min:0'],
            'score2' => ['sometimes', 'required', 'integer', 'min:0'],
            'status' => ['sometimes', 'string', 'in:Chưa bắt đầu,Đang diễn ra,Hoàn thành,Tạm hoãn,Bị hủy'],
        ]);

        // Prevent updating if used in next playoff rounds similar logic
        if ($match->round?->type === 'playoff') {
            $nextRoundNames = [];

            if ($match->round->playoff_level === 'quarter_final') {
                $nextRoundNames = ['Bán kết'];
            } elseif ($match->round->name === 'Bán kết') {
                $nextRoundNames = ['Chung kết', 'Tranh Hạng Ba'];
            }

            if (! empty($nextRoundNames)) {
                $winnerId = $request->input('score1', 0) > $request->input('score2', 0) ? $match->pair1_id : $match->pair2_id;
                $loserId = $request->input('score1', 0) < $request->input('score2', 0) ? $match->pair1_id : $match->pair2_id;

                $isUsedInNextRound = Matches::query()
                    ->whereRelation('round', 'tournament_id', $tournamentId)
                    ->whereRelation('round', function ($query) use ($nextRoundNames) {
                        $query->whereIn('name', $nextRoundNames);
                    })
                    ->where(function ($query) use ($winnerId, $loserId) {
                        $query->where('pair1_id', $winnerId)
                            ->orWhere('pair2_id', $winnerId)
                            ->orWhere('pair1_id', $loserId)
                            ->orWhere('pair2_id', $loserId);
                    })
                    ->exists();

                if ($isUsedInNextRound) {
                    return response()->json(['success' => false, 'message' => 'Không thể cập nhật tỷ số. Kết quả trận đấu này đã được sử dụng để tạo vòng đấu tiếp theo.'], 422);
                }
            }
        }

        try {
            if (isset($validated['score1']) && isset($validated['score2'])) {
                $match->score = $validated['score1'].'-'.$validated['score2'];
            }

            if (isset($validated['status'])) {
                $match->status = $validated['status'];
            } elseif (isset($validated['score1']) || isset($validated['score2'])) {
                $s1 = $validated['score1'] ?? 0;
                $s2 = $validated['score2'] ?? 0;
                if ($s1 > 0 || $s2 > 0) {
                    $match->status = 'Hoàn thành';
                }
            }

            $match->save();

            if ($match->status === 'Hoàn thành') {
                $this->eloService->updateEloForMatch($match);
            }

            // Log activity for organizer updates as well
            if ($match->round && $match->round->tournament_id) {
                $tournament = Tournament::find($match->round->tournament_id);
                if ($tournament) {
                    activity()
                        ->performedOn($tournament)
                        ->causedBy(auth()->user())
                        ->withProperties([
                            'event' => 'match_score_updated',
                            'event_label' => 'Cập nhật kết quả',
                            'match_id' => $match->id,
                            'round' => $match->round->name,
                            'score' => $match->score,
                            'status' => $match->status,
                            'pair1_id' => $match->pair1_id,
                            'pair2_id' => $match->pair2_id,
                            'by' => 'organizer',
                        ])
                        ->event('match_score_updated')
                        ->log('Cập nhật kết quả trận đấu');
                }
            }

            // Touch tournament to invalidate cache based on updated_at
            if ($tournamentId) {
                $tournament = Tournament::find($tournamentId);
                if ($tournament) {
                    $tournament->touch();
                }
            }

            // Invalidate cache after score update
            $this->cacheService->invalidateTournamentCache($tournamentId, ['standings', 'matches']);

            // Clear specific standings cache
            $this->cacheService->clearStandingsCache($tournamentId);

            return response()->json(['success' => true, 'message' => 'Cập nhật tỉ số và điểm Elo thành công!']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Lỗi server khi cập nhật tỷ số.'], 500);
        }
    }

    /**
     * Swap player1_id and player2_id for a given pair.
     * This is for the organizer role.
     */
    public function swapPlayers(Request $request, GroupPair $pair): JsonResponse
    {
        $this->ensureTournamentSession($pair->group->tournament_id);

        try {
            // Use a transaction to ensure atomicity
            DB::transaction(function () use ($pair) {
                $player1 = $pair->player1_id;
                $player2 = $pair->player2_id;

                $pair->player1_id = $player2;
                $pair->player2_id = $player1;
                $pair->save();
            });

            // Get updated pair with player information for frontend
            $pair->load(['player1', 'player2']);

            return response()->json([
                'success' => true,
                'message' => 'Đã hoán đổi vị trí VĐV thành công.',
                'pair' => [
                    'id' => $pair->id,
                    'player1' => [
                        'id' => $pair->player1->id,
                        'name' => $pair->player1->name,
                        'avatar' => $pair->player1->avatar,
                    ],
                    'player2' => [
                        'id' => $pair->player2->id,
                        'name' => $pair->player2->name,
                        'avatar' => $pair->player2->avatar,
                    ],
                ],
            ]);
        } catch (\Exception $e) {
            Log::error("Error swapping players for pair {$pair->id} by organizer: ".$e->getMessage());

            return response()->json(['success' => false, 'message' => 'Có lỗi xảy ra, không thể hoán đổi.'], 500);
        }
    }

    public function exportRegistrations(Tournament $tournament)
    {
        $this->ensureTournamentSession($tournament->id);

        $fileName = 'dang-ky-'.\Illuminate\Support\Str::slug($tournament->name).'-'.now()->format('Ymd_His').'.xlsx';

        return Excel::download(new RegistrationsExport($tournament), $fileName);
    }

    /**
     * Xuất lịch thi đấu ra file Excel cho Ban Tổ Chức.
     */
    public function exportSchedule(Tournament $tournament)
    {
        $this->ensureTournamentSession($tournament->id);
        $fileName = 'lich-thi-dau-'.\Illuminate\Support\Str::slug($tournament->name).'.xlsx';

        return Excel::download(new ScheduleExport($tournament), $fileName);
    }

    /**
     * Xuất Bảng xếp hạng Vòng Bảng cho Ban Tổ Chức.
     */
    public function exportStandings(Tournament $tournament)
    {
        $this->ensureTournamentSession($tournament->id);

        $fileName = 'bang-xep-hang-'.\Illuminate\Support\Str::slug($tournament->name).'.xlsx';

        return Excel::download(new StandingsExport($tournament), $fileName);
    }

    /**
     * Export pairs list per group (Organizer)
     */
    public function exportPairs(Tournament $tournament)
    {
        // Ensure organizer or referee session is valid for this tournament
        $this->ensureTournamentSession($tournament->id);

        $fileName = 'danh-sach-cap-dau-'.\Illuminate\Support\Str::slug($tournament->name).'.xlsx';

        return \Maatwebsite\Excel\Facades\Excel::download(new \App\Exports\PairsExport($tournament), $fileName);
    }

    /**
     * Organize playoff rounds into structured format for the view
     */
    private function organizePlayoffRounds($playoffRounds)
    {
        $organized = [
            'quarter_finals' => $playoffRounds->where('playoff_level', 'quarter_final'),
            'semi_finals' => $playoffRounds->where('playoff_level', 'semi_final'),
            'final' => $playoffRounds->where('playoff_level', 'final'),
            'third_place' => $playoffRounds->where('playoff_level', 'third_place'),
            'other' => $playoffRounds->filter(function ($round) {
                return ! in_array($round->playoff_level, ['quarter_final', 'semi_final', 'final', 'third_place']);
            }),
        ];

        return $organized;
    }

    /**
     * Get user role and permissions for tournament
     */
    private function getUserTournamentRole(Tournament $tournament): array
    {
        return $this->authService->getUserPermissions($tournament);
    }

    /**
     * Handle tournament operation errors with user-friendly messages
     */
    private function handleTournamentError(\Exception $e, string $operation, Tournament $tournament)
    {
        $context = [
            'tournament_id' => $tournament->id,
            'tournament_name' => $tournament->name,
            'operation' => $operation,
            'user_id' => auth()->id(),
            'session_role' => session('tournament_role_'.$tournament->id),
        ];

        Log::error("Tournament operation failed: {$operation}", [
            'exception' => $e->getMessage(),
            'context' => $context,
            'trace' => $e->getTraceAsString(),
        ]);

        $userMessage = $this->getUserFriendlyErrorMessage($e->getMessage(), $operation);

        return back()
            ->with('error', $userMessage)
            ->with('error_technical', $e->getMessage()) // For debugging if needed
            ->withInput();
    }

    /**
     * Convert technical errors to user-friendly messages
     */
    private function getUserFriendlyErrorMessage(string $technicalError, string $operation): string
    {
        $lowerError = strtolower($technicalError);

        // Memory related errors
        if (str_contains($lowerError, 'memory') || str_contains($lowerError, 'exceeded')) {
            return '🔄 Hệ thống đang xử lý quá nhiều dữ liệu. Vui lòng thử lại sau 2-3 phút.';
        }

        // Database related errors
        if (str_contains($lowerError, 'database') || str_contains($lowerError, 'connection') || str_contains($lowerError, 'lock')) {
            return '🔄 Hệ thống đang bận. Vui lòng chờ một chút và thử lại.';
        }

        // Validation errors
        if (str_contains($lowerError, 'không đủ') || str_contains($lowerError, 'insufficient')) {
            return '⚠️ Không đủ điều kiện để tạo vòng đấu. Vui lòng kiểm tra số lượng đội/trận đã hoàn thành.';
        }

        if (str_contains($lowerError, 'chưa hoàn thành') || str_contains($lowerError, 'incomplete')) {
            return '📋 Vui lòng hoàn thành các trận đấu trước khi tạo vòng tiếp theo.';
        }

        if (str_contains($lowerError, 'chưa có bảng') || str_contains($lowerError, 'no groups')) {
            return '📊 Chưa có bảng đấu. Vui lòng tạo bảng đấu trước.';
        }

        // Operation specific messages
        return match ($operation) {
            'generateGroups' => '⚠️ Không thể tạo bảng đấu. Vui lòng kiểm tra số lượng đăng ký được duyệt và thử lại.',
            'generateGroupRounds' => '⚠️ Không thể tạo lịch thi đấu vòng bảng. Vui lòng kiểm tra các bảng đấu đã tạo.',
            'generatePlayoffs' => '⚠️ Không thể tạo vòng playoffs. Vui lòng hoàn thành ít nhất 90% trận đấu vòng bảng.',
            'generateQuarterFinals' => '⚠️ Không thể tạo tứ kết. Vui lòng hoàn thành các trận vòng 1/16 trước.',
            'generateSemiFinals' => '⚠️ Không thể tạo bán kết. Vui lòng hoàn thành các trận tứ kết trước.',
            'generateFinals' => '⚠️ Không thể tạo chung kết. Vui lòng hoàn thành các trận bán kết trước.',
            default => '❌ Có lỗi xảy ra khi xử lý. Vui lòng thử lại sau ít phút hoặc liên hệ hỗ trợ.'
        };
    }
}