<?php

namespace App\Http\Controllers;

use App\Models\Tournament;
use Illuminate\Http\Request;

class TournamentListController extends Controller
{
    public function index(Request $request)
    {
        $query = Tournament::query();
        // Search by name
        if ($request->filled('search')) {
            $query->where('name', 'like', '%'.$request->search.'%');
        }
        // Filter by location
        if ($request->filled('location')) {
            $query->where('location', $request->location);
        }
        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        $today = now()->toDateString();
        $allTournaments = $query->orderBy('start_date', 'desc')->get();
        // Use date-only comparisons to avoid time-of-day edge cases (casts set time to 00:00:00)
        $ongoing = $allTournaments->filter(function ($t) use ($today) {
            if (! $t->start_date || ! $t->end_date) {
                return false;
            }

            return $t->start_date->toDateString() <= $today && $t->end_date->toDateString() >= $today;
        });
        $upcoming = $allTournaments->filter(function ($t) use ($today) {
            return $t->start_date && $t->start_date->toDateString() > $today;
        });
        $finished = $allTournaments->filter(function ($t) use ($today) {
            return $t->end_date && $t->end_date->toDateString() < $today;
        });

        // Gộp lại theo thứ tự ưu tiên: đang diễn ra, sắp diễn ra, đã kết thúc
        $tournaments = $ongoing->concat($upcoming)->concat($finished);

        // Phân trang thủ công
        $perPage = 20;
        $page = $request->input('page', 1);
        $pagedTournaments = $tournaments->slice(($page - 1) * $perPage, $perPage);
        $paginated = new \Illuminate\Pagination\LengthAwarePaginator(
            $pagedTournaments,
            $tournaments->count(),
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        // Get unique locations for filter dropdown
        $locations = Tournament::select('location')->distinct()->pluck('location')->filter()->values();

        return view('tournaments.list', [
            'tournaments' => $paginated,
            'locations' => $locations,
            'ongoing' => $ongoing,
            'upcoming' => $upcoming,
            'finished' => $finished,
        ]);
    }
}
