<?php

namespace App\Http\Controllers;

use App\Helpers\MemoryBooster;
use App\Http\Requests\StoreRegistrationRequest;
use App\Models\Tournament;
use App\Services\RegistrationService;
use App\Services\TournamentCacheService;
use App\Services\TournamentService;

// Initialize memory booster for tournament operations
if (file_exists(app_path('Helpers/MemoryBooster.php'))) {
    MemoryBooster::init('tournament');
}

class TournamentController extends Controller
{
    public function show(string $slug, TournamentService $tournamentService, TournamentCacheService $cacheService)
    {
        // Load tournament with minimal data first
        $tournament = Tournament::where('slug', $slug)->firstOrFail();

        // Try to get cached tournament data
        $tournamentData = $cacheService->getTournamentData($tournament->id, 'overview');

        if (! $tournamentData || ! $cacheService->isStandingsCacheValid($tournament->id)) {
            // Load with optimized eager loading - only when cache miss
            $tournament->load([
                'registrations' => function ($query) {
                    $query->select('id', 'tournament_id', 'player1_id', 'player2_id', 'status')
                        ->where('status', '!=', 'rejected')
                        ->with(['player1:id,name,avatar,location,doubles_score,status', 'player2:id,name,avatar,location,doubles_score,status']);
                },
                'groups:id,tournament_id,name',
                'groups.pairs:id,group_id,player1_id,player2_id',
                'groups.pairs.player1:id,name,avatar',
                'groups.pairs.player2:id,name,avatar',
                'groupRounds:id,tournament_id,name,type',
                'groupRounds.matches' => function ($query) {
                    $query->select('id', 'tournament_round_id', 'pair1_id', 'pair2_id', 'score', 'status')
                        ->orderBy('id', 'asc');
                },
                'groupRounds.matches.pair1:id,group_id,player1_id,player2_id',
                'groupRounds.matches.pair1.player1:id,name,avatar',
                'groupRounds.matches.pair1.player2:id,name,avatar',
                'groupRounds.matches.pair2:id,group_id,player1_id,player2_id',
                'groupRounds.matches.pair2.player1:id,name,avatar',
                'groupRounds.matches.pair2.player2:id,name,avatar',
            ]);
        }

        // Get cached or calculate standings
        // Use StandingsService for correct round_robin-only standings
        $standingsService = app(\App\Services\StandingsService::class);
        $standingsByGroup = $standingsService->getGroupStandings($tournament);
        // Optionally cache the standings for performance
        $cacheService->cacheStandings($tournament->id, $standingsByGroup);

        $groupRounds = $tournament->groupRounds ?? collect();
        $knockoutRounds = $tournamentService->getKnockoutRounds($tournament->id);

        return view('tournaments.show', compact('tournament', 'groupRounds', 'standingsByGroup') + $knockoutRounds);
    }

    // Xử lý đăng ký player cho giải đấu
    public function register(StoreRegistrationRequest $request, Tournament $tournament)
    {
        try {
            $registrationService = app(RegistrationService::class);
            $registration = $registrationService->createRegistration($tournament, $request->validated());

            return redirect()
                ->route('tournaments.show', $tournament->slug)
                ->with('success', 'Đăng ký thành công! Đơn của bạn sẽ được BTC duyệt.');

        } catch (\Illuminate\Validation\ValidationException $e) {
            return back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }
}
