<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class SystemHealthController extends Controller
{
    public function check()
    {
        try {
            // Initialize memory booster
            if (file_exists(app_path('Helpers/MemoryBooster.php'))) {
                require_once app_path('Helpers/MemoryBooster.php');
                $memoryStats = \MemoryBooster::init('default');
            } else {
                $memoryStats = [
                    'memory_limit' => ini_get('memory_limit'),
                    'current_usage' => memory_get_usage(true),
                    'peak_usage' => memory_get_peak_usage(true)
                ];
            }

            // Database connectivity check
            $dbStatus = 'OK';
            try {
                DB::select('SELECT 1');
            } catch (\Exception $e) {
                $dbStatus = 'ERROR: ' . $e->getMessage();
            }

            // Cache check
            $cacheStatus = 'OK';
            try {
                Cache::put('health_check', 'test', 60);
                $cacheTest = Cache::get('health_check');
                if ($cacheTest !== 'test') {
                    $cacheStatus = 'ERROR: Cache not working';
                }
            } catch (\Exception $e) {
                $cacheStatus = 'ERROR: ' . $e->getMessage();
            }

            // Tournament data check
            $tournamentStatus = 'OK';
            $tournamentCount = 0;
            try {
                $tournamentCount = \App\Models\Tournament::count();
            } catch (\Exception $e) {
                $tournamentStatus = 'ERROR: ' . $e->getMessage();
            }

            $status = [
                'status' => 'healthy',
                'timestamp' => now()->toISOString(),
                'memory' => [
                    'limit' => $memoryStats['memory_limit'],
                    'current' => $this->formatBytes($memoryStats['current_usage']),
                    'peak' => $this->formatBytes($memoryStats['peak_usage']),
                    'usage_percentage' => round(($memoryStats['current_usage'] / $this->parseLimit($memoryStats['memory_limit'])) * 100, 2)
                ],
                'database' => [
                    'status' => $dbStatus,
                    'tournaments_count' => $tournamentCount
                ],
                'cache' => [
                    'status' => $cacheStatus
                ],
                'php' => [
                    'version' => PHP_VERSION,
                    'max_execution_time' => ini_get('max_execution_time'),
                    'max_input_vars' => ini_get('max_input_vars')
                ]
            ];

            return response()->json($status, 200);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage(),
                'timestamp' => now()->toISOString()
            ], 500);
        }
    }

    public function memoryTest()
    {
        try {
            // Initialize ultra memory mode
            if (file_exists(app_path('Helpers/MemoryBooster.php'))) {
                require_once app_path('Helpers/MemoryBooster.php');
                \MemoryBooster::init('ultra');
                \MemoryBooster::monitor('Memory Test Start');
            }

            // Simulate heavy operation
            $data = [];
            for ($i = 0; $i < 10000; $i++) {
                $data[] = str_repeat('test', 100);
            }

            if (class_exists('\MemoryBooster')) {
                \MemoryBooster::monitor('After Data Generation');
                \MemoryBooster::cleanup();
                \MemoryBooster::monitor('After Cleanup');
            }

            unset($data);
            gc_collect_cycles();

            return response()->json([
                'status' => 'success',
                'message' => 'Memory test completed successfully',
                'final_memory' => $this->formatBytes(memory_get_usage(true)),
                'peak_memory' => $this->formatBytes(memory_get_peak_usage(true))
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error', 
                'message' => $e->getMessage()
            ], 500);
        }
    }

    private function formatBytes($size)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $size >= 1024 && $i < 3; $i++) {
            $size /= 1024;
        }
        return round($size, 2) . ' ' . $units[$i];
    }

    private function parseLimit($limit)
    {
        $limit = trim($limit);
        $last = strtolower($limit[strlen($limit) - 1]);
        $value = (int) $limit;
        
        switch ($last) {
            case 'g': return $value * 1024 * 1024 * 1024;
            case 'm': return $value * 1024 * 1024;
            case 'k': return $value * 1024;
            default: return $value;
        }
    }
}