<?php

namespace App\Http\Controllers;

use App\Http\Requests\RegisterMemberRequest;
use App\Models\Player;
use App\Models\User;
use App\Services\PlayerService;
use App\Services\TelegramService;
use Illuminate\Support\Facades\Log;

class RegisterMemberController extends Controller
{
    protected $playerService;
    protected $telegramService;

    public function __construct(PlayerService $playerService, TelegramService $telegramService)
    {
        $this->playerService = $playerService;
        $this->telegramService = $telegramService;
    }

    public function show()
    {
        // Load guide from DOCUMENT (could be stored in a markdown file or database)
        $guide = file_get_contents(resource_path('markdown/register-guide.md')); // Adjust as needed

        return view('register-member', compact('guide'));
    }

    public function register(RegisterMemberRequest $request)
    {
        $data = $request->validated();

        // Kiểm tra email đã tồn tại
        if (User::where('email', $data['email'])->exists()) {
            return back()->withErrors(['email' => 'Email đã tồn tại!'])->withInput();
        }

        // Kiểm tra phone đã tồn tại
        if (! empty($data['phone']) && Player::where('phone', $data['phone'])->exists()) {
            return back()->withErrors(['phone' => 'Số điện thoại đã tồn tại!'])->withInput();
        }

        try {
            // Gán file avatar vào data để service xử lý
            if ($request->hasFile('avatar')) {
                $data['avatar'] = $request->file('avatar');
            }

            // Gọi service để tạo player và user
            $result = $this->playerService->createPlayer($data);
            
            // Gửi thông báo Telegram
            try {
                if ($this->telegramService->isConfigured()) {
                    $this->telegramService->sendNewMemberNotification($data);
                    Log::info('Đã gửi thông báo Telegram cho đăng ký mới: ' . $data['email']);
                } else {
                    Log::warning('Telegram chưa được cấu hình - bỏ qua gửi thông báo');
                }
            } catch (\Exception $telegramException) {
                // Log lỗi nhưng không làm gián đoạn quá trình đăng ký
                Log::error('Lỗi gửi thông báo Telegram: ' . $telegramException->getMessage());
            }
            
            // Optional: Gửi email cho người dùng với mật khẩu $result['password']
        } catch (\Exception $e) {
            return back()->with('error', 'Đã có lỗi xảy ra: '.$e->getMessage())->withInput();
        }

        return redirect()->route('home')->with('success', 'Đăng ký thành công! Chờ duyệt để cấp tích xanh.');
    }

    /**
     * Test kết nối Telegram (chỉ dùng cho admin)
     */
    public function testTelegram()
    {
        if (!auth()->check() || !auth()->user()->hasRole('admin')) {
            abort(403, 'Unauthorized');
        }

        $result = $this->telegramService->testConnection();
        
        if ($result['success']) {
            return redirect()->back()->with('success', $result['message']);
        } else {
            return redirect()->back()->with('error', $result['message']);
        }
    }
}
