<?php

namespace App\Http\Controllers;

use App\Models\Player;
use App\Rules\ImageValidationRules;
use App\Rules\ValidImageContent;
use App\Services\TelegramService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PlayerController extends Controller
{
    protected $telegramService;

    public function __construct(TelegramService $telegramService)
    {
        $this->telegramService = $telegramService;
    }

    // API kiểm tra player theo phone, name hoặc email
    public function checkByPhone(Request $request)
    {
        $phone = $request->query('phone');
        $name = $request->query('name');
        $email = $request->query('email');

        $query = Player::query();

        if ($phone) {
            $query->where('phone', $phone);
        } elseif ($name) {
            $query->where('name', 'like', '%' . $name . '%');
        } elseif ($email) {
            $query->whereHas('user', function ($q) use ($email) {
                $q->where('email', $email);
            });
        } else {
            return response()->json([]);
        }

        $players = $query->with('user')->limit(10)->get();

        $results = $players->map(function ($player) {
            return [
                'id' => $player->id,
                'name' => $player->name,
                'avatar' => $player->avatar,
                'doubles_score' => $player->doubles_score,
                'singles_score' => $player->singles_score,
                'phone' => $player->phone,
                'email' => $player->user ? $player->user->email : null,
                'status' => $player->status,
                'location' => $player->location,
            ];
        });

        return response()->json($results);
    }

    // Đăng ký player mới
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'phone' => 'required|unique:players,phone',
            'avatar' => 'nullable|image|max:2048',
        ]);

        $player = new Player;
        $player->name = $request->name;
        $player->phone = $request->phone;
        $player->location = 'Unknown'; // Default location
        // Generate unique player_id
        do {
            $player_id = strtoupper(substr(str_shuffle('ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'), 0, 6));
        } while (Player::where('player_id', $player_id)->exists());
        $player->player_id = $player_id;

        if ($request->hasFile('avatar')) {
            // Optimize avatar quickly using ImageOptimizationService (single 256x256 JPEG)
            $imageService = app(\App\Services\ImageOptimizationService::class);
            $optimized = $imageService->optimizeAndStore($request->file('avatar'), 'avatar');
            $player->avatar = $optimized['medium']['jpeg'] ?? null;
        }
        $player->save();

        return response()->json(['success' => true, 'player_id' => $player->id]);
    }

    // Hiển thị chi tiết player
    public function show(Request $request, $id)
    {
        $player = \App\Models\Player::findOrFail($id);

        // Server-side paginated match history with optional filter by result (win/loss)
        $perPage = 8;
        $resultFilter = $request->query('result'); // expected: 'win', 'loss', or null

        $matchesQuery = \App\Models\MatchModel::with(['tournament', 'pair1.player1', 'pair1.player2', 'pair2.player1', 'pair2.player2'])
            ->where(function ($query) use ($id) {
                $query->whereHas('pair1', function ($q) use ($id) {
                    $q->where('player1_id', $id)->orWhere('player2_id', $id);
                })->orWhereHas('pair2', function ($q) use ($id) {
                    $q->where('player1_id', $id)->orWhere('player2_id', $id);
                });
            });

        // Filter by result với logic chính xác
        if ($resultFilter === 'win' || $resultFilter === 'loss') {
            $matchesQuery->where(function ($query) use ($id, $resultFilter) {
                $query->where(function ($q) use ($id, $resultFilter) {
                    // Player trong pair1
                    $q->whereHas('pair1', function ($pairQuery) use ($id) {
                        $pairQuery->where('player1_id', $id)->orWhere('player2_id', $id);
                    });

                    if ($resultFilter === 'win') {
                        $q->where('result', 'pair1');
                    } else {
                        $q->where('result', 'pair2');
                    }
                })->orWhere(function ($q) use ($id, $resultFilter) {
                    // Player trong pair2
                    $q->whereHas('pair2', function ($pairQuery) use ($id) {
                        $pairQuery->where('player1_id', $id)->orWhere('player2_id', $id);
                    });

                    if ($resultFilter === 'win') {
                        $q->where('result', 'pair2');
                    } else {
                        $q->where('result', 'pair1');
                    }
                });
            });
        }

        $matches = $matchesQuery->orderByDesc('date')->paginate($perPage)->withQueryString();

        return view('players.show', compact('player', 'matches', 'resultFilter'));
    }

    // Cập nhật avatar cho player (chờ duyệt 24h)
    public function updateAvatar(Request $request, $id)
    {
        try {
            $validationRules = array_merge(
                ['avatar' => 'required'],
                ImageValidationRules::avatar(),
                ['avatar' => new ValidImageContent]
            );

            $messages = array_merge([
                'avatar.required' => 'Vui lòng chọn một file ảnh.',
            ], ImageValidationRules::customMessages('avatar', 2048));

            $request->validate(['avatar' => $validationRules], $messages);
            $player = Player::findOrFail($id);
            if ($request->hasFile('avatar')) {
                // Sử dụng ImageOptimizationService để tối ưu hóa ảnh
                $imageService = app(\App\Services\ImageOptimizationService::class);
                // Store optimized avatar into pending folder for approval flow
                $optimizedImages = $imageService->optimizeAndStore($request->file('avatar'), 'avatar_pending', $player->avatar_pending);

                // Lưu path của medium size vào pending
                $path = $optimizedImages['medium']['jpeg'];

                $player->avatar_pending = $path;
                $player->avatar_pending_requested_at = now();
                $player->save();
                // Gửi thông báo Telegram đồng bộ nhưng với timeout ngắn (không ảnh hưởng tốc độ request)
                try {
                    $data = 'Yêu cầu cập nhật avatar mới từ vận động viên:'.
                            "\nTên: {$player->name}".
                            "\nSĐT: {$player->phone}".
                            "\nPlayer ID: {$player->player_id}".
                            "\nThời gian: ".now()->toDateTimeString();

                    if ($this->telegramService->isConfigured()) {
                        // Gửi với timeout nhỏ đã set trong TelegramService -> không block lâu
                        $this->telegramService->sendMessage($data);
                    } else {
                        Log::warning('Telegram chưa được cấu hình - bỏ qua gửi thông báo');
                    }
                } catch (\Throwable $telegramException) {
                    // Log lỗi nhưng không làm gián đoạn quá trình upload
                    Log::error('Lỗi xử lý thông báo Telegram: '.$telegramException->getMessage());
                }

                return redirect()->back()->with('avatar_success', 'Yêu cầu cập nhật avatar đã được gửi thành công! Avatar mới sẽ được admin duyệt và cập nhật trong vòng 24 giờ.');
            } else {
                return redirect()->back()->with('avatar_error', 'Vui lòng chọn một file ảnh hợp lệ.');
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            $errors = $e->validator->errors()->all();

            return redirect()->back()->with('avatar_error', implode(' ', $errors));
        } catch (\Exception $e) {
            \Log::error('Avatar upload error: '.$e->getMessage());

            return redirect()->back()->with('avatar_error', 'Có lỗi xảy ra khi tải ảnh lên. Vui lòng thử lại.');
        }
    }

    // Admin duyệt avatar chờ
    public function approveAvatar($id)
    {
        $player = Player::findOrFail($id);
        if ($player->avatar_pending) {
            // Xóa avatar cũ nếu có
            if ($player->avatar) {
                // Handle both old storage format and new assets format
                if (str_starts_with($player->avatar, 'avatars/')) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($player->avatar);
                } elseif (str_starts_with($player->avatar, 'assets/avatars/')) {
                    $oldPath = public_path($player->avatar);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }
            }

            // Di chuyển file từ pending sang avatars directory
            $pendingPath = public_path($player->avatar_pending);
            if (file_exists($pendingPath)) {
                $ext = pathinfo($pendingPath, PATHINFO_EXTENSION);
                $avatarName = 'player_'.$player->id.'_'.time().'.'.$ext;
                $newPath = public_path('assets/avatars/'.$avatarName);

                if (rename($pendingPath, $newPath)) {
                    $player->avatar = 'assets/avatars/'.$avatarName;
                    $player->avatar_pending = null;
                    $player->avatar_pending_requested_at = null;
                    $player->save();

                    // Cập nhật avatar cho User liên quan nếu có
                    if ($player->user) {
                        $player->user->avatar = 'assets/avatars/'.$avatarName;
                        $player->user->save();
                    }

                    return redirect()->back()->with('avatar_success', 'Avatar đã được duyệt và cập nhật thành công.');
                } else {
                    return redirect()->back()->with('avatar_error', 'Không thể di chuyển file avatar.');
                }
            } else {
                return redirect()->back()->with('avatar_error', 'File avatar chờ duyệt không tồn tại.');
            }
        } else {
            return redirect()->back()->with('avatar_error', 'Không có avatar chờ duyệt hoặc chưa đủ 24h.');
        }
    }
}
