<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Tournament;
use App\Models\MatchModel;
use App\Models\GroupPair;
use App\Models\Player;
use App\Services\TournamentAuthService;
use App\Services\TournamentCacheService;
use App\Services\EloService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class TournamentApiController extends Controller
{
    protected $authService;
    protected $cacheService;
    protected $eloService;

    public function __construct(
        TournamentAuthService $authService,
        TournamentCacheService $cacheService,
        EloService $eloService
    ) {
        $this->authService = $authService;
        $this->cacheService = $cacheService;
        $this->eloService = $eloService;
    }

    /**
     * Update match score via API
     */
    public function updateMatchScore(Request $request, int $matchId): JsonResponse
    {
        try {
            $match = MatchModel::with(['round', 'pair1', 'pair2'])->findOrFail($matchId);
            $tournament = Tournament::findOrFail($match->tournament_id);

            // Check authentication
            if (!$this->authService->hasPermission($tournament, 'update_scores')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bạn không có quyền cập nhật tỷ số'
                ], 403);
            }

            // Validate input
            $validator = Validator::make($request->all(), [
                'score1' => 'required|integer|min:0|max:99',
                'score2' => 'required|integer|min:0|max:99',
                'status' => 'sometimes|string|in:Chưa bắt đầu,Đang diễn ra,Hoàn thành,Tạm hoãn,Bị hủy'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Dữ liệu không hợp lệ',
                    'errors' => $validator->errors()
                ], 422);
            }

            $score1 = $request->input('score1');
            $score2 = $request->input('score2');

            // Validate no tie games
            if ($score1 === $score2) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tỷ số không thể hòa'
                ], 422);
            }

            // Check if match is locked due to next round usage
            if ($this->isMatchLocked($match)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Không thể cập nhật tỷ số. Kết quả trận đấu này đã được sử dụng để tạo vòng đấu tiếp theo.'
                ], 422);
            }

            // Update match
            $match->score = "{$score1}-{$score2}";
            $match->status = $request->input('status', 'Hoàn thành');
            $match->save();

            // Update ELO if match is completed
            if ($match->status === 'Hoàn thành') {
                $this->eloService->updateEloForMatch($match);
            }

            // Touch tournament to invalidate cache based on updated_at
            $tournament->touch();

            // Invalidate cache
            $this->cacheService->invalidateTournamentCache($match->tournament_id, ['standings', 'matches']);
            
            // Clear specific standings cache
            $this->cacheService->clearStandingsCache($match->tournament_id);

            return response()->json([
                'success' => true,
                'message' => 'Cập nhật tỷ số thành công!',
                'match' => [
                    'id' => $match->id,
                    'score' => $match->score,
                    'status' => $match->status,
                    'updated_at' => $match->updated_at->toISOString()
                ]
            ]);

        } catch (\Exception $e) {
            Log::error("API: Error updating match score", [
                'match_id' => $matchId,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi cập nhật tỷ số'
            ], 500);
        }
    }

    /**
     * Swap players in a pair via API
     */
    public function swapPlayers(Request $request, int $pairId): JsonResponse
    {
        try {
            $pair = GroupPair::with('group')->findOrFail($pairId);
            $tournament = Tournament::findOrFail($pair->group->tournament_id);

            // Check authentication
            if (!$this->authService->hasPermission($tournament, 'swap_players')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bạn không có quyền hoán đổi vị trí người chơi'
                ], 403);
            }

            // Swap players
            $player1Id = $pair->player1_id;
            $player2Id = $pair->player2_id;

            $pair->player1_id = $player2Id;
            $pair->player2_id = $player1Id;
            $pair->save();

            // Get updated pair with player information for frontend
            $pair->load(['player1', 'player2']);

            // Invalidate cache
            $this->cacheService->invalidateTournamentCache($tournament->id, ['groups', 'standings']);

            return response()->json([
                'success' => true,
                'message' => 'Hoán đổi vị trí thành công!',
                'pair' => [
                    'id' => $pair->id,
                    'player1_id' => $pair->player1_id,
                    'player2_id' => $pair->player2_id,
                    'updated_at' => $pair->updated_at->toISOString(),
                    'player1' => [
                        'id' => $pair->player1->id,
                        'name' => $pair->player1->name,
                        'avatar' => $pair->player1->avatar
                    ],
                    'player2' => [
                        'id' => $pair->player2->id,
                        'name' => $pair->player2->name,
                        'avatar' => $pair->player2->avatar
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            Log::error("API: Error swapping players", [
                'pair_id' => $pairId,
                'error' => $e->getMessage(),
                'user_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi hoán đổi vị trí'
            ], 500);
        }
    }

    /**
     * Get tournament data for mobile app
     */
    public function getTournamentData(int $tournamentId, string $dataType): JsonResponse
    {
        try {
            $tournament = Tournament::findOrFail($tournamentId);

            // Get cached data
            $data = $this->cacheService->getTournamentData($tournamentId, $dataType);

            if ($data === null) {
                return response()->json([
                    'success' => false,
                    'message' => 'Không tìm thấy dữ liệu'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $data,
                'cached_at' => now()->toISOString()
            ]);

        } catch (\Exception $e) {
            Log::error("API: Error getting tournament data", [
                'tournament_id' => $tournamentId,
                'data_type' => $dataType,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi tải dữ liệu'
            ], 500);
        }
    }

    /**
     * Tournament authentication via API
     */
    public function authenticate(Request $request, int $tournamentId): JsonResponse
    {
        try {
            $tournament = Tournament::findOrFail($tournamentId);

            $validator = Validator::make($request->all(), [
                'password' => 'required|string',
                'user_type' => 'sometimes|string|in:auto,organizer,referee'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Dữ liệu không hợp lệ',
                    'errors' => $validator->errors()
                ], 422);
            }

            $password = $request->input('password');
            $userType = $request->input('user_type', 'auto');

            $result = $this->authService->authenticate($tournament, $password, $userType);

            return response()->json($result, $result['success'] ? 200 : 401);

        } catch (\Exception $e) {
            Log::error("API: Error authenticating tournament", [
                'tournament_id' => $tournamentId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi xác thực'
            ], 500);
        }
    }

    /**
     * Get user session info
     */
    public function getSessionInfo(int $tournamentId): JsonResponse
    {
        try {
            $tournament = Tournament::findOrFail($tournamentId);
            $permissions = $this->authService->getUserPermissions($tournament);

            return response()->json([
                'success' => true,
                'session' => $permissions
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra'
            ], 500);
        }
    }

    /**
     * Logout from tournament
     */
    public function logout(int $tournamentId): JsonResponse
    {
        try {
            $tournament = Tournament::findOrFail($tournamentId);
            $this->authService->logout($tournament);

            return response()->json([
                'success' => true,
                'message' => 'Đăng xuất thành công'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi đăng xuất'
            ], 500);
        }
    }

    /**
     * Get matches for a tournament with pagination
     */
    public function getMatches(Request $request, int $tournamentId): JsonResponse
    {
        try {
            $tournament = Tournament::findOrFail($tournamentId);

            $roundType = $request->input('round_type', 'all');
            $page = $request->input('page', 1);
            $perPage = min($request->input('per_page', 20), 100); // Max 100 per page

            $query = $tournament->matches()
                ->with([
                    'pair1.player1:id,name',
                    'pair1.player2:id,name',
                    'pair2.player1:id,name',
                    'pair2.player2:id,name',
                    'round:id,name,type'
                ]);

            if ($roundType !== 'all') {
                $query->whereHas('round', fn($q) => $q->where('type', $roundType));
            }

            $matches = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'matches' => $matches->items(),
                'pagination' => [
                    'current_page' => $matches->currentPage(),
                    'last_page' => $matches->lastPage(),
                    'per_page' => $matches->perPage(),
                    'total' => $matches->total()
                ]
            ]);

        } catch (\Exception $e) {
            Log::error("API: Error getting matches", [
                'tournament_id' => $tournamentId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi tải danh sách trận đấu'
            ], 500);
        }
    }

    /**
     * Check if match is locked due to next round usage
     */
    private function isMatchLocked(MatchModel $match): bool
    {
        if ($match->round?->type !== 'playoff') {
            return false;
        }

        $nextRoundNames = [];
        if ($match->round->name === 'Tứ kết') {
            $nextRoundNames = ['Bán kết'];
        } elseif ($match->round->name === 'Bán kết') {
            $nextRoundNames = ['Chung kết', 'Tranh Hạng Ba'];
        }

        if (empty($nextRoundNames)) {
            return false;
        }

        // Check if any next round exists with teams from this match
        return MatchModel::query()
            ->whereRelation('round', 'tournament_id', $match->tournament_id)
            ->whereRelation('round', function ($query) use ($nextRoundNames) {
                $query->whereIn('name', $nextRoundNames);
            })
            ->where(function ($query) use ($match) {
                $query->where('pair1_id', $match->pair1_id)
                    ->orWhere('pair2_id', $match->pair1_id)
                    ->orWhere('pair1_id', $match->pair2_id)
                    ->orWhere('pair2_id', $match->pair2_id);
            })
            ->exists();
    }
}