<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\VenueRequest;
use App\Models\Venue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class VenueController extends Controller
{
    public function index(Request $request)
    {
        $query = $this->buildVenueQuery($request);
        $venues = $query->paginate(10);
        $locations = Venue::select('location')->distinct()->orderBy('location')->pluck('location');

        return view('admin.venues.index', compact('venues', 'locations'));
    }

    public function search(Request $request)
    {
        try {
            $query = $this->buildVenueQuery($request);
            $venues = $query->paginate(10);

            $html = view('admin.venues.partials._venues_table', compact('venues'))->render();

            // For AJAX requests, return just the HTML
            if ($request->ajax()) {
                return response($html);
            }

            return $html;

        } catch (\Exception $e) {
            \Log::error('Venues search error: '.$e->getMessage(), [
                'search' => $request->get('search'),
                'location' => $request->get('location'),
                'trace' => $e->getTraceAsString(),
            ]);

            if ($request->ajax()) {
                return response()->json([
                    'error' => 'Có lỗi xảy ra khi tìm kiếm: '.$e->getMessage(),
                ], 500);
            }

            return back()->with('error', 'Có lỗi xảy ra khi tìm kiếm: '.$e->getMessage());
        }
    }

    private function buildVenueQuery(Request $request)
    {
        // Start with basic query - remove user relationship loading for now
        $query = Venue::query();

        // Xử lý tìm kiếm
        if ($request->has('search') && ! empty(trim($request->search))) {
            $search = trim($request->search);
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%')
                    ->orWhere('address', 'like', '%'.$search.'%')
                    ->orWhere('location', 'like', '%'.$search.'%')
                    ->orWhere('contact_phone', 'like', '%'.$search.'%')
                    ->orWhere('contact_email', 'like', '%'.$search.'%');
            });
        }

        // Xử lý lọc theo địa điểm
        if ($request->has('location') && ! empty(trim($request->location))) {
            $query->where('location', trim($request->location));
        }

        // Xử lý sắp xếp
        $sort = $request->input('sort', 'created_at');
        $direction = $request->input('direction', 'desc');

        // Validate sort direction
        $direction = in_array(strtolower($direction), ['asc', 'desc']) ? $direction : 'desc';

        $query->orderBy($sort, $direction);

        return $query;
    }

    public function create()
    {
        $this->authorize('create', Venue::class);
        $venue = new Venue;
        $locations = Venue::select('location')->distinct()->orderBy('location')->pluck('location');

        return view('admin.venues.create', compact('venue', 'locations'));
    }

    public function store(VenueRequest $request)
    {
        $this->authorize('create', Venue::class);

        try {
            \DB::beginTransaction();

            $data = $request->validated();

            // Thêm user_id của user hiện tại
            $data['user_id'] = auth()->id();

            $baseSlug = Str::slug($data['name']);
            $slug = $baseSlug;
            $i = 1;
            while (Venue::where('slug', $slug)->exists()) {
                $slug = $baseSlug.'-'.$i;
                $i++;
            }
            $data['slug'] = $slug;

            // Chỉ xử lý ảnh nếu có file được tải lên
            if ($request->hasFile('images')) {
                $imagePaths = $this->handleImageUpload($request);
                $data['images'] = $imagePaths;
            }

            $venue = Venue::create($data);

            \DB::commit();

            return redirect()->route('admin.venues.index')
                ->with('success', 'Sân đã được tạo thành công!');

        } catch (\Exception $e) {
            \DB::rollBack();

            \Log::error('Error creating venue: '.$e->getMessage(), [
                'user_id' => auth()->id(),
                'request_data' => $request->except(['images']),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Có lỗi xảy ra khi tạo sân: '.$e->getMessage());
        }
    }

    public function edit(Venue $venue)
    {
        $this->authorize('update', $venue);
        $locations = Venue::select('location')->distinct()->orderBy('location')->pluck('location');

        return view('admin.venues.edit', compact('venue', 'locations'));
    }

    public function update(VenueRequest $request, Venue $venue)
    {
        $this->authorize('update', $venue);

        try {
            \DB::beginTransaction();

            $data = $request->validated();

            if ($request->name !== $venue->name) {
                $baseSlug = Str::slug($data['name']);
                $slug = $baseSlug;
                $i = 1;
                while (Venue::where('slug', $slug)->where('id', '!=', $venue->id)->exists()) {
                    $slug = $baseSlug.'-'.$i;
                    $i++;
                }
                $data['slug'] = $slug;
            }

            // Chỉ xử lý ảnh nếu có file mới được tải lên
            if ($request->hasFile('images')) {
                $imagePaths = $this->handleImageUpload($request, $venue);
                $data['images'] = $imagePaths;
            }

            $venue->update($data);

            \DB::commit();

            return redirect()->route('admin.venues.index')
                ->with('success', 'Sân đã được cập nhật thành công!');

        } catch (\Exception $e) {
            \DB::rollBack();

            \Log::error('Error updating venue: '.$e->getMessage(), [
                'venue_id' => $venue->id,
                'user_id' => auth()->id(),
                'request_data' => $request->except(['images']),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Có lỗi xảy ra khi cập nhật sân: '.$e->getMessage());
        }
    }

    public function destroy(Venue $venue)
    {
        $this->authorize('delete', $venue);
        if ($venue->images) {
            foreach ($venue->images as $image) {
                try {
                    // Handle both old storage format and new assets format
                    if (str_starts_with($image, '/storage/')) {
                        Storage::disk('public')->delete(str_replace('/storage/', '', $image));
                    } elseif (str_starts_with($image, 'assets/')) {
                        $imagePath = public_path($image);
                        if (file_exists($imagePath)) {
                            unlink($imagePath);
                        }
                    }
                } catch (\Exception $e) {
                    logger()->warning('Failed to delete venue image on destroy: '.$e->getMessage());
                }
            }
        }
        $venue->delete();

        return redirect()->route('admin.venues.index')->with('success', 'Venue deleted successfully!');
    }

    /**
     * Handle image upload and deletion of old images.
     */
    private function handleImageUpload(Request $request, ?Venue $venue = null): ?array
    {
        // Delete old images only if updating an existing venue and new images are uploaded
        if ($venue && ! empty($venue->images) && $request->hasFile('images')) {
            foreach ($venue->images as $image) {
                try {
                    // Handle both old storage format and new assets format
                    if (str_starts_with($image, '/storage/')) {
                        $stored = str_replace('/storage/', '', $image);
                        Storage::disk('public')->delete($stored);
                    } elseif (str_starts_with($image, 'assets/')) {
                        $imagePath = public_path($image);
                        if (file_exists($imagePath)) {
                            unlink($imagePath);
                        }
                    }
                } catch (\Exception $e) {
                    // Log and continue
                    logger()->warning('Failed to delete old venue image: '.$e->getMessage());
                }
            }
        }

        $files = $request->file('images') ?? [];
        $paths = [];

        if (empty($files)) {
            return null;
        }

        try {
            $imageService = app(\App\Services\ImageOptimizationService::class);

            foreach ($files as $image) {
                if (! $image || ! $image->isValid()) {
                    continue;
                }

                try {
                    $optimizedImages = $imageService->optimizeAndStore($image, 'venue');
                    $paths[] = $optimizedImages['large']['jpeg']; // Sử dụng large size cho venue images
                } catch (\Exception $e) {
                    // Fallback to original method if optimization fails
                    logger()->warning('Image optimization failed for venue, using fallback: '.$e->getMessage());
                    $filename = time().'_'.uniqid().'.'.$image->getClientOriginalExtension();
                    $image->move(public_path('assets/venues'), $filename);
                    $paths[] = 'assets/venues/'.$filename;
                }
            }

            return $paths;

        } catch (\Exception $e) {
            logger()->error('Image upload failed for venue: '.$e->getMessage());
            throw new \Exception('Không thể tải lên hình ảnh: '.$e->getMessage());
        }
    }
}
