<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\UserService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules;
use Spatie\Permission\Models\Role;
use App\Rules\ImageValidationRules;
use App\Rules\ValidImageContent;

class UserController extends Controller
{
    protected $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;

        // Apply middleware to ensure only authenticated users can access
        $this->middleware('auth');

        // Apply policy authorization for all methods
        $this->authorizeResource(User::class, 'user');
    }

    public function index(Request $request)
    {
        // Authorization check đã được xử lý bởi authorizeResource() trong constructor

        $query = $this->buildUserQuery();
        $users = $query->paginate(15)->withQueryString();

        // Get data for filter dropdowns
        $roles = \Spatie\Permission\Models\Role::all();
        $locations = User::select('location')->distinct()->whereNotNull('location')->pluck('location');
        $genders = ['Nam', 'Nữ', 'Khác'];

        return view('admin.users.index', compact('users', 'roles', 'locations', 'genders'));
    }

    /**
     * API endpoint cho tìm kiếm AJAX users
     */
    public function search(Request $request)
    {
        try {
            $query = $this->buildUserQuery();
            $users = $query->paginate(15)->withQueryString();

            // Render HTML table content
            $tableHtml = view('admin.users.partials.table', compact('users'))->render();
            $paginationHtml = $users->hasPages() ?
                view('admin.users.partials.pagination', compact('users'))->render() : '';

            return response()->json([
                'success' => true,
                'html' => $tableHtml,
                'pagination' => $paginationHtml,
                'total' => $users->total(),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi tìm kiếm.',
            ], 500);
        }
    }

    /**
     * Build user query with filters
     */
    private function buildUserQuery()
    {
        $query = User::with('roles');

        // Search by name, email or phone
        if (request('search') && request('search') !== '') {
            $search = request('search');
            $query = $query->searchAll($search);
        }

        // Filter by role
        if (request('role') && request('role') !== '') {
            $query->whereHas('roles', function ($q) {
                $q->where('name', request('role'));
            });
        }

        // Filter by gender
        if (request('gender') && request('gender') !== '') {
            $query->where('gender', request('gender'));
        }

        // Filter by location
        if (request('location') && request('location') !== '') {
            $query->where('location', request('location'));
        }

        // Sort by specified column
        $sortBy = request('sort_by', 'created_at');
        $sortDirection = request('sort_direction', 'desc');
        $allowedSorts = ['name', 'email', 'created_at', 'location', 'gender'];

        if (in_array($sortBy, $allowedSorts)) {
            $query->orderBy($sortBy, $sortDirection === 'asc' ? 'asc' : 'desc');
        }

        return $query;
    }

    public function create()
    {
        // Authorization check đã được xử lý bởi authorizeResource() trong constructor

        $roles = Role::all();

        return view('admin.users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        // Authorization check đã được xử lý bởi authorizeResource() trong constructor

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'exists:roles,name'],
            'phone' => ['nullable', 'string', 'max:20'],
            'location' => ['nullable', 'string', 'max:255'],
            'gender' => ['nullable', 'string', 'in:Nam,Nữ,Khác'],
            'avatar' => array_merge(ImageValidationRules::avatar(), [new ValidImageContent()]),
        ]);

        try {
            $this->userService->createUser($validated);
        } catch (\Exception $e) {
            return back()->with('error', 'Đã có lỗi xảy ra: '.$e->getMessage())->withInput();
        }

        return redirect()->route('admin.users.index')->with('success', 'Tạo người dùng thành công!');
    }

    public function edit(User $user)
    {
        // Authorization check đã được xử lý bởi authorizeResource() trong constructor

        $roles = Role::all();

        return view('admin.users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, User $user)
    {
        // Authorization check đã được xử lý bởi authorizeResource() trong constructor

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'exists:roles,name'],
            'phone' => ['nullable', 'string', 'max:20'],
            'location' => ['nullable', 'string', 'max:255'],
            'gender' => ['nullable', 'string', 'in:Nam,Nữ,Khác'],
            'avatar' => array_merge(ImageValidationRules::avatar(), [new ValidImageContent()]),
        ], array_merge([
            'name.required' => 'Tên là bắt buộc.',
            'email.required' => 'Email là bắt buộc.',
            'email.email' => 'Email không hợp lệ.',
            'email.unique' => 'Email đã được sử dụng.',
            'password.confirmed' => 'Xác nhận mật khẩu không khớp.',
            'role.required' => 'Vai trò là bắt buộc.',
            'role.exists' => 'Vai trò không hợp lệ.',
            'phone.max' => 'Số điện thoại không được vượt quá 20 ký tự.',
            'gender.in' => 'Giới tính phải là Nam, Nữ hoặc Khác.',
        ], ImageValidationRules::customMessages('avatar', 4096)));

        try {
            $this->userService->updateUser($user, $validated);
        } catch (\Exception $e) {
            return back()->with('error', 'Đã có lỗi xảy ra: '.$e->getMessage())->withInput();
        }

        return redirect()->route('admin.users.index')->with('success', 'Cập nhật người dùng thành công!');
    }

    public function destroy(User $user)
    {
        // Kiểm tra quyền xóa user
        $this->authorize('delete', $user);

        // Ngăn chặn việc xóa chính tài khoản admin đang đăng nhập
        if ($user->id === auth()->id()) {
            return back()->with('error', 'Bạn không thể xóa chính mình.');
        }

        // Ngăn chặn xóa admin khác
        if ($user->hasRole('admin')) {
            return back()->with('error', 'Không thể xóa tài khoản admin.');
        }

        // Xóa avatar khỏi storage trước khi xóa user
        if ($user->avatar) {
            // Handle both old storage format and new assets format
            if (str_starts_with($user->avatar, 'avatars/')) {
                if (Storage::disk('public')->exists($user->avatar)) {
                    Storage::disk('public')->delete($user->avatar);
                }
            } elseif (str_starts_with($user->avatar, 'assets/avatars/')) {
                $avatarPath = public_path($user->avatar);
                if (file_exists($avatarPath)) {
                    unlink($avatarPath);
                }
            }
        }

        $user->delete();

        return redirect()->route('admin.users.index')->with('success', 'Đã xóa người dùng!');
    }
}