<?php

namespace App\Http\Controllers\Admin;

use App\Exports\RegistrationsExport;
use App\Exports\ScheduleExport;
use App\Exports\StandingsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\StoreTournamentRequest;
use App\Http\Requests\Admin\UpdateTournamentRequest;
use App\Models\Registration;
use App\Models\Tournament;
use App\Services\RegistrationService;
use App\Services\StandingsService;
use App\Services\TournamentGenerationService;
use App\Services\TournamentStateService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TournamentController extends Controller
{
    protected $standingsService;

    protected $generationService;

    protected $stateService;

    protected $registrationService;

    public function __construct(
        StandingsService $standingsService,
        TournamentGenerationService $generationService,
        TournamentStateService $stateService,
        RegistrationService $registrationService
    ) {
        $this->standingsService = $standingsService;
        $this->generationService = $generationService;
        $this->stateService = $stateService;
        $this->registrationService = $registrationService;
    }

    /**
     * Display a listing of tournaments
     */
    public function index(Request $request)
    {
        try {
            $tournaments = $this->buildTournamentQuery()
                ->orderBy('created_at', 'desc')
                ->get();

            return view('admin.tournaments.index', compact('tournaments'));

        } catch (\Exception $e) {
            Log::error("Error loading tournaments list: {$e->getMessage()}");

            return back()->with('error', 'Có lỗi xảy ra khi tải danh sách giải đấu.');
        }
    }

    /**
     * API endpoint cho tìm kiếm AJAX tournaments
     */
    public function search(Request $request)
    {
        try {
            $tournaments = $this->buildTournamentQuery()
                ->orderBy('created_at', 'desc')
                ->get();

            // Render HTML table content
            $tableHtml = view('admin.tournaments.partials.table', compact('tournaments'))->render();

            return response()->json([
                'success' => true,
                'html' => $tableHtml,
                'total' => $tournaments->count(),
            ]);

        } catch (\Exception $e) {
            Log::error("Error searching tournaments: {$e->getMessage()}");

            return response()->json([
                'success' => false,
                'message' => 'Có lỗi xảy ra khi tìm kiếm.',
            ], 500);
        }
    }

    /**
     * Build tournament query with filters
     */
    private function buildTournamentQuery()
    {
        $query = Tournament::query();

        // Apply access control: mod only sees their tournaments, admin sees all
        if (auth()->check()) {
            $query->accessibleBy(auth()->user());
        }

        // Search theo tên, địa điểm
        if (request('search') && request('search') !== '') {
            $search = request('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%')
                    ->orWhere('location', 'like', '%'.$search.'%')
                    ->orWhere('description', 'like', '%'.$search.'%');
            });
        }

        // Filter theo loại giải đấu
        if (request('type') && request('type') !== '') {
            $query->where('type', request('type'));
        }

        // Filter theo status
        if (request('status') && request('status') !== '') {
            $query->where('status', request('status'));
        }

        // Filter theo khoảng thời gian
        if (request('date_from') && request('date_from') !== '') {
            $query->whereDate('start_date', '>=', request('date_from'));
        }

        if (request('date_to') && request('date_to') !== '') {
            $query->whereDate('end_date', '<=', request('date_to'));
        }

        return $query;
    }

    /**
     * Show the form for creating a new tournament
     */
    public function create()
    {
        return view('admin.tournaments.create', [
            'tournament' => new Tournament,
        ]);
    }

    /**
     * Store a newly created tournament
     */
    public function store(StoreTournamentRequest $request)
    {
        $validated = $request->validated();

        // Remove file placeholder from validated payload
        unset($validated['image']);

        try {
            DB::beginTransaction();

            $data = $this->prepareTournamentPayload($validated);

            if ($request->hasFile('image')) {
                $imageService = app(\App\Services\ImageOptimizationService::class);
                $optimizedImages = $imageService->optimizeAndStore(
                    $request->file('image'),
                    'tournament'
                );
                $data['image'] = $optimizedImages['medium']['jpeg'];
            }

            $data['slug'] = $this->generateUniqueSlug($data['name']);

            $tournament = Tournament::create($data);

            // Set creator if user is authenticated
            if (auth()->check()) {
                $tournament->created_by = auth()->id();
                $tournament->save();
            }

            DB::commit();

            return redirect()->route('admin.tournaments.index')
                ->with('success', 'Giải đấu đã được tạo thành công!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error creating tournament: {$e->getMessage()}");

            return back()->withInput()->withErrors(['error' => 'Có lỗi xảy ra khi tạo giải đấu: '.$e->getMessage()]);
        }
    }

    /**
     * Show the form for editing the tournament
     */
    public function edit(Tournament $tournament)
    {
        return view('admin.tournaments.edit', compact('tournament'));
    }

    /**
     * Update the specified tournament
     */
    public function update(UpdateTournamentRequest $request, Tournament $tournament)
    {
        $validated = $request->validated();

        // Remove file placeholder from validated payload
        unset($validated['image']);

        try {
            DB::beginTransaction();

            $data = $this->prepareTournamentPayload($validated, true, $tournament);

            if ($request->hasFile('image')) {
                $imageService = app(\App\Services\ImageOptimizationService::class);
                $optimizedImages = $imageService->optimizeAndStore(
                    $request->file('image'),
                    'tournament',
                    $tournament->image
                );
                $data['image'] = $optimizedImages['medium']['jpeg'];
            }

            $tournament->fill($data);
            $tournament->save();

            DB::commit();

            return redirect()->route('admin.tournaments.index')
                ->with('success', 'Giải đấu đã được cập nhật thành công!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error updating tournament {$tournament->id}: {$e->getMessage()}");

            return back()->withInput()->withErrors(['error' => 'Có lỗi xảy ra khi cập nhật giải đấu: '.$e->getMessage()]);
        }
    }

    /**
     * Remove the specified tournament
     */
    public function destroy(Tournament $tournament)
    {

        try {
            DB::beginTransaction();

            // Delete associated image
            if ($tournament->image && file_exists(public_path($tournament->image))) {
                unlink(public_path($tournament->image));
            }

            // Tournament will cascade delete related data via foreign keys
            $tournament->delete();

            DB::commit();

            return redirect()->route('admin.tournaments.index')
                ->with('success', 'Giải đấu đã được xóa thành công!');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error deleting tournament {$tournament->id}: {$e->getMessage()}");

            return back()->with('error', 'Có lỗi xảy ra khi xóa giải đấu.');
        }
    }

    // Validation moved to dedicated Form Requests for clarity and reusability

    private function prepareTournamentPayload(array $data, bool $isUpdate = false, ?Tournament $tournament = null): array
    {
        $payload = $data;

        if ($isUpdate) {
            unset($payload['password']);
        } else {
            $payload['status'] = 'draft';
            $payload['password'] = $payload['password'] ?? $this->generateOrganizerPassword();
            $payload['referee_password'] = $payload['referee_password'] ?? $this->generateRefereePassword();
        }

        // Normalize nullable numeric fields
        $payload['max_pairs'] = isset($payload['max_pairs']) && $payload['max_pairs'] !== null && $payload['max_pairs'] !== ''
            ? (int) $payload['max_pairs']
            : null;

        $payload['fee'] = isset($payload['fee']) && $payload['fee'] !== null && $payload['fee'] !== ''
            ? (float) $payload['fee']
            : null;

        if (isset($payload['prizes']) && $payload['prizes'] === '') {
            $payload['prizes'] = null;
        }

        if (isset($payload['description']) && $payload['description'] === '') {
            $payload['description'] = null;
        }

        if (isset($payload['rules']) && $payload['rules'] === '') {
            $payload['rules'] = null;
        }

        $payload['skill_point'] = isset($payload['skill_point']) ? (float) $payload['skill_point'] : null;
        $payload['skill_tolerance'] = isset($payload['skill_tolerance']) ? (float) $payload['skill_tolerance'] : null;

        if (array_key_exists('referee_password', $payload)) {
            $payload['referee_password'] = $payload['referee_password'] !== ''
                ? $payload['referee_password']
                : null;
        } elseif ($isUpdate && $tournament) {
            $payload['referee_password'] = $tournament->referee_password;
        }

        return $payload;
    }

    private function generateUniqueSlug(string $name): string
    {
        $baseSlug = Str::slug($name);
        $slug = $baseSlug;
        $counter = 1;

        while (Tournament::where('slug', $slug)->exists()) {
            $slug = $baseSlug.'-'.$counter++;
        }

        return $slug;
    }

    private function generateOrganizerPassword(): string
    {
        return str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    }

    private function generateRefereePassword(): string
    {
        return str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    }

    /**
     * Display tournament dashboard with enhanced analytics (OPTIMIZED)
     */
    public function show(Tournament $tournament)
    {
        try {
            // Get user's tournament role first for UI customization
            $userRole = $this->getUserTournamentRole($tournament);

            // Use caching for expensive operations
            $cacheKey = "tournament_dashboard_{$tournament->id}_".md5($tournament->updated_at);

            $cachedData = cache()->remember($cacheKey, 300, function () use ($tournament) { // 5 minutes cache
                // Load only essential relationships
                $tournament->load([
                    'groups' => fn ($query) => $query->select('id', 'name', 'tournament_id'),
                    'rounds' => fn ($query) => $query->select('id', 'name', 'type', 'tournament_id', 'playoff_level')->orderBy('created_at'),
                    'rounds.matches' => fn ($query) => $query->select('id', 'tournament_round_id', 'tournament_id', 'pair1_id', 'pair2_id', 'score', 'status', 'round_number'),
                    'rounds.matches.pair1' => fn ($query) => $query->select('id', 'group_id', 'player1_id', 'player2_id'),
                    'rounds.matches.pair1.player1' => fn ($query) => $query->select('id', 'name', 'avatar'),
                    'rounds.matches.pair1.player2' => fn ($query) => $query->select('id', 'name', 'avatar'),
                    'rounds.matches.pair2' => fn ($query) => $query->select('id', 'group_id', 'player1_id', 'player2_id'),
                    'rounds.matches.pair2.player1' => fn ($query) => $query->select('id', 'name', 'avatar'),
                    'rounds.matches.pair2.player2' => fn ($query) => $query->select('id', 'name', 'avatar'),
                ]);

                // Get basic tournament state (faster operations)
                $tournamentState = $this->stateService->getCurrentPhase($tournament);
                $tournamentStats = $this->stateService->getTournamentStats($tournament);
                $validation = $this->stateService->validateTournamentState($tournament);

                return compact('tournamentState', 'tournamentStats', 'validation');
            });

            // Extract cached data
            extract($cachedData);

            // Load standings only if needed (lazy loading)
            $standingsByGroup = $this->getStandingsLazy($tournament);

            // Organize playoff rounds efficiently
            $playoffRounds = $this->organizePlayoffRounds($tournament->rounds->where('type', 'playoff'));

            return view('admin.tournaments.show', compact(
                'tournament',
                'standingsByGroup',
                'playoffRounds',
                'tournamentState',
                'tournamentStats',
                'validation',
                'userRole'
            ));

        } catch (\Exception $e) {
            Log::error("Error loading tournament {$tournament->id}: {$e->getMessage()}");

            return back()->with('error', 'Có lỗi xảy ra khi tải thông tin giải đấu. Vui lòng thử lại.');
        }
    }

    /**
     * Enhanced group generation with job queue for large tournaments
     */
    public function autoGenerateGroups(Request $request, Tournament $tournament)
    {
        $this->authorize('generateBrackets', $tournament);

        $validated = $request->validate([
            'number_of_groups' => 'required|integer|min:1|max:26',
        ]);

        try {
            // Check if we should use job queue for large tournaments
            $registrationCount = $tournament->registrations()
                ->where('status', 'approved')
                ->count();

            if ($registrationCount > 50) {
                // Use job queue for large tournaments
                \App\Jobs\GenerateTournamentGroupsJob::dispatch(
                    $tournament,
                    (int) $validated['number_of_groups'],
                    auth()->id()
                );

                return response()->json([
                    'success' => true,
                    'message' => 'Đã khởi tạo tạo bảng đấu. Vui lòng chờ trong giây lát...',
                    'use_polling' => true,
                    'poll_url' => route('api.job-status', $tournament->id).'?type=groups',
                ]);
            }

            // For smaller tournaments, process immediately
            DB::transaction(function () use ($tournament, $validated) {
                $this->generationService->generateGroups($tournament, (int) $validated['number_of_groups']);

                // Log activity
                activity()
                    ->performedOn($tournament)
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'event' => 'groups_generated',
                        'event_label' => 'Tạo bảng đấu',
                        'number_of_groups' => $validated['number_of_groups'],
                    ])
                    ->event('groups_generated')
                    ->log('Tạo bảng đấu tự động');
            });

            // Clear state cache
            $this->stateService->clearStateCache($tournament);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Đã tự động chia bảng theo trình độ thành công!',
                ]);
            }

            return back()
                ->with('success', 'Đã tự động chia bảng theo trình độ thành công!')
                ->with('open_tab', 'groups-pairs');

        } catch (\Exception $e) {
            Log::error("Failed to generate groups for tournament {$tournament->id}: {$e->getMessage()}");

            $errorMessage = $this->getReadableError($e->getMessage());

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $errorMessage,
                ], 422);
            }

            return back()->with('error', $errorMessage);
        }
    }

    /**
     * Enhanced group rounds generation with job queue
     */
    public function autoGenerateGroupRounds(Tournament $tournament)
    {
        $this->authorize('generateBrackets', $tournament);

        try {
            // Check if we should use job queue for tournaments with many groups
            $groupCount = $tournament->groups()->count();
            $totalPairs = $tournament->groups()->withCount('pairs')->get()->sum('pairs_count');

            if ($totalPairs > 100) {
                // Use job queue for large tournaments
                \App\Jobs\GenerateTournamentRoundsJob::dispatch($tournament, auth()->id());

                return response()->json([
                    'success' => true,
                    'message' => 'Đã khởi tạo tạo vòng đấu. Vui lòng chờ trong giây lát...',
                    'use_polling' => true,
                    'poll_url' => route('api.job-status', $tournament->id).'?type=rounds',
                ]);
            }

            // For smaller tournaments, process immediately
            DB::transaction(function () use ($tournament) {
                $this->generationService->generateGroupRounds($tournament);

                activity()
                    ->performedOn($tournament)
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'event' => 'group_rounds_generated',
                        'event_label' => 'Tạo vòng bảng',
                    ])
                    ->event('group_rounds_generated')
                    ->log('Tạo vòng bảng tự động');
            });

            $this->stateService->clearStateCache($tournament);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Đã tự động tạo các trận đấu vòng bảng thành công!',
                ]);
            }

            return back()
                ->with('success', 'Đã tự động tạo các trận đấu vòng bảng thành công!')
                ->with('open_tab', 'schedule');

        } catch (\Exception $e) {
            Log::error("Failed to generate group rounds for tournament {$tournament->id}: {$e->getMessage()}");

            $errorMessage = $this->getReadableError($e->getMessage());

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $errorMessage,
                ], 422);
            }

            return back()->with('error', $errorMessage);
        }
    }

    /**
     * Enhanced playoffs generation with improved bye handling and job queue
     */
    public function generatePlayoffs(Request $request, Tournament $tournament)
    {
        $this->authorize('generateBrackets', $tournament);

        $validated = $request->validate([
            'teams_to_advance' => 'required|integer|min:1|max:4',
        ]);

        try {
            // Check if we should use job queue for complex playoffs
            $groupCount = $tournament->groups()->count();
            $estimatedTeams = $groupCount * $validated['teams_to_advance'];

            if ($estimatedTeams > 32) {
                // Use job queue for large playoffs
                \App\Jobs\GeneratePlayoffsJob::dispatch(
                    $tournament,
                    (int) $validated['teams_to_advance'],
                    auth()->id()
                );

                return response()->json([
                    'success' => true,
                    'message' => 'Đã khởi tạo tạo playoffs. Vui lòng chờ trong giây lát...',
                    'use_polling' => true,
                    'poll_url' => route('api.job-status', $tournament->id).'?type=playoffs',
                ]);
            }

            // For smaller tournaments, process immediately
            $roundName = DB::transaction(function () use ($tournament, $validated) {
                $roundName = $this->generationService->generatePlayoffsImproved(
                    $tournament,
                    (int) $validated['teams_to_advance']
                );

                activity()
                    ->performedOn($tournament)
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'event' => 'playoffs_generated',
                        'event_label' => 'Tạo vòng playoffs',
                        'teams_to_advance' => $validated['teams_to_advance'],
                    ])
                    ->event('playoffs_generated')
                    ->log('Tạo vòng playoffs');

                return $roundName;
            });

            $this->stateService->clearStateCache($tournament);

            return back()
                ->with('success', "Vòng {$roundName} đã được tạo thành công!")
                ->with('open_tab', 'playoffs');

        } catch (\Exception $e) {
            Log::error("Failed to generate playoffs for tournament {$tournament->id}: {$e->getMessage()}");

            return back()->with('error', $this->getReadableError($e->getMessage()));
        }
    }

    /**
     * Generate Round of 32 from Round of 64
     */
    public function generateRoundOf32(Tournament $tournament)
    {
        $roundOf64Round = $tournament->rounds()
            ->where('playoff_level', 'round_of_64')
            ->first();

        if (! $roundOf64Round) {
            return back()->with('error', 'Không tìm thấy vòng 1/64 để tạo vòng 1/32.');
        }

        return $this->generateNextRound($tournament, $roundOf64Round->name, 'Vòng 1/32');
    }

    /**
     * Generate Round of 16 from Round of 32
     */
    public function generateRoundOf16(Tournament $tournament)
    {
        $roundOf32Round = $tournament->rounds()
            ->where('playoff_level', 'round_of_32')
            ->first();

        if (! $roundOf32Round) {
            return back()->with('error', 'Không tìm thấy vòng 1/32 để tạo vòng 1/16.');
        }

        return $this->generateNextRound($tournament, $roundOf32Round->name, 'Vòng 1/16');
    }

    /**
     * Enhanced next round generation
     */
    public function generateQuarterFinals(Tournament $tournament)
    {
        // Find Round of 16 to generate Quarter Finals
        $roundOf16Round = $tournament->rounds()
            ->where('playoff_level', 'round_of_16')
            ->first();

        if (! $roundOf16Round) {
            return back()->with('error', 'Không tìm thấy vòng 1/16 để tạo tứ kết.');
        }

        return $this->generateNextRound($tournament, $roundOf16Round->name, 'Tứ kết');
    }

    public function generateSemiFinals(Tournament $tournament)
    {
        // Only generate Semi Finals from Quarter Finals (not from Round of 16)
        $quarterFinalsRound = $tournament->rounds()
            ->where('playoff_level', 'quarter_final')
            ->first();

        if (! $quarterFinalsRound) {
            return back()->with('error', 'Không tìm thấy vòng tứ kết để tạo bán kết. Vui lòng tạo tứ kết trước.');
        }

        return $this->generateNextRound($tournament, $quarterFinalsRound->name, 'Bán kết');
    }

    public function generateFinals(Tournament $tournament)
    {
        return $this->generateNextRound($tournament, 'Bán kết', 'Chung kết');
    }

    private function generateNextRound(Tournament $tournament, string $previousRound, string $nextRound)
    {
        $this->authorize('generateBrackets', $tournament);

        try {
            $message = DB::transaction(function () use ($tournament, $previousRound, $nextRound) {
                $message = $this->generationService->generateNextPlayoffRoundImproved($tournament, $previousRound);

                activity()
                    ->performedOn($tournament)
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'event' => 'next_round_generated',
                        'event_label' => 'Tạo vòng tiếp theo',
                        'previous_round' => $previousRound,
                        'next_round' => $nextRound,
                    ])
                    ->event('next_round_generated')
                    ->log("Tạo vòng tiếp theo từ {$previousRound}");

                return $message;
            });

            $this->stateService->clearStateCache($tournament);

            return back()
                ->with('success', $message)
                ->with('open_tab', 'playoffs');

        } catch (\Exception $e) {
            Log::error("Failed to generate next round for tournament {$tournament->id}: {$e->getMessage()}");

            return back()->with('error', $this->getReadableError($e->getMessage()));
        }
    }

    /**
     * Enhanced registration management
     */
    public function registrations(Tournament $tournament)
    {
        $this->authorize('approveRegistrations', $tournament);

        $registrations = $tournament->registrations()
            ->with(['player1', 'player2'])
            ->latest()
            ->paginate(20);

        $stats = [
            'total' => $tournament->registrations->count(),
            'approved' => $tournament->registrations->where('status', 'approved')->count(),
            'pending' => $tournament->registrations->where('status', 'pending')->count(),
            'rejected' => $tournament->registrations->where('status', 'rejected')->count(),
        ];

        return view('admin.tournaments.registrations', compact('tournament', 'registrations', 'stats'));
    }

    /**
     * Enhanced registration status update
     */
    public function updateRegistrationStatus(Request $request, Tournament $tournament, Registration $registration)
    {
        $this->authorize('approveRegistrations', $registration->tournament);

        $validated = $request->validate([
            'status' => 'required|in:approved,rejected,pending',
            'reason' => 'nullable|string|max:500',
        ]);

        try {
            DB::transaction(function () use ($registration, $validated) {
                $oldStatus = $registration->status;

                $this->registrationService->updateStatus(
                    $registration,
                    $validated['status'],
                    $validated['reason'] ?? null
                );

                activity()
                    ->performedOn($registration->tournament)
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'event' => 'registration_status_updated',
                        'event_label' => 'Duyệt đăng ký',
                        'registration_id' => $registration->id,
                        'old_status' => $oldStatus,
                        'new_status' => $validated['status'],
                        'reason' => $validated['reason'] ?? null,
                    ])
                    ->event('registration_status_updated')
                    ->log('Cập nhật trạng thái đăng ký');
            });

            // Clear tournament cache since registration counts may affect validation
            $this->stateService->clearStateCache($registration->tournament);

            return back()->with('success', 'Cập nhật trạng thái đăng ký thành công!');

        } catch (\Exception $e) {
            Log::error("Failed to update registration {$registration->id}: {$e->getMessage()}");

            return back()->with('error', $this->getReadableError($e->getMessage()));
        }
    }

    /**
     * Tournament health check endpoint
     */
    public function healthCheck(Tournament $tournament)
    {
        $this->authorize('view', $tournament);

        try {
            $validation = $this->stateService->validateTournamentState($tournament);
            $phase = $this->stateService->getCurrentPhase($tournament);
            $stats = $this->stateService->getTournamentStats($tournament);

            return response()->json([
                'health_score' => $validation['tournament_health_score'] ?? 0,
                'phase' => $phase,
                'validation' => $validation,
                'stats' => $stats,
                'timestamp' => now(),
            ]);

        } catch (\Exception $e) {
            Log::error("Health check failed for tournament {$tournament->id}: {$e->getMessage()}");

            return response()->json(['error' => 'Health check failed'], 500);
        }
    }

    /**
     * Enhanced activity log with filtering
     */
    public function activityLog(Tournament $tournament, Request $request)
    {
        $this->authorize('viewActivityLogs', $tournament);

        $query = $tournament->activities()->with('causer');

        // Apply filters
        if ($request->filled('event')) {
            $event = $request->string('event');
            $query->where(function ($q) use ($event) {
                $q->where('description', 'like', '%'.$event.'%')
                    ->orWhere('properties->event', $event)
                    ->orWhere('properties->event_label', 'like', '%'.$event.'%');
            });
        }

        if ($request->filled('causer')) {
            $query->whereHas('causer', function ($q) use ($request) {
                $q->where('name', 'like', '%'.$request->causer.'%');
            });
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $activities = $query->latest()->paginate(20);

        return view('admin.tournaments.activity_log', compact('tournament', 'activities'));
    }

    /**
     * Organize playoff rounds for better display
     */
    private function organizePlayoffRounds($playoffRounds)
    {
        $organized = [
            'round_of_16' => $playoffRounds->where('playoff_level', 'round_of_16'),
            'quarter_finals' => $playoffRounds->where('playoff_level', 'quarter_final'),
            'semi_finals' => $playoffRounds->where('playoff_level', 'semi_final'),
            'final' => $playoffRounds->where('playoff_level', 'final'),
            'third_place' => $playoffRounds->where('playoff_level', 'third_place'),
            'other' => $playoffRounds->filter(function ($round) {
                return ! in_array($round->playoff_level, ['round_of_16', 'quarter_final', 'semi_final', 'final', 'third_place']);
            }),
        ];

        return $organized;
    }

    /**
     * Get user's tournament role for UI customization
     */
    private function getUserTournamentRole(Tournament $tournament): array
    {
        $user = auth()->user();
        $role = 'guest';
        $permissions = [];

        // Prioritize authenticated user roles (admin/mod) over session roles
        if ($user) {
            if ($user->hasRole('admin')) {
                $role = 'admin';
                $permissions = ['all'];
            } elseif ($user->hasRole('mod')) {
                $role = 'mod';
                $permissions = ['manage', 'approve', 'export'];
            }
        }

        // Only check session-based roles if user is not admin/mod
        if (! $user || (! $user->hasRole('admin') && ! $user->hasRole('mod'))) {
            $sessionRole = session('tournament_role_'.$tournament->id);
            if ($sessionRole === 'organizer') {
                $role = 'organizer';
                $permissions = ['manage', 'approve', 'export'];
            } elseif ($sessionRole === 'referee') {
                $role = 'referee';
                $permissions = ['view', 'update_scores', 'export'];
            }
        }

        return [
            'role' => $role,
            'permissions' => $permissions,
            'can_manage' => in_array($role, ['admin', 'mod', 'organizer']),
            'can_approve' => in_array($role, ['admin', 'mod', 'organizer']),
            'can_update_scores' => in_array($role, ['admin', 'mod', 'organizer', 'referee']),
            'can_export' => in_array($role, ['admin', 'mod', 'organizer', 'referee']),
        ];
    }

    /**
     * Convert technical errors to user-friendly messages
     */
    private function getReadableError(string $error): string
    {
        $errorMappings = [
            'Chưa có cặp đấu nào được duyệt' => 'Vui lòng duyệt các đăng ký trước khi tạo bảng đấu.',
            'Không đủ đội để tạo vòng Play-off' => 'Cần hoàn thành vòng bảng và có đủ đội vào playoffs.',
            'Không tìm thấy vòng' => 'Vòng đấu trước chưa được tạo hoặc đã bị xóa.',
            'Connection refused' => 'Lỗi kết nối cơ sở dữ liệu. Vui lòng thử lại sau.',
        ];

        foreach ($errorMappings as $pattern => $message) {
            if (str_contains($error, $pattern)) {
                return $message;
            }
        }

        // Return original error if no mapping found
        return $error;
    }

    // Keep existing methods for compatibility
    public function clearGroups(Tournament $tournament)
    {
        $this->authorize('generateBrackets', $tournament);

        try {
            $this->generationService->clearGroups($tournament);
            $this->stateService->clearStateCache($tournament);

            return back()->with('success', 'Đã xóa tất cả các bảng đấu.');
        } catch (\Exception $e) {
            return back()->with('error', 'Không thể xóa bảng đấu: '.$e->getMessage());
        }
    }

    public function clearGroupRounds(Tournament $tournament)
    {
        $this->authorize('generateBrackets', $tournament);

        try {
            $this->generationService->clearGroupRounds($tournament);
            $this->stateService->clearStateCache($tournament);

            return back()->with('success', 'Đã xóa tất cả các vòng đấu bảng.');
        } catch (\Exception $e) {
            return back()->with('error', 'Không thể xóa vòng đấu bảng: '.$e->getMessage());
        }
    }

    public function clearPlayoffs(Tournament $tournament)
    {
        $this->authorize('generateBrackets', $tournament);

        try {
            $this->generationService->clearPlayoffs($tournament);
            $this->stateService->clearStateCache($tournament);

            return back()->with('success', 'Đã xóa tất cả các vòng đấu Play-off.');
        } catch (\Exception $e) {
            return back()->with('error', 'Không thể xóa playoffs: '.$e->getMessage());
        }
    }

    public function exportRegistrations(Tournament $tournament)
    {
        $this->authorize('export', $tournament);

        $fileName = 'dang-ky-'.Str::slug($tournament->name).'-'.now()->format('Ymd_His').'.xlsx';

        return Excel::download(new RegistrationsExport($tournament), $fileName);
    }

    public function exportStandings(Tournament $tournament)
    {
        $this->authorize('export', $tournament);

        $fileName = 'bang-xep-hang-'.Str::slug($tournament->name).'.xlsx';

        return Excel::download(new StandingsExport($tournament), $fileName);
    }

    public function exportSchedule(Tournament $tournament)
    {
        $this->authorize('export', $tournament);

        $fileName = 'lich-thi-dau-'.Str::slug($tournament->name).'.xlsx';

        return Excel::download(new ScheduleExport($tournament), $fileName);
    }

    public function exportPairs(Tournament $tournament)
    {
        $this->authorize('export', $tournament);

        $fileName = 'danh-sach-cap-dau-'.Str::slug($tournament->name).'.xlsx';

        return Excel::download(new \App\Exports\PairsExport($tournament), $fileName);
    }

    /**
     * Lazy load standings data to improve performance
     */
    private function getStandingsLazy(Tournament $tournament): array
    {
        $standingsByGroup = [];

        // Only load standings if there are groups AND group rounds
        if ($tournament->groups->count() > 0 && $tournament->rounds()->where('type', 'round_robin')->count() > 0) {
            // Add version tag to invalidate old cached standings when logic changes
            $cacheKey = "tournament_standings_v2_{$tournament->id}_".md5($tournament->updated_at);

            $standingsByGroup = cache()->remember($cacheKey, 600, function () use ($tournament) { // 10 minutes cache
                $standings = [];

                // Load groups with minimal data needed for standings
                $groups = $tournament->groups()->with([
                    'pairs' => fn ($q) => $q->select('id', 'group_id', 'player1_id', 'player2_id'),
                    'pairs.player1' => fn ($q) => $q->select('id', 'name'),
                    'pairs.player2' => fn ($q) => $q->select('id', 'name'),
                ])->get();

                $rounds = $tournament->rounds()->where('type', 'round_robin')->get();

                // Only calculate standings if we actually have rounds
                if ($rounds->count() > 0) {
                    foreach ($groups as $group) {
                        try {
                            $standings[$group->name] = $this->standingsService->calculateForGroup($group, $rounds);
                        } catch (\Exception $e) {
                            Log::warning("Failed to calculate standings for group {$group->name}: {$e->getMessage()}");
                            $standings[$group->name] = [];
                        }
                    }
                }

                return $standings;
            });
        }

        return $standingsByGroup;
    }
}
