<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Player;
use App\Models\Registration;
use App\Models\Tournament;
use App\Services\RegistrationService;
use Illuminate\Http\Request;

class RegistrationController extends Controller
{
    protected $registrationService;

    public function __construct(RegistrationService $registrationService)
    {
        $this->registrationService = $registrationService;
    }

    public function index(Request $request)
    {
        $query = Registration::with(['tournament', 'player1', 'player2']);

        // Filter by tournament
        if ($request->filled('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Get statistics before pagination
        $baseQuery = clone $query;
        $statistics = [
            'pending' => (clone $baseQuery)->where('status', 'pending')->count(),
            'approved' => (clone $baseQuery)->where('status', 'approved')->count(),
            'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
            'total' => (clone $baseQuery)->count(),
        ];

        // Sort by status (pending first), then by created_at desc
        $registrations = $query->orderByRaw("CASE 
            WHEN status = 'pending' THEN 1 
            WHEN status = 'approved' THEN 2 
            WHEN status = 'rejected' THEN 3 
            ELSE 4 END")
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Get tournaments for filter dropdown
        $tournaments = Tournament::orderBy('name')->get();

        return view('admin.registrations.index', compact('registrations', 'tournaments', 'statistics'));
    }

    public function create()
    {
        $tournaments = Tournament::all();
        $players = Player::all();

        return view('admin.registrations.create', compact('tournaments', 'players'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'tournament_id' => 'required|exists:tournaments,id',
            'player1_id' => 'required|exists:players,id|different:player2_id',
            'player2_id' => 'required|exists:players,id',
            'score_sum' => 'required|numeric',
            'status' => 'required|in:pending,approved,rejected',
            'reason' => 'nullable|string',
        ]);
        Registration::create($data);

        return redirect()->route('admin.registrations.index')->with('success', 'Đăng ký thành công!');
    }

    public function edit($id)
    {
        $registration = Registration::findOrFail($id);
        $tournaments = Tournament::all();
        $players = Player::all();

        return view('admin.registrations.edit', compact('registration', 'tournaments', 'players'));
    }

    public function update(Request $request, $id)
    {
        $registration = Registration::findOrFail($id);
        $data = $request->validate([
            'tournament_id' => 'required|exists:tournaments,id',
            'player1_id' => 'required|exists:players,id|different:player2_id',
            'player2_id' => 'required|exists:players,id',
            'score_sum' => 'required|numeric',
            'status' => 'required|in:pending,approved,rejected',
            'reason' => 'nullable|string',
        ]);
        $registration->update($data);

        return redirect()->route('admin.registrations.index')->with('success', 'Cập nhật thành công!');
    }

    public function destroy($id)
    {
        $registration = Registration::findOrFail($id);
        $registration->delete();

        return redirect()->route('admin.registrations.index')->with('success', 'Đã xóa đăng ký!');
    }

    public function updateStatus(Request $request, Registration $registration)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,approved,rejected',
            'reason' => 'nullable|string',
        ]);

        try {
            $this->registrationService->updateStatus($registration, $validated['status'], $validated['reason']);

            return back()->with('success', 'Cập nhật trạng thái thành công!');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }
}
