<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\PlayerRequest;
use App\Models\Player;
use App\Services\PlayerService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class PlayerController extends Controller
{
    protected $playerService;

    public function __construct(PlayerService $playerService)
    {
        $this->playerService = $playerService;
    }

    // Hiển thị danh sách player
    public function index()
    {
        // // Tránh xử lý pending avatars trên mỗi lần mở trang (có thể nặng).
        // // Chỉ cho phép chạy tối đa 1 lần mỗi 30 phút để giảm tải.
        // if (Cache::add('admin_process_pending_avatars_lock', 1, now()->addMinutes(30))) {
        //     try {
        //         $this->processPendingAvatars(false);
        //     } catch (\Throwable $e) {
        //         Log::warning('processPendingAvatars skipped due to error: '.$e->getMessage());
        //     }
        // }

        $query = $this->buildQuery();
        $players = $query->paginate(20)->withQueryString();

        // Get unique values for filters
        $genders = Player::distinct()->pluck('gender')->filter()->sort();
        $locations = Player::distinct()->pluck('location')->filter()->sort();

        return view('admin.players.index', compact('players', 'genders', 'locations'));
    }

    // API endpoint cho tìm kiếm AJAX
    public function search()
    {
        $query = $this->buildQuery();
        $players = $query->paginate(20)->withQueryString();

        // Get unique values for filters
        $genders = Player::distinct()->pluck('gender')->filter()->sort();
        $locations = Player::distinct()->pluck('location')->filter()->sort();

        // Render HTML table content
        $tableHtml = view('admin.players.partials.table', compact('players'))->render();
        $paginationHtml = $players->hasPages() ?
            view('admin.players.partials.pagination', compact('players'))->render() : '';

        return response()->json([
            'success' => true,
            'html' => $tableHtml,
            'pagination' => $paginationHtml,
            'total' => $players->total(),
            'approved_count' => Player::where('status', 1)->count(),
            'pending_count' => Player::where('status', 0)->count(),
            'avatar_pending_count' => Player::whereNotNull('avatar_pending')->count(),
        ]);
    }

    // Phương thức chung để build query
    private function buildQuery()
    {
        $query = Player::query();

        // NOTE: Mods should be able to view/manage all players (like admin).
        // We intentionally do not apply per-mod scoping here so mods can see all records.

        // Filter theo status
        if (request('status') !== null && request('status') !== '') {
            $query->where('status', request('status'));
        }

        // Filter theo gender
        if (request('gender') && request('gender') !== '') {
            $query->where('gender', request('gender'));
        }

        // Filter theo location
        if (request('location') && request('location') !== '') {
            $query->where('location', 'like', '%'.request('location').'%');
        }

        // Search theo tên hoặc phone
        if (request('search') && request('search') !== '') {
            $search = request('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%'.$search.'%')
                    ->orWhere('phone', 'like', '%'.$search.'%')
                    ->orWhere('player_id', 'like', '%'.$search.'%');
            });
        }

        // Filter theo avatar pending
        if (request('avatar_pending') !== null && request('avatar_pending') !== '') {
            if (request('avatar_pending') == '1') {
                $query->whereNotNull('avatar_pending');
            } else {
                $query->whereNull('avatar_pending');
            }
        }

        // Sorting
        $sortBy = request('sort_by', 'id');
        $sortOrder = request('sort_order', 'desc');

        $allowedSorts = ['id', 'name', 'doubles_score', 'singles_score', 'location', 'status', 'created_at'];
        if (in_array($sortBy, $allowedSorts)) {
            $query->orderBy($sortBy, $sortOrder);
        }

        return $query;
    }

    // Hiển thị form tạo mới player
    public function create()
    {
        return view('admin.players.create');
    }

    // Lưu player mới
    public function store(PlayerRequest $request)
    {
        // Kiểm tra email đã tồn tại
        if (\App\Models\User::where('email', $request->email)->exists()) {
            return back()->withErrors(['email' => 'Email đã tồn tại!'])->withInput();
        }
        // Kiểm tra phone đã tồn tại
        if (Player::where('phone', $request->phone)->exists()) {
            return back()->withErrors(['phone' => 'Số điện thoại đã tồn tại!'])->withInput();
        }

        try {
            $data = $request->validated();
            if ($request->hasFile('avatar')) {
                $data['avatar'] = $request->file('avatar');
            }

            // Set creator if user is authenticated
            if (auth()->check()) {
                $data['created_by'] = auth()->id();
            }

            $this->playerService->createPlayer($data);
            // Optional: Gửi email cho người dùng với mật khẩu được tạo
        } catch (\Exception $e) {
            return back()->with('error', 'Đã có lỗi xảy ra: '.$e->getMessage())->withInput();
        }

        return redirect()->route('admin.players.index')->with('success', 'Thêm player thành công!');
    }

    // Hiển thị form chỉnh sửa player
    public function edit(Player $player)
    {
        return view('admin.players.edit', compact('player'));
    }

    // Cập nhật thông tin player
    public function update(PlayerRequest $request, Player $player)
    {
        try {
            $data = $request->validated();
            if ($request->hasFile('avatar')) {
                $data['avatar'] = $request->file('avatar');
            }
            $this->playerService->updatePlayer($player, $data);
        } catch (\Exception $e) {
            return back()->with('error', 'Đã có lỗi xảy ra: '.$e->getMessage())->withInput();
        }

        return redirect()->route('admin.players.index')->with('success', 'Cập nhật thành công!');
    }

    // Xóa player
    public function destroy(Player $player)
    {
        // Xóa avatar khỏi storage trước khi xóa player
        if ($player->avatar) {
            // Handle both old storage format and new assets format
            if (str_starts_with($player->avatar, 'avatars/')) {
                if (\Illuminate\Support\Facades\Storage::disk('public')->exists($player->avatar)) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($player->avatar);
                }
            } elseif (str_starts_with($player->avatar, 'assets/avatars/')) {
                $avatarPath = public_path($player->avatar);
                if (is_file($avatarPath)) {
                    @unlink($avatarPath);
                }
            }
        }

        // Xóa avatar pending nếu có
        if ($player->avatar_pending) {
            if (str_starts_with($player->avatar_pending, 'assets/avatars/pending/')) {
                $pendingPath = public_path($player->avatar_pending);
                if (is_file($pendingPath)) {
                    @unlink($pendingPath);
                }
            }
        }

        $player->delete();

        return redirect()->route('admin.players.index')->with('success', 'Đã xóa player!');
    }

    // Hàm này nên được gọi bởi scheduler (cron) hoặc khi admin vào trang quản lý
    // Nếu $redirect=true thì trả về back, nếu false thì chỉ xử lý không redirect
    public function processPendingAvatars(bool $redirect = true)
    {
        $processedCount = 0;
        $errorCount = 0;

        // Sử dụng chunkById để xử lý theo từng khối, tiết kiệm bộ nhớ
        Player::whereNotNull('avatar_pending')
            ->whereNotNull('avatar_pending_requested_at')
            ->where('avatar_pending_requested_at', '<=', now()->subHours(24))
            ->chunkById(100, function ($players) use (&$processedCount, &$errorCount) {
                foreach ($players as $player) {
                    try {
                        $this->_approveAvatarForPlayer($player);
                        $processedCount++;
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::error("Failed to process pending avatar for player #{$player->id}: ".$e->getMessage());
                        $errorCount++;
                    }
                }
            });

        if ($redirect) {
            $message = "Đã tự động xử lý {$processedCount} avatar chờ duyệt.";
            if ($errorCount > 0) {
                $message .= " Có {$errorCount} lỗi xảy ra, vui lòng kiểm tra logs.";
            }

            return back()->with('success', $message);
        }
    }

    // Admin duyệt avatar ngay lập tức
    public function approveAvatar($id)
    {
        $player = Player::findOrFail($id);

        try {
            $this->_approveAvatarForPlayer($player);

            return back()->with('success', 'Avatar đã được duyệt và cập nhật thành công!');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    // Admin từ chối avatar và xóa khỏi pending
    public function rejectAvatar($id)
    {
        $player = Player::findOrFail($id);

        try {
            $this->_rejectAvatarForPlayer($player);

            return back()->with('success', 'Avatar đã được từ chối và xóa khỏi trạng thái chờ duyệt!');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    /**
     * Helper method to approve a pending avatar for a specific player.
     * This encapsulates the logic for file moving, database updating, and error handling.
     *
     * @throws \Exception
     */
    private function _approveAvatarForPlayer(Player $player): void
    {
        $pendingPath = $player->avatar_pending;

        if (! $pendingPath) {
            throw new \Exception('Không có avatar chờ duyệt cho VĐV này.');
        }

        // Check if file exists in new assets format
        $pendingFilePath = public_path($pendingPath);
        if (! file_exists($pendingFilePath)) {
            \Illuminate\Support\Facades\Log::warning("Avatar pending file not found for player #{$player->id}: {$pendingPath}");
            // Dọn dẹp DB nếu file không tồn tại
            $player->update([
                'avatar_pending' => null,
                'avatar_pending_requested_at' => null,
            ]);
            throw new \Exception('Không tìm thấy tệp avatar đang chờ xử lý. Đã dọn dẹp thông tin.');
        }

        \Illuminate\Support\Facades\DB::transaction(function () use ($player, $pendingFilePath) {
            // Tạo đường dẫn mới an toàn và duy nhất trong thư mục 'assets/avatars'
            $extension = pathinfo($pendingFilePath, PATHINFO_EXTENSION);
            $newFilename = 'player_'.$player->id.'_'.time().'.'.$extension;
            $newPath = 'assets/avatars/'.$newFilename;
            $newFilePath = public_path($newPath);

            // Di chuyển tệp từ vị trí chờ sang vị trí chính thức
            if (! rename($pendingFilePath, $newFilePath)) {
                throw new \Exception('Không thể di chuyển file avatar.');
            }

            // Xóa avatar cũ nếu có
            if ($player->avatar) {
                // Handle both old storage format and new assets format
                if (str_starts_with($player->avatar, 'avatars/')) {
                    // Old storage format
                    if (\Illuminate\Support\Facades\Storage::disk('public')->exists($player->avatar)) {
                        \Illuminate\Support\Facades\Storage::disk('public')->delete($player->avatar);
                    }
                } elseif (str_starts_with($player->avatar, 'assets/avatars/')) {
                    // New assets format
                    $oldAvatarPath = public_path($player->avatar);
                    if (file_exists($oldAvatarPath)) {
                        unlink($oldAvatarPath);
                    }
                }
            }

            // Cập nhật avatar mới cho Player và User liên quan
            $player->avatar = $newPath;
            $player->avatar_pending = null;
            $player->avatar_pending_requested_at = null;
            $player->save();

            // Cập nhật avatar cho User liên quan nếu có
            if ($player->user) {
                $player->user->avatar = $newPath;
                $player->user->save();
            }
        });
    }

    /**
     * Helper method to reject a pending avatar for a specific player.
     * This removes the pending avatar file and clears the pending status.
     *
     * @throws \Exception
     */
    private function _rejectAvatarForPlayer(Player $player): void
    {
        $pendingPath = $player->avatar_pending;

        if (! $pendingPath) {
            throw new \Exception('Không có avatar chờ duyệt cho VĐV này.');
        }

        \Illuminate\Support\Facades\DB::transaction(function () use ($player, $pendingPath) {
            // Xóa file avatar pending nếu tồn tại
            $pendingFilePath = public_path($pendingPath);
            if (file_exists($pendingFilePath)) {
                if (! unlink($pendingFilePath)) {
                    \Illuminate\Support\Facades\Log::warning("Could not delete pending avatar file for player #{$player->id}: {$pendingPath}");
                }
            }

            // Xóa thông tin avatar pending khỏi database
            $player->avatar_pending = null;
            $player->avatar_pending_requested_at = null;
            $player->save();
        });
    }
}
