<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\MatchesRequest;
use App\Models\MatchModel as Matches;
use App\Models\Tournament;
use App\Services\EloService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class MatchesController extends Controller
{
    protected $eloService;

    public function __construct(EloService $eloService)
    {
        $this->eloService = $eloService;
    }

    /**
     * Cập nhật score cho trận đấu (admin/mod)
     */
    public function updateScore(Request $request, Matches $match)
    {
        // Tải các quan hệ cần thiết để tính Elo và kiểm tra.
        $match->load(['round.tournament', 'pair1', 'pair2']);
        // Since this is in the admin context, we can assume authorization is handled by the middleware group.
        // If you have a specific policy for Matches, you can uncomment this:
        // $this->authorize('update', $match);

        $validated = $request->validate([
            'score1' => ['sometimes', 'required', 'integer', 'min:0'],
            'score2' => ['sometimes', 'required', 'integer', 'min:0'],
            'status' => ['sometimes', 'string', 'in:Chưa bắt đầu,Đang diễn ra,Hoàn thành,Tạm hoãn,Bị hủy'],
        ]);

        // Validation: Prevent score updates if the match result was used to create the next playoff round.
        if ($match->round?->type === 'playoff') {
            $tournamentId = $match->round->tournament_id;
            $nextRoundNames = [];

            if ($match->round->name === 'Tứ kết') {
                $nextRoundNames = ['Bán kết'];
            } elseif ($match->round->name === 'Bán kết') {
                $nextRoundNames = ['Chung kết', 'Tranh Hạng Ba'];
            }

            if (! empty($nextRoundNames)) {
                // Determine winner/loser from the incoming request scores
                $winnerId = $request->score1 > $request->score2 ? $match->pair1_id : $match->pair2_id;
                $loserId = $request->score1 < $request->score2 ? $match->pair1_id : $match->pair2_id;

                // More efficient check: Check if any match in the next rounds involves the winner or loser.
                // Using whereRelation for cleaner syntax (available in Laravel 9+).
                $isUsedInNextRound = Matches::query()
                    ->whereRelation('round', 'tournament_id', $tournamentId)
                    ->whereRelation('round', function ($query) use ($nextRoundNames) {
                        $query->whereIn('name', $nextRoundNames);
                    })
                    ->where(function ($query) use ($winnerId, $loserId) {
                        $query->where('pair1_id', $winnerId)
                            ->orWhere('pair2_id', $winnerId)
                            ->orWhere('pair1_id', $loserId)
                            ->orWhere('pair2_id', $loserId);
                    })
                    ->exists();

                if ($isUsedInNextRound) {
                    $errorMessage = 'Không thể cập nhật tỷ số. Kết quả trận đấu này đã được sử dụng để tạo vòng đấu tiếp theo.';
                    if ($request->expectsJson()) {
                        return response()->json(['success' => false, 'message' => $errorMessage], 422);
                    }

                    return back()->with('error', $errorMessage);
                }
            }
        }

        try {
            // Lưu tỷ số dưới dạng "score1-score2", ví dụ: "11-5"
            $match->score = $validated['score1'].'-'.$validated['score2'];

            // Cập nhật trạng thái nếu có trong request, nếu không thì tự động set là "Hoàn thành"
            if (isset($validated['status'])) {
                $match->status = $validated['status'];
            } elseif ($validated['score1'] > 0 || $validated['score2'] > 0) {
                $match->status = 'Hoàn thành';
            }

            $match->save();

            // Chỉ cập nhật Elo khi trận đấu hoàn thành
            if ($match->status === 'Hoàn thành') {
                $this->eloService->updateEloForMatch($match);
            }

            // Log activity on the tournament when a match score is updated
            if ($match->round && $match->round->tournament) {
                activity()
                    ->performedOn($match->round->tournament)
                    ->causedBy(auth()->user())
                    ->withProperties([
                        'event' => 'match_score_updated',
                        'event_label' => 'Cập nhật kết quả',
                        'match_id' => $match->id,
                        'round' => $match->round->name,
                        'score' => $match->score,
                        'status' => $match->status,
                        'pair1_id' => $match->pair1_id,
                        'pair2_id' => $match->pair2_id,
                    ])
                    ->event('match_score_updated')
                    ->log('Cập nhật kết quả trận đấu');
            }

            // Touch tournament to invalidate cache based on updated_at
            $tournament = $match->round?->tournament;
            if ($tournament) {
                $tournament->touch();

                // Clear standings cache if we have cache service
                if (app()->bound(\App\Services\TournamentCacheService::class)) {
                    $cacheService = app(\App\Services\TournamentCacheService::class);
                    $cacheService->clearStandingsCache($tournament->id);
                    $cacheService->invalidateTournamentCache($tournament->id, ['standings', 'matches']);
                }
            }

            if ($request->expectsJson()) {
                return response()->json(['success' => true, 'message' => 'Cập nhật tỷ số và điểm Elo thành công!']);
            }

            return back()->with('success', 'Cập nhật tỷ số và điểm Elo thành công!');
        } catch (\Exception $e) {
            // Log the error if needed: \Log::error($e);
            return response()->json(['success' => false, 'message' => 'Lỗi server khi cập nhật tỷ số.'], 500);
        }
    }

    public function index(Tournament $tournament)
    {
        $this->authorize('viewAny', Matches::class);
        $matches = $tournament->matches()->with(['player1', 'player2', 'opponent1', 'opponent2'])->paginate(20);

        return view('admin.matches.index', compact('tournament', 'matches'));
    }

    public function create(Tournament $tournament)
    {
        $this->authorize('create', Matches::class);
        $players = Player::select('id', 'player_id', 'name')->get();

        return view('admin.matches.create', compact('tournament', 'players'));
    }

    public function store(MatchesRequest $request, Tournament $tournament)
    {
        $this->authorize('create', Matches::class);
        $data = $request->validated();
        $data['match_id'] = 'M'.strtoupper(uniqid());
        $data['tournament_id'] = $tournament->id;

        if ($request->hasFile('images')) {
            $images = [];
            $imageService = app(\App\Services\ImageOptimizationService::class);

            foreach ($request->file('images') as $image) {
                try {
                    $optimizedImages = $imageService->optimizeAndStore($image, 'match');
                    $images[] = $optimizedImages['medium']['jpeg'];
                } catch (\Exception $e) {
                    // Fallback to original method if optimization fails
                    logger()->warning('Image optimization failed for match, using fallback: '.$e->getMessage());
                    $path = $image->store('matches', 'public');
                    $images[] = '/storage/'.$path;
                }
            }
            $data['images'] = $images;
        }

        Matches::create($data);

        return redirect()->route('admin.matches.index', $tournament)->with('success', 'Match created successfully!');
    }

    public function edit(Tournament $tournament, Matches $match)
    {
        $this->authorize('update', $match);
        $players = Player::select('id', 'player_id', 'name')->get();

        return view('admin.matches.edit', compact('tournament', 'match', 'players'));
    }

    public function update(MatchesRequest $request, Tournament $tournament, Matches $match)
    {
        $this->authorize('update', $match);
        $data = $request->validated();

        if ($request->hasFile('images')) {
            // Delete old images
            if ($match->images) {
                foreach ($match->images as $image) {
                    Storage::disk('public')->delete(str_replace('/storage/', '', $image));
                }
            }

            $images = [];
            $imageService = app(\App\Services\ImageOptimizationService::class);

            foreach ($request->file('images') as $image) {
                try {
                    $optimizedImages = $imageService->optimizeAndStore($image, 'match');
                    $images[] = $optimizedImages['medium']['jpeg'];
                } catch (\Exception $e) {
                    // Fallback to original method if optimization fails
                    logger()->warning('Image optimization failed for match, using fallback: '.$e->getMessage());
                    $path = $image->store('matches', 'public');
                    $images[] = '/storage/'.$path;
                }
            }
            $data['images'] = $images;
        }

        $match->update($data);

        return redirect()->route('admin.matches.index', $tournament)->with('success', 'Match updated successfully!');
    }

    public function destroy(Tournament $tournament, Matches $match)
    {
        $this->authorize('delete', $match);
        if ($match->images) {
            foreach ($match->images as $image) {
                Storage::disk('public')->delete(str_replace('/storage/', '', $image));
            }
        }
        $match->delete();

        return redirect()->route('admin.matches.index', $tournament)->with('success', 'Match deleted successfully!');
    }
}
