<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GroupPair;
use App\Models\Pair;
use App\Models\Player;
use App\Models\TournamentGroup as Group;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class GroupPairController extends Controller
{
    public function index(Group $group) // $group is TournamentGroup
    {
        $pairs = $group->pairs()->with(['player1', 'player2'])->get();

        return view('admin.pairs.index', compact('group', 'pairs'));
    }

    public function create(Group $group) // $group is TournamentGroup
    {
        // Lấy các đăng ký (pair) đã được duyệt cho giải đấu này
        $registrations = DB::table('registrations')
            ->where('tournament_id', $group->tournament_id)
            ->where('status', 'approved')
            ->select('id', 'player1_id', 'player2_id')
            ->get();

        // Lấy các cặp đã được thêm vào bất kỳ bảng nào trong cùng giải đấu (để loại trừ)
        $existingPairs = DB::table('group_pairs')
            ->join('tournament_groups', 'group_pairs.group_id', '=', 'tournament_groups.id')
            ->where('tournament_groups.tournament_id', $group->tournament_id)
            ->select('group_pairs.player1_id', 'group_pairs.player2_id')
            ->get();

        // Chuẩn hóa key cho cặp (để so sánh không phân biệt thứ tự)
        $existingPairKeys = $existingPairs->map(function ($p) {
            $a = (int) $p->player1_id;
            $b = (int) $p->player2_id;
            if ($a > $b) {
                [$a, $b] = [$b, $a];
            }

            return $a.'-'.$b;
        })->unique()->toArray();

        // Lọc các registration pairs: chỉ lấy những cặp đã đăng ký và chưa có trong existingPairKeys
        $availableRegistrations = $registrations->filter(function ($r) use ($existingPairKeys) {
            $a = (int) $r->player1_id;
            $b = (int) $r->player2_id;
            if ($a > $b) {
                [$a, $b] = [$b, $a];
            }
            $key = $a.'-'.$b;

            return ! in_array($key, $existingPairKeys, true);
        })->values();

        if ($availableRegistrations->isEmpty()) {
            return back()->with('info', 'Không còn cặp đăng ký nào chưa được thêm vào bảng.');
        }

        // Lấy model Player cho hiển thị thông tin tên
        $playerIds = $availableRegistrations->pluck('player1_id')->merge($availableRegistrations->pluck('player2_id'))->unique()->filter()->values();
        $playersMap = Player::whereIn('id', $playerIds)->get()->keyBy('id');

        // Chuyển dữ liệu sang mảng dễ dùng trong view: mỗi item có player models
        $pairs = $availableRegistrations->map(function ($r) use ($playersMap) {
            return [
                'registration_id' => $r->id,
                'player1_id' => (int) $r->player1_id,
                'player2_id' => (int) $r->player2_id,
                'player1' => $playersMap->get($r->player1_id),
                'player2' => $playersMap->get($r->player2_id),
            ];
        })->all();

        return view('admin.pairs.create', compact('group', 'pairs'));
    }

    public function store(Request $request, Group $group) // $group is TournamentGroup
    {
        $data = $request->validate([
            'player1_id' => 'required|exists:players,id|different:player2_id',
            'player2_id' => 'required|exists:players,id',
        ], [
            'player1_id.different' => 'Hai vận động viên không được trùng nhau.',
        ]);

        // Kiểm tra xem cặp đấu này đã tồn tại ở bất kỳ bảng nào khác trong cùng giải đấu chưa
        $player1_id = $data['player1_id'];
        $player2_id = $data['player2_id'];

        $existingPair = GroupPair::whereHas('group', function ($query) use ($group) { // group() relation now points to TournamentGroup
            $query->where('tournament_id', $group->tournament_id);
        })->where(function ($query) use ($player1_id, $player2_id) {
            $query->where(function ($q) use ($player1_id, $player2_id) {
                $q->where('player1_id', $player1_id)->where('player2_id', $player2_id);
            })->orWhere(function ($q) use ($player1_id, $player2_id) {
                $q->where('player1_id', $player2_id)->where('player2_id', $player1_id);
            });
        })->first();

        if ($existingPair) {
            return back()->withErrors(['player1_id' => 'Cặp đấu này đã tồn tại ở Bảng '.$existingPair->group->name.'.'])->withInput();
        }

        $data['group_id'] = $group->id;
        GroupPair::create($data);

        // Quay lại trang chi tiết giải đấu sau khi thêm
        return redirect()->route('admin.tournaments.show', $group->tournament_id)->with('success', 'Thêm cặp đấu vào bảng '.$group->name.' thành công!')->with('open_tab', 'groups-pairs');
    }

    public function edit($groupId, $pairId) // $group is TournamentGroup
    {
        // Manually find the models to avoid nested route model binding issues
        $group = Group::findOrFail($groupId);
        $pair = GroupPair::findOrFail($pairId);

        // Tối ưu hóa: Lấy tất cả ID người chơi đã đăng ký và được duyệt trong một truy vấn
        $registeredPlayers = DB::table('registrations')
            ->where('tournament_id', $group->tournament_id)
            ->where('status', 'approved')
            ->select('player1_id', 'player2_id')
            ->get();
        $registeredPlayerIds = $registeredPlayers->pluck('player1_id')->merge($registeredPlayers->pluck('player2_id'))->unique()->filter();

        // Tối ưu hóa: Lấy tất cả ID người chơi đã được xếp cặp (trừ cặp hiện tại) trong một truy vấn
        $otherPairedPlayerIds = DB::table('group_pairs')
            ->join('tournament_groups', 'group_pairs.group_id', '=', 'tournament_groups.id')
            ->where('tournament_groups.tournament_id', $group->tournament_id)
            ->where('group_pairs.id', '!=', $pair->id)
            ->select('group_pairs.player1_id', 'group_pairs.player2_id')
            ->get()
            ->pluck('player1_id')->merge(DB::table('group_pairs')->where('group_pairs.id', '!=', $pair->id)->pluck('player2_id'))->unique()->filter();

        // Danh sách người chơi có sẵn bao gồm những người đã đăng ký và chưa được ghép cặp ở nơi khác
        $players = Player::whereIn('id', $registeredPlayerIds)->whereNotIn('id', $otherPairedPlayerIds)->get();
        // Thêm 2 người chơi của cặp hiện tại vào danh sách để có thể chọn lại họ
        $players = $players->merge(Player::whereIn('id', [$pair->player1_id, $pair->player2_id])->get())->unique('id');

        return view('admin.pairs.edit', compact('group', 'pair', 'players'))->with('open_tab', 'groups-pairs');
    }

    public function update(Request $request, $groupId, $pairId) // $group is TournamentGroup
    {
        $group = Group::findOrFail($groupId);
        $pair = GroupPair::findOrFail($pairId);

        $data = $request->validate([
            'player1_id' => 'required|exists:players,id|different:player2_id',
            'player2_id' => 'required|exists:players,id',
        ]);
        $pair->update($data);

        // Redirect back to the main tournament show page for consistency
        return redirect()->route('admin.tournaments.show', $group->tournament_id)->with('success', 'Cập nhật cặp đấu thành công!')->with('open_tab', 'groups-pairs');
    }

    public function destroy(Group $group, GroupPair $pair) // $group is TournamentGroup
    {
        $pair->delete();

        return back()->with('success', 'Đã xóa cặp đấu!')->with('open_tab', 'groups-pairs');
    }

    /**
     * Swap player1_id and player2_id for a given pair.
     *
     * @param  GroupPair  $pair
     * @return JsonResponse
     */
    public function swapPlayers(GroupPair $pair): JsonResponse
    {
        try {
            // Use a transaction to ensure atomicity
            DB::transaction(function () use ($pair) {
                $player1 = $pair->player1_id;
                $player2 = $pair->player2_id;

                $pair->player1_id = $player2;
                $pair->player2_id = $player1;
                $pair->save();
            });

            // Get updated pair with player information for frontend
            $pair->load(['player1', 'player2']);

            return response()->json([
                'success' => true, 
                'message' => 'Đã hoán đổi vị trí VĐV thành công.',
                'pair' => [
                    'id' => $pair->id,
                    'player1' => [
                        'id' => $pair->player1->id,
                        'name' => $pair->player1->name,
                        'avatar' => $pair->player1->avatar
                    ],
                    'player2' => [
                        'id' => $pair->player2->id,
                        'name' => $pair->player2->name,
                        'avatar' => $pair->player2->avatar
                    ]
                ]
            ]);
        } catch (\Exception $e) {
            Log::error("Error swapping players for pair {$pair->id}: ".$e->getMessage());

            return response()->json(['success' => false, 'message' => 'Có lỗi xảy ra, không thể hoán đổi.'], 500);
        }
    }
}